"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventField = exports.RuleTargetInput = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * The input to send to the event target.
 *
 * @stability stable
 */
class RuleTargetInput {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * Pass text to the event target.
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     *
     * @stability stable
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by EventField.from() to substitute in parts
     * of the matched event.
     *
     * @stability stable
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target.
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     *
     * @stability stable
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON.
     *
     * @stability stable
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
}
exports.RuleTargetInput = RuleTargetInput;
_a = JSII_RTTI_SYMBOL_1;
RuleTargetInput[_a] = { fqn: "@aws-cdk/aws-events.RuleTargetInput", version: "1.128.0" };
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by EventBridge, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return `<${key}>`;
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        const keys = Object.keys(inputPathsMap);
        if (keys.length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved, keys),
            inputPathsMap,
        };
    }
    /**
     * Removing surrounding quotes from any object placeholders
     * when key is the lone value.
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     *
     * Do not remove quotes when the key is part of a larger string.
     *
     * Valid: { "data": "Some string with \"quotes\"<key>" } // key will be string
     * Valid: { "data": <key> } // Key could be number, bool, obj, or string
     */
    unquoteKeyPlaceholders(sub, keys) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.uncachedString({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return keys.reduce((r, key) => r.replace(new RegExp(`(?<!\\\\)\"\<${key}\>\"`, 'g'), `<${key}>`), resolved);
            }
            return resolved;
        }
    }
}
/**
 * Represents a field in the event pattern.
 *
 * @stability stable
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event.
     *
     * @stability stable
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event.
     *
     * @stability stable
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event.
     *
     * @stability stable
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event.
     *
     * @stability stable
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event.
     *
     * @stability stable
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event.
     *
     * @stability stable
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event.
     *
     * @stability stable
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    /**
     * Produce the Token's value at resolution time.
     *
     * @stability stable
     */
    resolve(_ctx) {
        return this.path;
    }
    /**
     * Return a string representation of this resolvable object.
     *
     * Returns a reversible string representation.
     *
     * @stability stable
     */
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON.
     *
     * @stability stable
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
_b = JSII_RTTI_SYMBOL_1;
EventField[_b] = { fqn: "@aws-cdk/aws-events.EventField", version: "1.128.0" };
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
//# sourceMappingURL=data:application/json;base64,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