# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['regenmaschine', 'regenmaschine.endpoints']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.0', 'typing-extensions>=4.3.0,<5.0.0']

setup_kwargs = {
    'name': 'regenmaschine',
    'version': '2022.11.1',
    'description': 'A simple API for RainMachine sprinkler controllers',
    'long_description': '# 💧 Regenmaschine: A Simple Python Library for RainMachine™\n\n[![CI](https://github.com/bachya/regenmaschine/workflows/CI/badge.svg)](https://github.com/bachya/regenmaschine/actions)\n[![PyPi](https://img.shields.io/pypi/v/regenmaschine.svg)](https://pypi.python.org/pypi/regenmaschine)\n[![Version](https://img.shields.io/pypi/pyversions/regenmaschine.svg)](https://pypi.python.org/pypi/regenmaschine)\n[![License](https://img.shields.io/pypi/l/regenmaschine.svg)](https://github.com/bachya/regenmaschine/blob/main/LICENSE)\n[![Code Coverage](https://codecov.io/gh/bachya/regenmaschine/branch/main/graph/badge.svg)](https://codecov.io/gh/bachya/regenmaschine)\n[![Maintainability](https://api.codeclimate.com/v1/badges/a99a88d28ad37a79dbf6/maintainability)](https://codeclimate.com/github/bachya/regenmaschine/maintainability)\n[![Say Thanks](https://img.shields.io/badge/SayThanks-!-1EAEDB.svg)](https://saythanks.io/to/bachya)\n\n<a href="https://www.buymeacoffee.com/bachya1208P" target="_blank"><img src="https://cdn.buymeacoffee.com/buttons/default-orange.png" alt="Buy Me A Coffee" height="41" width="174"></a>\n\n`regenmaschine` (German for "rain machine") is a simple, clean, well-tested\nPython library for interacting with\n[RainMachine™ smart sprinkler controllers](http://www.rainmachine.com/).\nIt gives developers an easy API to manage their controllers over their local\nLAN or remotely via the RainMachine™ cloud.\n\n- [Remote Access Announcement](#remote-access-announcement-2022-06-26)\n- [Python Versions](#python-versions)\n- [Installation](#installation)\n- [Usage](#usage)\n- [Loading Controllers Multiple Times](#loading-controllers-multiple-times)\n- [Contributing](#contributing)\n\n# Remote Access Announcement (2022-06-26)\n\nOn June 2, 2022, RainMachine announced a\n[Premium Services](https://www.rainmachine.com/premium/) addition; under this new\nmodel, remote access is _only_ available to subscribers of these Premium Services.\n\nI do not currently intend to subscribe to Premium Services; as such, the remote access\nabilities of `regenmaschine` will remain as-is from here on out unless spurred on by\nothers. They may stop working at any time. PRs from subscribing users are always\nwelcome.\n\n# Python Versions\n\n`regenmaschine` is currently supported on:\n\n- Python 3.9\n- Python 3.10\n- Python 3.11\n\n# Installation\n\n```bash\npip install regenmaschine\n```\n\n# Usage\n\nCreating a `regenmaschine` `Client` might be the easiest thing you do all day:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    client = Client()\n\n    # ...\n\n\nasyncio.run(main())\n```\n\nBy default, the library creates a new connection to the sprinkler controller with each\ncoroutine. If you are calling a large number of coroutines (or merely want to squeeze\nout every second of runtime savings possible), an\n[`aiohttp`](https://github.com/aio-libs/aiohttp) `ClientSession` can be used for connection\npooling:\n\nSee the module docstrings throughout the library for full info on all parameters, return\ntypes, etc.\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        client = Client(session=session)\n\n\nasyncio.run(main())\n```\n\n## Loading Local (Accessible Over the LAN) Controllers\n\nOnce you have a client, you can load a local controller (i.e., one that is\naccessible over the LAN) very easily:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        client = Client(session=session)\n\n        await client.load_local("192.168.1.101", "my_password", port=8080, use_ssl=True)\n\n        controllers = client.controllers\n        # >>> {\'ab:cd:ef:12:34:56\': <LocalController>}\n\n\nasyncio.run(main())\n```\n\n## Loading Remote (Accessible Over the RainMachine Cloud) Controllers\n\nIf you have 1, 2 or 100 other local controllers, you can load them in the same\nway – `client.controllers` will keep your controllers all organized.\n\nWhat if you have controllers around the world and can\'t access them all over\nthe same local network? No problem! `regenmaschine` allows you to load remote\ncontrollers very easily, as well:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        client = Client(session=session)\n\n        await client.load_remote("rainmachine_email@host.com", "my_password")\n\n        controllers = client.controllers\n        # >>> {\'xx:xx:xx:xx:xx:xx\': <RemoteController>, ...}\n\n\nasyncio.run(main())\n```\n\nBonus tip: `client.load_remote` will load _all_ controllers owned by that email\naddress.\n\n## Using the Controller\n\nRegardless of the type of controller you have loaded (local or remote), the\nsame properties and methods are available to each:\n\n```python\nimport asyncio\nimport datetime\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        client = Client(session=session)\n\n        # Load a local controller:\n        await client.load_local("192.168.1.101", "my_password", port=8080, use_ssl=True)\n\n        # Load all remote controllers associated with an account:\n        await client.load_remote("rainmachine_email@host.com", "my_password")\n\n        # They all act the same! The only difference is that remote API calls\n        # will pass through the RainMachine™ cloud:\n        for mac_address, controller in client.controllers:\n            # Print some client properties:\n            print(f"Name: {controller.name}")\n            print(f"Host: {controller.host}")\n            print(f"MAC Address: {controller.mac}")\n            print(f"API Version: {controller.api_version}")\n            print(f"Software Version: {controller.software_version}")\n            print(f"Hardware Version: {controller.hardware_version}")\n\n            # Get all diagnostic information:\n            diagnostics = await controller.diagnostics.current()\n\n            # Get all weather parsers:\n            parsers = await controller.parsers.current()\n\n            # Get all programs:\n            programs = await controller.programs.all()\n\n            # Include inactive programs:\n            programs = await controller.programs.all(include_inactive=True)\n\n            # Get a specific program:\n            program_1 = await controller.programs.get(1)\n\n            # Enable or disable a specific program:\n            await controller.programs.enable(1)\n            await controller.programs.disable(1)\n\n            # Get the next run time for all programs:\n            runs = await controller.programs.next()\n\n            # Get all running programs:\n            programs = await controller.programs.running()\n\n            # Start and stop a program:\n            await controller.programs.start(1)\n            await controller.programs.stop(1)\n\n            # Get basic details about all zones:\n            zones = await controller.zones.all()\n\n            # Get advanced details about all zones:\n            zones = await controller.zones.all(details=True)\n\n            # Include inactive zones:\n            zones = await controller.zones.all(include_inactive=True)\n\n            # Get basic details about a specific zone:\n            zone_1 = await controller.zones.get(1)\n\n            # Get advanced details about a specific zone:\n            zone_1 = await controller.zones.get(1, details=True)\n\n            # Enable or disable a specific zone:\n            await controller.zones.enable(1)\n            await controller.zones.disable(1)\n\n            # Start a zone for 60 seconds:\n            await controller.zones.start(1, 60)\n\n            # ...and stop it:\n            await controller.zones.stop(1)\n\n            # Get the device name:\n            name = await controller.provisioning.device_name\n\n            # Get all provisioning settings:\n            settings = await controller.provisioning.settings()\n\n            # Get all networking info related to the device:\n            wifi = await controller.provisioning.wifi()\n\n            # Get various types of active watering restrictions:\n            current = await controller.restrictions.current()\n            universal = await controller.restrictions.universal()\n            hourly = await controller.restrictions.hourly()\n            raindelay = await controller.restrictions.raindelay()\n\n            # Set universal restrictions – note that the payload is the same structure\n            # as returned by controller.restrictions.universal():\n            await controller.restrictions.set_universal(\n                {\n                    "hotDaysExtraWatering": False,\n                    "freezeProtectEnabled": True,\n                }\n            )\n\n            # Get watering stats:\n            today = await controller.stats.on_date(datetime.date.today())\n            upcoming_days = await controller.stats.upcoming(details=True)\n\n            # Get info on various watering activities not already covered:\n            log = await controller.watering.log(datetime.date.today(), 2)\n            queue = await controller.watering.queue()\n            runs = await controller.watering.runs(datetime.date.today())\n\n            # Pause all watering activities for 30 seconds:\n            await controller.watering.pause_all(30)\n\n            # Unpause all watering activities:\n            await controller.watering.unpause_all()\n\n            # Stop all watering activities:\n            await controller.watering.stop_all()\n\n            # See if a firmware update is available:\n            update_data = await controller.machine.get_firmware_update_status()\n            # ...and request the update:\n            update_data = await controller.machine.update_firmware()\n\n            # Reboot the controller:\n            update_data = await controller.machine.reboot()\n\n            # Return the current flow meter data:\n            flowmeter = await controller.watering.flowmeter()\n\n            # Add values to flowmeter counters from an external smart water meter\n            # not wired directly to the controller.\n            # Units can be "clicks", "gal", "m3" and "litre".\n            await controller.watering.post_flowmeter({"value": 2000, "units": "clicks"})\n\n\nasyncio.run(main())\n```\n\nCheck out `example.py`, the tests, and the source files themselves for method\nsignatures and more examples. For additional reference, the full RainMachine™ API documentation is available [here](https://rainmachine.docs.apiary.io/).\n\n# Loading Controllers Multiple Times\n\nIt is technically possible to load a controller multiple times. Let\'s pretend\nfor a moment that:\n\n- We have a local controller named `Home` (available at `192.168.1.101`).\n- We have a remote controller named `Grandma\'s House`.\n- Both controllers live under our email address: `user@host.com`\n\nIf we load them thus:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        client = Client(session=session)\n\n        # Load "Home" locally:\n        await client.load_local("192.168.1.101", "my_password")\n\n        # Load all of my controllers remotely:\n        await client.load_remote("user@host.com", "my_password")\n\n\nasyncio.run(main())\n```\n\n...then we will have the following:\n\n1. `Home` will be a `LocalController` and accessible over the LAN.\n2. `Grandma\'s House` will be a `RemoteController` and accessible only over the\n   RainMachine™ cloud.\n\nNotice that `regenmaschine` is smart enough to not overwrite a controller that\nalready exists: even though `Home` exists as a remote controller owned by\n`user@host.com`, it had already been loaded locally. By default,\n`regenmaschine` will only load a controller if it hasn\'t been loaded before\n(locally _or_ remotely). If you want to change this behavior, both `load_local`\nand `load_remote` accept an optional `skip_existing` parameter:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom regenmaschine import Client\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        client = Client(session=session)\n\n        # Load all of my controllers remotely:\n        await client.load_remote("user@host.com", "my_password")\n\n        # Load "Home" locally, overwriting the existing remote controller:\n        await client.load_local("192.168.1.101", "my_password", skip_existing=False)\n\n\nasyncio.run(main())\n```\n\n# Contributing\n\n1. [Check for open features/bugs](https://github.com/bachya/regenmaschine/issues)\n   or [initiate a discussion on one](https://github.com/bachya/regenmaschine/issues/new).\n2. [Fork the repository](https://github.com/bachya/regenmaschine/fork).\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./.venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `poetry run pytest --cov regenmaschine tests`\n9. Update `README.md` with any new documentation.\n10. Add yourself to `AUTHORS.md`.\n11. Submit a pull request!\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/bachya/regenmaschine',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9.0,<4.0.0',
}


setup(**setup_kwargs)
