"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define an Application Target Group
 */
class ApplicationTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props = {}) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        super(scope, id, { ...props }, {
            protocol,
            port,
        });
        this.protocol = protocol;
        this.port = port;
        this.connectableMembers = [];
        this.listeners = [];
        if (props) {
            if (props.slowStart !== undefined) {
                this.setAttribute('slow_start.duration_seconds', props.slowStart.toSeconds().toString());
            }
            if (props.stickinessCookieDuration !== undefined) {
                this.enableCookieStickiness(props.stickinessCookieDuration);
            }
            this.addTarget(...(props.targets || []));
        }
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedApplicationTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing target group
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return ApplicationTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Enable sticky routing via a cookie to members of this target group
     */
    enableCookieStickiness(duration) {
        this.setAttribute('stickiness.enabled', 'true');
        this.setAttribute('stickiness.type', 'lb_cookie');
        this.setAttribute('stickiness.lb_cookie.duration_seconds', duration.toSeconds().toString());
    }
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     */
    registerConnectable(connectable, portRange) {
        portRange = portRange || ec2.Port.tcp(this.defaultPort);
        // Notify all listeners that we already know about of this new connectable.
        // Then remember for new listeners that might get added later.
        this.connectableMembers.push({ connectable, portRange });
        for (const listener of this.listeners) {
            listener.registerConnectable(connectable, portRange);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener, associatingConstruct) {
        // Notify this listener of all connectables that we know about.
        // Then remember for new connectables that might get added later.
        for (const member of this.connectableMembers) {
            listener.registerConnectable(member.connectable, member.portRange);
        }
        this.listeners.push(listener);
        this.loadBalancerAttachedDependencies.add(associatingConstruct || listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Return the given named metric for this Application Load Balancer Target Group
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: {
                TargetGroup: this.targetGroupFullName,
                LoadBalancer: this.firstLoadBalancerFullName,
            },
            ...props
        }).attachTo(this);
    }
    /**
     * The number of IPv6 requests received by the target group
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The number of healthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props
        });
    }
    /**
     * The number of unhealthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricUnhealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCountPerTarget(props) {
        return this.metric('RequestCountPerTarget', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props
        });
    }
    validate() {
        const ret = super.validate();
        if (this.targetType !== undefined && this.targetType !== enums_1.TargetType.LAMBDA
            && (this.protocol === undefined || this.port === undefined)) {
            ret.push(`At least one of 'port' or 'protocol' is required for a non-Lambda TargetGroup`);
        }
        if (this.healthCheck && this.healthCheck.protocol && !ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
            ret.push([
                `Health check protocol '${this.healthCheck.protocol}' is not supported. `,
                `Must be one of [${ALB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`
            ].join(''));
        }
        return ret;
    }
}
exports.ApplicationTargetGroup = ApplicationTargetGroup;
/**
 * An imported application target group
 */
class ImportedApplicationTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener, _associatingConstruct) {
        // Nothing to do, we know nothing of our members
        this.node.addWarning(`Cannot register listener on imported target group -- security groups might need to be updated manually`);
    }
    registerConnectable(_connectable, _portRange) {
        this.node.addWarning(`Cannot register connectable on imported target group -- security groups might need to be updated manually`);
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const ALB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tdGFyZ2V0LWdyb3VwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXBwbGljYXRpb24tdGFyZ2V0LWdyb3VwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsc0RBQXNEO0FBQ3RELHdDQUF3QztBQUV4QyxtRUFHcUM7QUFDckMsMkNBQTRFO0FBQzVFLGlEQUE2RDtBQUM3RCx5Q0FBMEQ7QUF3RDFEOztHQUVHO0FBQ0gsTUFBYSxzQkFBdUIsU0FBUSxtQ0FBZTtJQXNCekQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFxQyxFQUFFO1FBQy9FLE1BQU0sQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLEdBQUcsK0JBQXdCLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDOUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsRUFBRSxHQUFHLEtBQUssRUFBRSxFQUFFO1lBQzdCLFFBQVE7WUFDUixJQUFJO1NBQ0wsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7UUFFakIsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEVBQUUsQ0FBQztRQUM3QixJQUFJLENBQUMsU0FBUyxHQUFHLEVBQUUsQ0FBQztRQUVwQixJQUFJLEtBQUssRUFBRTtZQUNULElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLEVBQUU7Z0JBQ2pDLElBQUksQ0FBQyxZQUFZLENBQUMsNkJBQTZCLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO2FBQzFGO1lBQ0QsSUFBSSxLQUFLLENBQUMsd0JBQXdCLEtBQUssU0FBUyxFQUFFO2dCQUNoRCxJQUFJLENBQUMsc0JBQXNCLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDLENBQUM7YUFDN0Q7WUFDRCxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7U0FDMUM7SUFDSCxDQUFDO0lBM0NEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLHlCQUF5QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBQ2hHLE9BQU8sSUFBSSw4QkFBOEIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUM5RSxPQUFPLHNCQUFzQixDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQStCRDs7T0FFRztJQUNJLFNBQVMsQ0FBQyxHQUFHLE9BQXlDO1FBQzNELEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzVCLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQyw4QkFBOEIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMzRCxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDcEM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxzQkFBc0IsQ0FBQyxRQUFrQjtRQUM5QyxJQUFJLENBQUMsWUFBWSxDQUFDLG9CQUFvQixFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxZQUFZLENBQUMsaUJBQWlCLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDbEQsSUFBSSxDQUFDLFlBQVksQ0FBQyx1Q0FBdUMsRUFBRSxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztJQUM5RixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG1CQUFtQixDQUFDLFdBQTZCLEVBQUUsU0FBb0I7UUFDNUUsU0FBUyxHQUFHLFNBQVMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFeEQsMkVBQTJFO1FBQzNFLDhEQUE4RDtRQUM5RCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLEVBQUUsV0FBVyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDekQsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ3JDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLEVBQUUsU0FBUyxDQUFDLENBQUM7U0FDdEQ7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGdCQUFnQixDQUFDLFFBQThCLEVBQUUsb0JBQWlDO1FBQ3ZGLCtEQUErRDtRQUMvRCxpRUFBaUU7UUFDakUsS0FBSyxNQUFNLE1BQU0sSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDNUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDOUIsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsSUFBSSxRQUFRLENBQUMsQ0FBQztJQUM5RSxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLHlCQUF5QjtRQUNsQyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMvQixNQUFNLElBQUksS0FBSyxDQUFDLHdGQUF3RixDQUFDLENBQUM7U0FDM0c7UUFDRCxPQUFPLG1EQUErQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLE1BQU0sQ0FBQyxVQUFrQixFQUFFLEtBQWdDO1FBQ2hFLE9BQU8sSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQzNCLFNBQVMsRUFBRSxvQkFBb0I7WUFDL0IsVUFBVTtZQUNWLFVBQVUsRUFBRTtnQkFDVixXQUFXLEVBQUUsSUFBSSxDQUFDLG1CQUFtQjtnQkFDckMsWUFBWSxFQUFFLElBQUksQ0FBQyx5QkFBeUI7YUFDN0M7WUFDRCxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3BCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBZ0M7UUFDNUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixFQUFFO1lBQ3JDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0IsQ0FBQyxLQUFnQztRQUN4RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsY0FBYyxFQUFFO1lBQ2pDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBZ0M7UUFDNUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixFQUFFO1lBQ3JDLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksd0JBQXdCLENBQUMsS0FBZ0M7UUFDOUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLG9CQUFvQixFQUFFO1lBQ3ZDLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0IsQ0FBQyxJQUFvQixFQUFFLEtBQWdDO1FBQ2hGLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUU7WUFDdkIsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQixDQUFDLEtBQWdDO1FBQ2pFLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyx1QkFBdUIsRUFBRTtZQUMxQyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGdDQUFnQyxDQUFDLEtBQWdDO1FBQ3RFLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyw0QkFBNEIsRUFBRTtZQUMvQyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHdCQUF3QixDQUFDLEtBQWdDO1FBQzlELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsRUFBRTtZQUN2QyxTQUFTLEVBQUUsU0FBUztZQUNwQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DLENBQUMsS0FBZ0M7UUFDMUUsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGdDQUFnQyxFQUFFO1lBQ25ELFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFUyxRQUFRO1FBQ2hCLE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUU3QixJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssU0FBUyxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssa0JBQVUsQ0FBQyxNQUFNO2VBQ3JFLENBQUMsSUFBSSxDQUFDLFFBQVEsS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxTQUFTLENBQUMsRUFBRTtZQUMzRCxHQUFHLENBQUMsSUFBSSxDQUFDLCtFQUErRSxDQUFDLENBQUM7U0FDN0Y7UUFFRCxJQUFJLElBQUksQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLElBQUksQ0FBQywwQkFBMEIsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNwSCxHQUFHLENBQUMsSUFBSSxDQUFDO2dCQUNQLDBCQUEwQixJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsc0JBQXNCO2dCQUN6RSxtQkFBbUIsMEJBQTBCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHO2FBQzVELENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7U0FDYjtRQUVELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztDQUNGO0FBclFELHdEQXFRQztBQXlDRDs7R0FFRztBQUNILE1BQU0sOEJBQStCLFNBQVEsa0NBQXVCO0lBQzNELGdCQUFnQixDQUFDLFNBQStCLEVBQUUscUJBQWtDO1FBQ3pGLGdEQUFnRDtRQUNoRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx3R0FBd0csQ0FBQyxDQUFDO0lBQ2pJLENBQUM7SUFFTSxtQkFBbUIsQ0FBQyxZQUE4QixFQUFFLFVBQWlDO1FBQzFGLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLDJHQUEyRyxDQUFDLENBQUM7SUFDcEksQ0FBQztJQUVNLFNBQVMsQ0FBQyxHQUFHLE9BQXlDO1FBQzNELEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzVCLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQyw4QkFBOEIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUUzRCxJQUFJLE1BQU0sQ0FBQyxVQUFVLEtBQUssU0FBUyxFQUFFO2dCQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLHdHQUF3RyxDQUFDLENBQUM7YUFDM0g7U0FDRjtJQUNILENBQUM7Q0FDRjtBQWVELE1BQU0sMEJBQTBCLEdBQUcsQ0FBQyxnQkFBUSxDQUFDLElBQUksRUFBRSxnQkFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBJQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQge1xuICBCYXNlVGFyZ2V0R3JvdXBQcm9wcywgSVRhcmdldEdyb3VwLCBsb2FkQmFsYW5jZXJOYW1lRnJvbUxpc3RlbmVyQXJuLCBMb2FkQmFsYW5jZXJUYXJnZXRQcm9wcyxcbiAgVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzLCBUYXJnZXRHcm91cEJhc2UsIFRhcmdldEdyb3VwSW1wb3J0UHJvcHNcbn0gZnJvbSAnLi4vc2hhcmVkL2Jhc2UtdGFyZ2V0LWdyb3VwJztcbmltcG9ydCB7IEFwcGxpY2F0aW9uUHJvdG9jb2wsIFByb3RvY29sLCBUYXJnZXRUeXBlIH0gZnJvbSAnLi4vc2hhcmVkL2VudW1zJztcbmltcG9ydCB7IEltcG9ydGVkVGFyZ2V0R3JvdXBCYXNlIH0gZnJvbSAnLi4vc2hhcmVkL2ltcG9ydGVkJztcbmltcG9ydCB7IGRldGVybWluZVByb3RvY29sQW5kUG9ydCB9IGZyb20gJy4uL3NoYXJlZC91dGlsJztcbmltcG9ydCB7IElBcHBsaWNhdGlvbkxpc3RlbmVyIH0gZnJvbSAnLi9hcHBsaWNhdGlvbi1saXN0ZW5lcic7XG5pbXBvcnQgeyBIdHRwQ29kZVRhcmdldCB9IGZyb20gJy4vYXBwbGljYXRpb24tbG9hZC1iYWxhbmNlcic7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgZGVmaW5pbmcgYW4gQXBwbGljYXRpb24gVGFyZ2V0IEdyb3VwXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25UYXJnZXRHcm91cFByb3BzIGV4dGVuZHMgQmFzZVRhcmdldEdyb3VwUHJvcHMge1xuICAvKipcbiAgICogVGhlIHByb3RvY29sIHRvIHVzZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERldGVybWluZWQgZnJvbSBwb3J0IGlmIGtub3duLCBvcHRpb25hbCBmb3IgTGFtYmRhIHRhcmdldHMuXG4gICAqL1xuICByZWFkb25seSBwcm90b2NvbD86IEFwcGxpY2F0aW9uUHJvdG9jb2w7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0IG9uIHdoaWNoIHRoZSBsaXN0ZW5lciBsaXN0ZW5zIGZvciByZXF1ZXN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZXRlcm1pbmVkIGZyb20gcHJvdG9jb2wgaWYga25vd24sIG9wdGlvbmFsIGZvciBMYW1iZGEgdGFyZ2V0cy5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIHBlcmlvZCBkdXJpbmcgd2hpY2ggdGhlIGxvYWQgYmFsYW5jZXIgc2VuZHMgYSBuZXdseSByZWdpc3RlcmVkXG4gICAqIHRhcmdldCBhIGxpbmVhcmx5IGluY3JlYXNpbmcgc2hhcmUgb2YgdGhlIHRyYWZmaWMgdG8gdGhlIHRhcmdldCBncm91cC5cbiAgICpcbiAgICogVGhlIHJhbmdlIGlzIDMwLTkwMCBzZWNvbmRzICgxNSBtaW51dGVzKS5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgcmVhZG9ubHkgc2xvd1N0YXJ0PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBzdGlja2luZXNzIGNvb2tpZSBleHBpcmF0aW9uIHBlcmlvZC5cbiAgICpcbiAgICogU2V0dGluZyB0aGlzIHZhbHVlIGVuYWJsZXMgbG9hZCBiYWxhbmNlciBzdGlja2luZXNzLlxuICAgKlxuICAgKiBBZnRlciB0aGlzIHBlcmlvZCwgdGhlIGNvb2tpZSBpcyBjb25zaWRlcmVkIHN0YWxlLiBUaGUgbWluaW11bSB2YWx1ZSBpc1xuICAgKiAxIHNlY29uZCBhbmQgdGhlIG1heGltdW0gdmFsdWUgaXMgNyBkYXlzICg2MDQ4MDAgc2Vjb25kcykuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLmRheXMoMSlcbiAgICovXG4gIHJlYWRvbmx5IHN0aWNraW5lc3NDb29raWVEdXJhdGlvbj86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgdGFyZ2V0cyB0byBhZGQgdG8gdGhpcyB0YXJnZXQgZ3JvdXAuXG4gICAqXG4gICAqIENhbiBiZSBgSW5zdGFuY2VgLCBgSVBBZGRyZXNzYCwgb3IgYW55IHNlbGYtcmVnaXN0ZXJpbmcgbG9hZCBiYWxhbmNpbmdcbiAgICogdGFyZ2V0LiBJZiB5b3UgdXNlIGVpdGhlciBgSW5zdGFuY2VgIG9yIGBJUEFkZHJlc3NgIGFzIHRhcmdldHMsIGFsbFxuICAgKiB0YXJnZXQgbXVzdCBiZSBvZiB0aGUgc2FtZSB0eXBlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHRhcmdldHMuXG4gICAqL1xuICByZWFkb25seSB0YXJnZXRzPzogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0W107XG59XG5cbi8qKlxuICogRGVmaW5lIGFuIEFwcGxpY2F0aW9uIFRhcmdldCBHcm91cFxuICovXG5leHBvcnQgY2xhc3MgQXBwbGljYXRpb25UYXJnZXRHcm91cCBleHRlbmRzIFRhcmdldEdyb3VwQmFzZSBpbXBsZW1lbnRzIElBcHBsaWNhdGlvblRhcmdldEdyb3VwIHtcbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyB0YXJnZXQgZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVRhcmdldEdyb3VwQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzKTogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRBcHBsaWNhdGlvblRhcmdldEdyb3VwKHNjb3BlLCBpZCwgYXR0cnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyB0YXJnZXQgZ3JvdXBcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBmcm9tVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzYCBpbnN0ZWFkXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGltcG9ydChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVGFyZ2V0R3JvdXBJbXBvcnRQcm9wcyk6IElBcHBsaWNhdGlvblRhcmdldEdyb3VwIHtcbiAgICByZXR1cm4gQXBwbGljYXRpb25UYXJnZXRHcm91cC5mcm9tVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzKHNjb3BlLCBpZCwgcHJvcHMpO1xuICB9XG5cbiAgcHJpdmF0ZSByZWFkb25seSBjb25uZWN0YWJsZU1lbWJlcnM6IENvbm5lY3RhYmxlTWVtYmVyW107XG4gIHByaXZhdGUgcmVhZG9ubHkgbGlzdGVuZXJzOiBJQXBwbGljYXRpb25MaXN0ZW5lcltdO1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3RvY29sPzogQXBwbGljYXRpb25Qcm90b2NvbDtcbiAgcHJpdmF0ZSByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBsaWNhdGlvblRhcmdldEdyb3VwUHJvcHMgPSB7fSkge1xuICAgIGNvbnN0IFtwcm90b2NvbCwgcG9ydF0gPSBkZXRlcm1pbmVQcm90b2NvbEFuZFBvcnQocHJvcHMucHJvdG9jb2wsIHByb3BzLnBvcnQpO1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgeyAuLi5wcm9wcyB9LCB7XG4gICAgICBwcm90b2NvbCxcbiAgICAgIHBvcnQsXG4gICAgfSk7XG5cbiAgICB0aGlzLnByb3RvY29sID0gcHJvdG9jb2w7XG4gICAgdGhpcy5wb3J0ID0gcG9ydDtcblxuICAgIHRoaXMuY29ubmVjdGFibGVNZW1iZXJzID0gW107XG4gICAgdGhpcy5saXN0ZW5lcnMgPSBbXTtcblxuICAgIGlmIChwcm9wcykge1xuICAgICAgaWYgKHByb3BzLnNsb3dTdGFydCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHRoaXMuc2V0QXR0cmlidXRlKCdzbG93X3N0YXJ0LmR1cmF0aW9uX3NlY29uZHMnLCBwcm9wcy5zbG93U3RhcnQudG9TZWNvbmRzKCkudG9TdHJpbmcoKSk7XG4gICAgICB9XG4gICAgICBpZiAocHJvcHMuc3RpY2tpbmVzc0Nvb2tpZUR1cmF0aW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhpcy5lbmFibGVDb29raWVTdGlja2luZXNzKHByb3BzLnN0aWNraW5lc3NDb29raWVEdXJhdGlvbik7XG4gICAgICB9XG4gICAgICB0aGlzLmFkZFRhcmdldCguLi4ocHJvcHMudGFyZ2V0cyB8fCBbXSkpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBsb2FkIGJhbGFuY2luZyB0YXJnZXQgdG8gdGhpcyB0YXJnZXQgZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBhZGRUYXJnZXQoLi4udGFyZ2V0czogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0W10pIHtcbiAgICBmb3IgKGNvbnN0IHRhcmdldCBvZiB0YXJnZXRzKSB7XG4gICAgICBjb25zdCByZXN1bHQgPSB0YXJnZXQuYXR0YWNoVG9BcHBsaWNhdGlvblRhcmdldEdyb3VwKHRoaXMpO1xuICAgICAgdGhpcy5hZGRMb2FkQmFsYW5jZXJUYXJnZXQocmVzdWx0KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRW5hYmxlIHN0aWNreSByb3V0aW5nIHZpYSBhIGNvb2tpZSB0byBtZW1iZXJzIG9mIHRoaXMgdGFyZ2V0IGdyb3VwXG4gICAqL1xuICBwdWJsaWMgZW5hYmxlQ29va2llU3RpY2tpbmVzcyhkdXJhdGlvbjogRHVyYXRpb24pIHtcbiAgICB0aGlzLnNldEF0dHJpYnV0ZSgnc3RpY2tpbmVzcy5lbmFibGVkJywgJ3RydWUnKTtcbiAgICB0aGlzLnNldEF0dHJpYnV0ZSgnc3RpY2tpbmVzcy50eXBlJywgJ2xiX2Nvb2tpZScpO1xuICAgIHRoaXMuc2V0QXR0cmlidXRlKCdzdGlja2luZXNzLmxiX2Nvb2tpZS5kdXJhdGlvbl9zZWNvbmRzJywgZHVyYXRpb24udG9TZWNvbmRzKCkudG9TdHJpbmcoKSk7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXIgYSBjb25uZWN0YWJsZSBhcyBhIG1lbWJlciBvZiB0aGlzIHRhcmdldCBncm91cC5cbiAgICpcbiAgICogRG9uJ3QgY2FsbCB0aGlzIGRpcmVjdGx5LiBJdCB3aWxsIGJlIGNhbGxlZCBieSBsb2FkIGJhbGFuY2luZyB0YXJnZXRzLlxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyQ29ubmVjdGFibGUoY29ubmVjdGFibGU6IGVjMi5JQ29ubmVjdGFibGUsIHBvcnRSYW5nZT86IGVjMi5Qb3J0KSB7XG4gICAgcG9ydFJhbmdlID0gcG9ydFJhbmdlIHx8IGVjMi5Qb3J0LnRjcCh0aGlzLmRlZmF1bHRQb3J0KTtcblxuICAgIC8vIE5vdGlmeSBhbGwgbGlzdGVuZXJzIHRoYXQgd2UgYWxyZWFkeSBrbm93IGFib3V0IG9mIHRoaXMgbmV3IGNvbm5lY3RhYmxlLlxuICAgIC8vIFRoZW4gcmVtZW1iZXIgZm9yIG5ldyBsaXN0ZW5lcnMgdGhhdCBtaWdodCBnZXQgYWRkZWQgbGF0ZXIuXG4gICAgdGhpcy5jb25uZWN0YWJsZU1lbWJlcnMucHVzaCh7IGNvbm5lY3RhYmxlLCBwb3J0UmFuZ2UgfSk7XG4gICAgZm9yIChjb25zdCBsaXN0ZW5lciBvZiB0aGlzLmxpc3RlbmVycykge1xuICAgICAgbGlzdGVuZXIucmVnaXN0ZXJDb25uZWN0YWJsZShjb25uZWN0YWJsZSwgcG9ydFJhbmdlKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXIgYSBsaXN0ZW5lciB0aGF0IGlzIGxvYWQgYmFsYW5jaW5nIHRvIHRoaXMgdGFyZ2V0IGdyb3VwLlxuICAgKlxuICAgKiBEb24ndCBjYWxsIHRoaXMgZGlyZWN0bHkuIEl0IHdpbGwgYmUgY2FsbGVkIGJ5IGxpc3RlbmVycy5cbiAgICovXG4gIHB1YmxpYyByZWdpc3Rlckxpc3RlbmVyKGxpc3RlbmVyOiBJQXBwbGljYXRpb25MaXN0ZW5lciwgYXNzb2NpYXRpbmdDb25zdHJ1Y3Q/OiBJQ29uc3RydWN0KSB7XG4gICAgLy8gTm90aWZ5IHRoaXMgbGlzdGVuZXIgb2YgYWxsIGNvbm5lY3RhYmxlcyB0aGF0IHdlIGtub3cgYWJvdXQuXG4gICAgLy8gVGhlbiByZW1lbWJlciBmb3IgbmV3IGNvbm5lY3RhYmxlcyB0aGF0IG1pZ2h0IGdldCBhZGRlZCBsYXRlci5cbiAgICBmb3IgKGNvbnN0IG1lbWJlciBvZiB0aGlzLmNvbm5lY3RhYmxlTWVtYmVycykge1xuICAgICAgbGlzdGVuZXIucmVnaXN0ZXJDb25uZWN0YWJsZShtZW1iZXIuY29ubmVjdGFibGUsIG1lbWJlci5wb3J0UmFuZ2UpO1xuICAgIH1cbiAgICB0aGlzLmxpc3RlbmVycy5wdXNoKGxpc3RlbmVyKTtcbiAgICB0aGlzLmxvYWRCYWxhbmNlckF0dGFjaGVkRGVwZW5kZW5jaWVzLmFkZChhc3NvY2lhdGluZ0NvbnN0cnVjdCB8fCBsaXN0ZW5lcik7XG4gIH1cblxuICAvKipcbiAgICogRnVsbCBuYW1lIG9mIGZpcnN0IGxvYWQgYmFsYW5jZXJcbiAgICovXG4gIHB1YmxpYyBnZXQgZmlyc3RMb2FkQmFsYW5jZXJGdWxsTmFtZSgpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLmxpc3RlbmVycy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIFRhcmdldEdyb3VwIG5lZWRzIHRvIGJlIGF0dGFjaGVkIHRvIGEgTG9hZEJhbGFuY2VyIGJlZm9yZSB5b3UgY2FuIGNhbGwgdGhpcyBtZXRob2QnKTtcbiAgICB9XG4gICAgcmV0dXJuIGxvYWRCYWxhbmNlck5hbWVGcm9tTGlzdGVuZXJBcm4odGhpcy5saXN0ZW5lcnNbMF0ubGlzdGVuZXJBcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgZ2l2ZW4gbmFtZWQgbWV0cmljIGZvciB0aGlzIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIgVGFyZ2V0IEdyb3VwXG4gICAqXG4gICAqIFJldHVybnMgdGhlIG1ldHJpYyBmb3IgdGhpcyB0YXJnZXQgZ3JvdXAgZnJvbSB0aGUgcG9pbnQgb2YgdmlldyBvZiB0aGUgZmlyc3RcbiAgICogbG9hZCBiYWxhbmNlciBsb2FkIGJhbGFuY2luZyB0byBpdC4gSWYgeW91IGhhdmUgbXVsdGlwbGUgbG9hZCBiYWxhbmNlcnMgbG9hZFxuICAgKiBzZW5kaW5nIHRyYWZmaWMgdG8gdGhlIHNhbWUgdGFyZ2V0IGdyb3VwLCB5b3Ugd2lsbCBoYXZlIHRvIG92ZXJyaWRlIHRoZSBkaW1lbnNpb25zXG4gICAqIG9uIHRoaXMgbWV0cmljLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBdmVyYWdlIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljKG1ldHJpY05hbWU6IHN0cmluZywgcHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICBuYW1lc3BhY2U6ICdBV1MvQXBwbGljYXRpb25FTEInLFxuICAgICAgbWV0cmljTmFtZSxcbiAgICAgIGRpbWVuc2lvbnM6IHtcbiAgICAgICAgVGFyZ2V0R3JvdXA6IHRoaXMudGFyZ2V0R3JvdXBGdWxsTmFtZSxcbiAgICAgICAgTG9hZEJhbGFuY2VyOiB0aGlzLmZpcnN0TG9hZEJhbGFuY2VyRnVsbE5hbWUsXG4gICAgICB9LFxuICAgICAgLi4ucHJvcHNcbiAgICB9KS5hdHRhY2hUbyh0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIElQdjYgcmVxdWVzdHMgcmVjZWl2ZWQgYnkgdGhlIHRhcmdldCBncm91cFxuICAgKlxuICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNJcHY2UmVxdWVzdENvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdJUHY2UmVxdWVzdENvdW50Jywge1xuICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiByZXF1ZXN0cyBwcm9jZXNzZWQgb3ZlciBJUHY0IGFuZCBJUHY2LlxuICAgKlxuICAgKiBUaGlzIGNvdW50IGluY2x1ZGVzIG9ubHkgdGhlIHJlcXVlc3RzIHdpdGggYSByZXNwb25zZSBnZW5lcmF0ZWQgYnkgYSB0YXJnZXQgb2YgdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY1JlcXVlc3RDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnUmVxdWVzdENvdW50Jywge1xuICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBoZWFsdGh5IGhvc3RzIGluIHRoZSB0YXJnZXQgZ3JvdXBcbiAgICpcbiAgICogQGRlZmF1bHQgQXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY0hlYWx0aHlIb3N0Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0hlYWx0aHlIb3N0Q291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdBdmVyYWdlJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiB1bmhlYWx0aHkgaG9zdHMgaW4gdGhlIHRhcmdldCBncm91cFxuICAgKlxuICAgKiBAZGVmYXVsdCBBdmVyYWdlIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljVW5oZWFsdGh5SG9zdENvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdVbkhlYWx0aHlIb3N0Q291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdBdmVyYWdlJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBIVFRQIDJ4eC8zeHgvNHh4LzV4eCByZXNwb25zZSBjb2RlcyBnZW5lcmF0ZWQgYnkgYWxsIHRhcmdldHMgaW4gdGhpcyB0YXJnZXQgZ3JvdXAuXG4gICAqXG4gICAqIFRoaXMgZG9lcyBub3QgaW5jbHVkZSBhbnkgcmVzcG9uc2UgY29kZXMgZ2VuZXJhdGVkIGJ5IHRoZSBsb2FkIGJhbGFuY2VyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNIdHRwQ29kZVRhcmdldChjb2RlOiBIdHRwQ29kZVRhcmdldCwgcHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoY29kZSwge1xuICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGF2ZXJhZ2UgbnVtYmVyIG9mIHJlcXVlc3RzIHJlY2VpdmVkIGJ5IGVhY2ggdGFyZ2V0IGluIGEgdGFyZ2V0IGdyb3VwLlxuICAgKlxuICAgKiBUaGUgb25seSB2YWxpZCBzdGF0aXN0aWMgaXMgU3VtLiBOb3RlIHRoYXQgdGhpcyByZXByZXNlbnRzIHRoZSBhdmVyYWdlIG5vdCB0aGUgc3VtLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNSZXF1ZXN0Q291bnRQZXJUYXJnZXQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1JlcXVlc3RDb3VudFBlclRhcmdldCcsIHtcbiAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgY29ubmVjdGlvbnMgdGhhdCB3ZXJlIG5vdCBzdWNjZXNzZnVsbHkgZXN0YWJsaXNoZWQgYmV0d2VlbiB0aGUgbG9hZCBiYWxhbmNlciBhbmQgdGFyZ2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNUYXJnZXRDb25uZWN0aW9uRXJyb3JDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnVGFyZ2V0Q29ubmVjdGlvbkVycm9yQ291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgLi4ucHJvcHNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgdGltZSBlbGFwc2VkLCBpbiBzZWNvbmRzLCBhZnRlciB0aGUgcmVxdWVzdCBsZWF2ZXMgdGhlIGxvYWQgYmFsYW5jZXIgdW50aWwgYSByZXNwb25zZSBmcm9tIHRoZSB0YXJnZXQgaXMgcmVjZWl2ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNUYXJnZXRSZXNwb25zZVRpbWUocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1RhcmdldFJlc3BvbnNlVGltZScsIHtcbiAgICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgICAgLi4ucHJvcHNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIFRMUyBjb25uZWN0aW9ucyBpbml0aWF0ZWQgYnkgdGhlIGxvYWQgYmFsYW5jZXIgdGhhdCBkaWQgbm90IGVzdGFibGlzaCBhIHNlc3Npb24gd2l0aCB0aGUgdGFyZ2V0LlxuICAgKlxuICAgKiBQb3NzaWJsZSBjYXVzZXMgaW5jbHVkZSBhIG1pc21hdGNoIG9mIGNpcGhlcnMgb3IgcHJvdG9jb2xzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNUYXJnZXRUTFNOZWdvdGlhdGlvbkVycm9yQ291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1RhcmdldFRMU05lZ290aWF0aW9uRXJyb3JDb3VudCcsIHtcbiAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdICB7XG4gICAgY29uc3QgcmV0ID0gc3VwZXIudmFsaWRhdGUoKTtcblxuICAgIGlmICh0aGlzLnRhcmdldFR5cGUgIT09IHVuZGVmaW5lZCAmJiB0aGlzLnRhcmdldFR5cGUgIT09IFRhcmdldFR5cGUuTEFNQkRBXG4gICAgICAmJiAodGhpcy5wcm90b2NvbCA9PT0gdW5kZWZpbmVkIHx8IHRoaXMucG9ydCA9PT0gdW5kZWZpbmVkKSkge1xuICAgICAgICByZXQucHVzaChgQXQgbGVhc3Qgb25lIG9mICdwb3J0JyBvciAncHJvdG9jb2wnIGlzIHJlcXVpcmVkIGZvciBhIG5vbi1MYW1iZGEgVGFyZ2V0R3JvdXBgKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5oZWFsdGhDaGVjayAmJiB0aGlzLmhlYWx0aENoZWNrLnByb3RvY29sICYmICFBTEJfSEVBTFRIX0NIRUNLX1BST1RPQ09MUy5pbmNsdWRlcyh0aGlzLmhlYWx0aENoZWNrLnByb3RvY29sKSkge1xuICAgICAgcmV0LnB1c2goW1xuICAgICAgICBgSGVhbHRoIGNoZWNrIHByb3RvY29sICcke3RoaXMuaGVhbHRoQ2hlY2sucHJvdG9jb2x9JyBpcyBub3Qgc3VwcG9ydGVkLiBgLFxuICAgICAgICBgTXVzdCBiZSBvbmUgb2YgWyR7QUxCX0hFQUxUSF9DSEVDS19QUk9UT0NPTFMuam9pbignLCAnKX1dYFxuICAgICAgXS5qb2luKCcnKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHJldDtcbiAgfVxufVxuXG4vKipcbiAqIEEgY29ubmVjdGFibGUgbWVtYmVyIG9mIGEgdGFyZ2V0IGdyb3VwXG4gKi9cbmludGVyZmFjZSBDb25uZWN0YWJsZU1lbWJlciB7XG4gIC8qKlxuICAgKiBUaGUgY29ubmVjdGFibGUgbWVtYmVyXG4gICAqL1xuICBjb25uZWN0YWJsZTogZWMyLklDb25uZWN0YWJsZTtcblxuICAvKipcbiAgICogVGhlIHBvcnQgKHJhbmdlKSB0aGUgbWVtYmVyIGlzIGxpc3RlbmluZyBvblxuICAgKi9cbiAgcG9ydFJhbmdlOiBlYzIuUG9ydDtcbn1cblxuLyoqXG4gKiBBIFRhcmdldCBHcm91cCBmb3IgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlcnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQXBwbGljYXRpb25UYXJnZXRHcm91cCBleHRlbmRzIElUYXJnZXRHcm91cCB7XG4gIC8qKlxuICAgKiBSZWdpc3RlciBhIGxpc3RlbmVyIHRoYXQgaXMgbG9hZCBiYWxhbmNpbmcgdG8gdGhpcyB0YXJnZXQgZ3JvdXAuXG4gICAqXG4gICAqIERvbid0IGNhbGwgdGhpcyBkaXJlY3RseS4gSXQgd2lsbCBiZSBjYWxsZWQgYnkgbGlzdGVuZXJzLlxuICAgKi9cbiAgcmVnaXN0ZXJMaXN0ZW5lcihsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXIsIGFzc29jaWF0aW5nQ29uc3RydWN0PzogSUNvbnN0cnVjdCk6IHZvaWQ7XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVyIGEgY29ubmVjdGFibGUgYXMgYSBtZW1iZXIgb2YgdGhpcyB0YXJnZXQgZ3JvdXAuXG4gICAqXG4gICAqIERvbid0IGNhbGwgdGhpcyBkaXJlY3RseS4gSXQgd2lsbCBiZSBjYWxsZWQgYnkgbG9hZCBiYWxhbmNpbmcgdGFyZ2V0cy5cbiAgICovXG4gIHJlZ2lzdGVyQ29ubmVjdGFibGUoY29ubmVjdGFibGU6IGVjMi5JQ29ubmVjdGFibGUsIHBvcnRSYW5nZT86IGVjMi5Qb3J0KTogdm9pZDtcblxuICAvKipcbiAgICogQWRkIGEgbG9hZCBiYWxhbmNpbmcgdGFyZ2V0IHRvIHRoaXMgdGFyZ2V0IGdyb3VwXG4gICAqL1xuICBhZGRUYXJnZXQoLi4udGFyZ2V0czogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0W10pOiB2b2lkO1xufVxuXG4vKipcbiAqIEFuIGltcG9ydGVkIGFwcGxpY2F0aW9uIHRhcmdldCBncm91cFxuICovXG5jbGFzcyBJbXBvcnRlZEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgZXh0ZW5kcyBJbXBvcnRlZFRhcmdldEdyb3VwQmFzZSBpbXBsZW1lbnRzIElBcHBsaWNhdGlvblRhcmdldEdyb3VwIHtcbiAgcHVibGljIHJlZ2lzdGVyTGlzdGVuZXIoX2xpc3RlbmVyOiBJQXBwbGljYXRpb25MaXN0ZW5lciwgX2Fzc29jaWF0aW5nQ29uc3RydWN0PzogSUNvbnN0cnVjdCkge1xuICAgIC8vIE5vdGhpbmcgdG8gZG8sIHdlIGtub3cgbm90aGluZyBvZiBvdXIgbWVtYmVyc1xuICAgIHRoaXMubm9kZS5hZGRXYXJuaW5nKGBDYW5ub3QgcmVnaXN0ZXIgbGlzdGVuZXIgb24gaW1wb3J0ZWQgdGFyZ2V0IGdyb3VwIC0tIHNlY3VyaXR5IGdyb3VwcyBtaWdodCBuZWVkIHRvIGJlIHVwZGF0ZWQgbWFudWFsbHlgKTtcbiAgfVxuXG4gIHB1YmxpYyByZWdpc3RlckNvbm5lY3RhYmxlKF9jb25uZWN0YWJsZTogZWMyLklDb25uZWN0YWJsZSwgX3BvcnRSYW5nZT86IGVjMi5Qb3J0IHwgdW5kZWZpbmVkKTogdm9pZCB7XG4gICAgdGhpcy5ub2RlLmFkZFdhcm5pbmcoYENhbm5vdCByZWdpc3RlciBjb25uZWN0YWJsZSBvbiBpbXBvcnRlZCB0YXJnZXQgZ3JvdXAgLS0gc2VjdXJpdHkgZ3JvdXBzIG1pZ2h0IG5lZWQgdG8gYmUgdXBkYXRlZCBtYW51YWxseWApO1xuICB9XG5cbiAgcHVibGljIGFkZFRhcmdldCguLi50YXJnZXRzOiBJQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJUYXJnZXRbXSkge1xuICAgIGZvciAoY29uc3QgdGFyZ2V0IG9mIHRhcmdldHMpIHtcbiAgICAgIGNvbnN0IHJlc3VsdCA9IHRhcmdldC5hdHRhY2hUb0FwcGxpY2F0aW9uVGFyZ2V0R3JvdXAodGhpcyk7XG5cbiAgICAgIGlmIChyZXN1bHQudGFyZ2V0SnNvbiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGFkZCBhIG5vbi1zZWxmIHJlZ2lzdGVyaW5nIHRhcmdldCB0byBhbiBpbXBvcnRlZCBUYXJnZXRHcm91cC4gQ3JlYXRlIGEgbmV3IFRhcmdldEdyb3VwIGluc3RlYWQuJyk7XG4gICAgICB9XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBjb25zdHJ1Y3RzIHRoYXQgY2FuIGJlIHRhcmdldHMgb2YgYW4gYXBwbGljYXRpb24gbG9hZCBiYWxhbmNlclxuICovXG5leHBvcnQgaW50ZXJmYWNlIElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlclRhcmdldCB7XG4gIC8qKlxuICAgKiBBdHRhY2ggbG9hZC1iYWxhbmNlZCB0YXJnZXQgdG8gYSBUYXJnZXRHcm91cFxuICAgKlxuICAgKiBNYXkgcmV0dXJuIEpTT04gdG8gZGlyZWN0bHkgYWRkIHRvIHRoZSBbVGFyZ2V0c10gbGlzdCwgb3IgcmV0dXJuIHVuZGVmaW5lZFxuICAgKiBpZiB0aGUgdGFyZ2V0IHdpbGwgcmVnaXN0ZXIgaXRzZWxmIHdpdGggdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAqL1xuICBhdHRhY2hUb0FwcGxpY2F0aW9uVGFyZ2V0R3JvdXAodGFyZ2V0R3JvdXA6IElBcHBsaWNhdGlvblRhcmdldEdyb3VwKTogTG9hZEJhbGFuY2VyVGFyZ2V0UHJvcHM7XG59XG5cbmNvbnN0IEFMQl9IRUFMVEhfQ0hFQ0tfUFJPVE9DT0xTID0gW1Byb3RvY29sLkhUVFAsIFByb3RvY29sLkhUVFBTXTtcbiJdfQ==