"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const base_load_balancer_1 = require("../shared/base-load-balancer");
const network_listener_1 = require("./network-listener");
/**
 * Define a new network load balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class NetworkLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    static fromNetworkLoadBalancerAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.loadBalancerArn = attrs.loadBalancerArn;
                this.vpc = attrs.vpc;
            }
            addListener(lid, props) {
                return new network_listener_1.NetworkListener(this, lid, {
                    loadBalancer: this,
                    ...props
                });
            }
            get loadBalancerCanonicalHostedZoneId() {
                if (attrs.loadBalancerCanonicalHostedZoneId) {
                    return attrs.loadBalancerCanonicalHostedZoneId;
                }
                // tslint:disable-next-line:max-line-length
                throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
            get loadBalancerDnsName() {
                if (attrs.loadBalancerDnsName) {
                    return attrs.loadBalancerDnsName;
                }
                // tslint:disable-next-line:max-line-length
                throw new Error(`'loadBalancerDnsName' was not provided when constructing Network Load Balancer ${this.node.path} from attributes`);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: "network",
        });
        if (props.crossZoneEnabled) {
            this.setAttribute('load_balancing.cross_zone.enabled', 'true');
        }
    }
    /**
     * Add a listener to this load balancer
     *
     * @returns The newly created listener
     */
    addListener(id, props) {
        return new network_listener_1.NetworkListener(this, id, {
            loadBalancer: this,
            ...props
        });
    }
    /**
     * Return the given named metric for this Network Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props
        }).attachTo(this);
    }
    /**
     * The total number of concurrent TCP flows (or connections) from clients to targets.
     *
     * This metric includes connections in the SYN_SENT and ESTABLISHED states.
     * TCP connections are not terminated at the load balancer, so a client
     * opening a TCP connection to a target counts as a single flow.
     *
     * @default Average over 5 minutes
     */
    metricActiveFlowCount(props) {
        return this.metric('ActiveFlowCount', {
            statistic: 'Average',
            ...props
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props
        });
    }
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props
        });
    }
    /**
     * The total number of new TCP flows (or connections) established from clients to targets in the time period.
     *
     * @default Sum over 5 minutes
     */
    metricNewFlowCount(props) {
        return this.metric('NewFlowCount', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The total number of bytes processed by the load balancer, including TCP/IP headers.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The total number of reset (RST) packets sent from a client to a target.
     *
     * These resets are generated by the client and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpClientResetCount(props) {
        return this.metric('TCP_Client_Reset_Count', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The total number of reset (RST) packets generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpElbResetCount(props) {
        return this.metric('TCP_ELB_Reset_Count', {
            statistic: 'Sum',
            ...props
        });
    }
    /**
     * The total number of reset (RST) packets sent from a target to a client.
     *
     * These resets are generated by the target and forwarded by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricTcpTargetResetCount(props) {
        return this.metric('TCP_Target_Reset_Count', {
            statistic: 'Sum',
            ...props
        });
    }
}
exports.NetworkLoadBalancer = NetworkLoadBalancer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmV0d29yay1sb2FkLWJhbGFuY2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibmV0d29yay1sb2FkLWJhbGFuY2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsc0RBQXNEO0FBRXRELHdDQUFvRDtBQUNwRCxxRUFBd0c7QUFDeEcseURBQStFO0FBOEMvRTs7OztHQUlHO0FBQ0gsTUFBYSxtQkFBb0IsU0FBUSxxQ0FBZ0I7SUFDaEQsTUFBTSxDQUFDLGlDQUFpQyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9DO1FBQ2hILE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixvQkFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUM7Z0JBQ3hDLFFBQUcsR0FBYyxLQUFLLENBQUMsR0FBRyxDQUFDO1lBbUI3QyxDQUFDO1lBbEJRLFdBQVcsQ0FBQyxHQUFXLEVBQUUsS0FBK0I7Z0JBQzdELE9BQU8sSUFBSSxrQ0FBZSxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUU7b0JBQ3BDLFlBQVksRUFBRSxJQUFJO29CQUNsQixHQUFHLEtBQUs7aUJBQ1QsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztZQUVELElBQVcsaUNBQWlDO2dCQUMxQyxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsRUFBRTtvQkFBRSxPQUFPLEtBQUssQ0FBQyxpQ0FBaUMsQ0FBQztpQkFBRTtnQkFDaEcsMkNBQTJDO2dCQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLGdHQUFnRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksa0JBQWtCLENBQUMsQ0FBQztZQUNwSixDQUFDO1lBRUQsSUFBVyxtQkFBbUI7Z0JBQzVCLElBQUksS0FBSyxDQUFDLG1CQUFtQixFQUFFO29CQUFFLE9BQU8sS0FBSyxDQUFDLG1CQUFtQixDQUFDO2lCQUFFO2dCQUNwRSwyQ0FBMkM7Z0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsa0ZBQWtGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxDQUFDO1lBQ3RJLENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCO1FBQ3ZFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRTtZQUN0QixJQUFJLEVBQUUsU0FBUztTQUNoQixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsbUNBQW1DLEVBQUUsTUFBTSxDQUFDLENBQUM7U0FBRTtJQUNqRyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFdBQVcsQ0FBQyxFQUFVLEVBQUUsS0FBK0I7UUFDNUQsT0FBTyxJQUFJLGtDQUFlLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNuQyxZQUFZLEVBQUUsSUFBSTtZQUNsQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxVQUFrQixFQUFFLEtBQWdDO1FBQ2hFLE9BQU8sSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQzNCLFNBQVMsRUFBRSxnQkFBZ0I7WUFDM0IsVUFBVTtZQUNWLFVBQVUsRUFBRSxFQUFFLFlBQVksRUFBRSxJQUFJLENBQUMsb0JBQW9CLEVBQUU7WUFDdkQsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUFnQztRQUMzRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLEVBQUU7WUFDcEMsU0FBUyxFQUFFLFNBQVM7WUFDcEIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxrQkFBa0IsQ0FBQyxLQUFnQztRQUN4RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsY0FBYyxFQUFFO1lBQ2pDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBZ0M7UUFDNUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixFQUFFO1lBQ3JDLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksd0JBQXdCLENBQUMsS0FBZ0M7UUFDOUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLG9CQUFvQixFQUFFO1lBQ3ZDLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0JBQWtCLENBQUMsS0FBZ0M7UUFDeEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRTtZQUNqQyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG9CQUFvQixDQUFDLEtBQWdDO1FBQzFELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsRUFBRTtZQUNuQyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCLENBQUMsS0FBZ0M7UUFDL0QsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLHdCQUF3QixFQUFFO1lBQzNDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBZ0M7UUFDNUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLHFCQUFxQixFQUFFO1lBQ3hDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUIsQ0FBQyxLQUFnQztRQUMvRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsd0JBQXdCLEVBQUU7WUFDM0MsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBakxELGtEQWlMQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBSZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQmFzZUxvYWRCYWxhbmNlciwgQmFzZUxvYWRCYWxhbmNlclByb3BzLCBJTG9hZEJhbGFuY2VyVjIgfSBmcm9tICcuLi9zaGFyZWQvYmFzZS1sb2FkLWJhbGFuY2VyJztcbmltcG9ydCB7IEJhc2VOZXR3b3JrTGlzdGVuZXJQcm9wcywgTmV0d29ya0xpc3RlbmVyIH0gZnJvbSAnLi9uZXR3b3JrLWxpc3RlbmVyJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIG5ldHdvcmsgbG9hZCBiYWxhbmNlclxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5ldHdvcmtMb2FkQmFsYW5jZXJQcm9wcyBleHRlbmRzIEJhc2VMb2FkQmFsYW5jZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciBjcm9zcy16b25lIGxvYWQgYmFsYW5jaW5nIGlzIGVuYWJsZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBjcm9zc1pvbmVFbmFibGVkPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIHJlZmVyZW5jZSBhbiBleGlzdGluZyBsb2FkIGJhbGFuY2VyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTmV0d29ya0xvYWRCYWxhbmNlckF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogQVJOIG9mIHRoZSBsb2FkIGJhbGFuY2VyXG4gICAqL1xuICByZWFkb25seSBsb2FkQmFsYW5jZXJBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNhbm9uaWNhbCBob3N0ZWQgem9uZSBJRCBvZiB0aGlzIGxvYWQgYmFsYW5jZXJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBXaGVuIG5vdCBwcm92aWRlZCwgTEIgY2Fubm90IGJlIHVzZWQgYXMgUm91dGU1MyBBbGlhcyB0YXJnZXQuXG4gICAqL1xuICByZWFkb25seSBsb2FkQmFsYW5jZXJDYW5vbmljYWxIb3N0ZWRab25lSWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBETlMgbmFtZSBvZiB0aGlzIGxvYWQgYmFsYW5jZXJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBXaGVuIG5vdCBwcm92aWRlZCwgTEIgY2Fubm90IGJlIHVzZWQgYXMgUm91dGU1MyBBbGlhcyB0YXJnZXQuXG4gICAqL1xuICByZWFkb25seSBsb2FkQmFsYW5jZXJEbnNOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBsb2FkIGJhbGFuY2VyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFdoZW4gbm90IHByb3ZpZGVkLCBsaXN0ZW5lcnMgY2Fubm90IGJlIGNyZWF0ZWQgb24gaW1wb3J0ZWQgbG9hZFxuICAgKiBiYWxhbmNlcnMuXG4gICAqL1xuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcbn1cblxuLyoqXG4gKiBEZWZpbmUgYSBuZXcgbmV0d29yayBsb2FkIGJhbGFuY2VyXG4gKlxuICogQHJlc291cmNlIEFXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TG9hZEJhbGFuY2VyXG4gKi9cbmV4cG9ydCBjbGFzcyBOZXR3b3JrTG9hZEJhbGFuY2VyIGV4dGVuZHMgQmFzZUxvYWRCYWxhbmNlciBpbXBsZW1lbnRzIElOZXR3b3JrTG9hZEJhbGFuY2VyIHtcbiAgcHVibGljIHN0YXRpYyBmcm9tTmV0d29ya0xvYWRCYWxhbmNlckF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IE5ldHdvcmtMb2FkQmFsYW5jZXJBdHRyaWJ1dGVzKTogSU5ldHdvcmtMb2FkQmFsYW5jZXIge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSU5ldHdvcmtMb2FkQmFsYW5jZXIge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGxvYWRCYWxhbmNlckFybiA9IGF0dHJzLmxvYWRCYWxhbmNlckFybjtcbiAgICAgIHB1YmxpYyByZWFkb25seSB2cGM/OiBlYzIuSVZwYyA9IGF0dHJzLnZwYztcbiAgICAgIHB1YmxpYyBhZGRMaXN0ZW5lcihsaWQ6IHN0cmluZywgcHJvcHM6IEJhc2VOZXR3b3JrTGlzdGVuZXJQcm9wcyk6IE5ldHdvcmtMaXN0ZW5lciB7XG4gICAgICAgIHJldHVybiBuZXcgTmV0d29ya0xpc3RlbmVyKHRoaXMsIGxpZCwge1xuICAgICAgICAgIGxvYWRCYWxhbmNlcjogdGhpcyxcbiAgICAgICAgICAuLi5wcm9wc1xuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgcHVibGljIGdldCBsb2FkQmFsYW5jZXJDYW5vbmljYWxIb3N0ZWRab25lSWQoKTogc3RyaW5nIHtcbiAgICAgICAgaWYgKGF0dHJzLmxvYWRCYWxhbmNlckNhbm9uaWNhbEhvc3RlZFpvbmVJZCkgeyByZXR1cm4gYXR0cnMubG9hZEJhbGFuY2VyQ2Fub25pY2FsSG9zdGVkWm9uZUlkOyB9XG4gICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAnbG9hZEJhbGFuY2VyQ2Fub25pY2FsSG9zdGVkWm9uZUlkJyB3YXMgbm90IHByb3ZpZGVkIHdoZW4gY29uc3RydWN0aW5nIE5ldHdvcmsgTG9hZCBCYWxhbmNlciAke3RoaXMubm9kZS5wYXRofSBmcm9tIGF0dHJpYnV0ZXNgKTtcbiAgICAgIH1cblxuICAgICAgcHVibGljIGdldCBsb2FkQmFsYW5jZXJEbnNOYW1lKCk6IHN0cmluZyB7XG4gICAgICAgIGlmIChhdHRycy5sb2FkQmFsYW5jZXJEbnNOYW1lKSB7IHJldHVybiBhdHRycy5sb2FkQmFsYW5jZXJEbnNOYW1lOyB9XG4gICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAnbG9hZEJhbGFuY2VyRG5zTmFtZScgd2FzIG5vdCBwcm92aWRlZCB3aGVuIGNvbnN0cnVjdGluZyBOZXR3b3JrIExvYWQgQmFsYW5jZXIgJHt0aGlzLm5vZGUucGF0aH0gZnJvbSBhdHRyaWJ1dGVzYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBOZXR3b3JrTG9hZEJhbGFuY2VyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzLCB7XG4gICAgICB0eXBlOiBcIm5ldHdvcmtcIixcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy5jcm9zc1pvbmVFbmFibGVkKSB7IHRoaXMuc2V0QXR0cmlidXRlKCdsb2FkX2JhbGFuY2luZy5jcm9zc196b25lLmVuYWJsZWQnLCAndHJ1ZScpOyB9XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgbGlzdGVuZXIgdG8gdGhpcyBsb2FkIGJhbGFuY2VyXG4gICAqXG4gICAqIEByZXR1cm5zIFRoZSBuZXdseSBjcmVhdGVkIGxpc3RlbmVyXG4gICAqL1xuICBwdWJsaWMgYWRkTGlzdGVuZXIoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2VOZXR3b3JrTGlzdGVuZXJQcm9wcyk6IE5ldHdvcmtMaXN0ZW5lciB7XG4gICAgcmV0dXJuIG5ldyBOZXR3b3JrTGlzdGVuZXIodGhpcywgaWQsIHtcbiAgICAgIGxvYWRCYWxhbmNlcjogdGhpcyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIHRoZSBnaXZlbiBuYW1lZCBtZXRyaWMgZm9yIHRoaXMgTmV0d29yayBMb2FkIEJhbGFuY2VyXG4gICAqXG4gICAqIEBkZWZhdWx0IEF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9OZXR3b3JrRUxCJyxcbiAgICAgIG1ldHJpY05hbWUsXG4gICAgICBkaW1lbnNpb25zOiB7IExvYWRCYWxhbmNlcjogdGhpcy5sb2FkQmFsYW5jZXJGdWxsTmFtZSB9LFxuICAgICAgLi4ucHJvcHNcbiAgICB9KS5hdHRhY2hUbyh0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgdG90YWwgbnVtYmVyIG9mIGNvbmN1cnJlbnQgVENQIGZsb3dzIChvciBjb25uZWN0aW9ucykgZnJvbSBjbGllbnRzIHRvIHRhcmdldHMuXG4gICAqXG4gICAqIFRoaXMgbWV0cmljIGluY2x1ZGVzIGNvbm5lY3Rpb25zIGluIHRoZSBTWU5fU0VOVCBhbmQgRVNUQUJMSVNIRUQgc3RhdGVzLlxuICAgKiBUQ1AgY29ubmVjdGlvbnMgYXJlIG5vdCB0ZXJtaW5hdGVkIGF0IHRoZSBsb2FkIGJhbGFuY2VyLCBzbyBhIGNsaWVudFxuICAgKiBvcGVuaW5nIGEgVENQIGNvbm5lY3Rpb24gdG8gYSB0YXJnZXQgY291bnRzIGFzIGEgc2luZ2xlIGZsb3cuXG4gICAqXG4gICAqIEBkZWZhdWx0IEF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNBY3RpdmVGbG93Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0FjdGl2ZUZsb3dDb3VudCcsIHtcbiAgICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnLFxuICAgICAgLi4ucHJvcHNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGxvYWQgYmFsYW5jZXIgY2FwYWNpdHkgdW5pdHMgKExDVSkgdXNlZCBieSB5b3VyIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY0NvbnN1bWVkTENVcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ29uc3VtZWRMQ1VzJywge1xuICAgICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiB0YXJnZXRzIHRoYXQgYXJlIGNvbnNpZGVyZWQgaGVhbHRoeS5cbiAgICpcbiAgICogQGRlZmF1bHQgQXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY0hlYWx0aHlIb3N0Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0hlYWx0aHlIb3N0Q291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdBdmVyYWdlJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiB0YXJnZXRzIHRoYXQgYXJlIGNvbnNpZGVyZWQgdW5oZWFsdGh5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBBdmVyYWdlIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljVW5IZWFsdGh5SG9zdENvdW50KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdVbkhlYWx0aHlIb3N0Q291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdBdmVyYWdlJyxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHRvdGFsIG51bWJlciBvZiBuZXcgVENQIGZsb3dzIChvciBjb25uZWN0aW9ucykgZXN0YWJsaXNoZWQgZnJvbSBjbGllbnRzIHRvIHRhcmdldHMgaW4gdGhlIHRpbWUgcGVyaW9kLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNOZXdGbG93Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ05ld0Zsb3dDb3VudCcsIHtcbiAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB0b3RhbCBudW1iZXIgb2YgYnl0ZXMgcHJvY2Vzc2VkIGJ5IHRoZSBsb2FkIGJhbGFuY2VyLCBpbmNsdWRpbmcgVENQL0lQIGhlYWRlcnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY1Byb2Nlc3NlZEJ5dGVzKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdQcm9jZXNzZWRCeXRlcycsIHtcbiAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB0b3RhbCBudW1iZXIgb2YgcmVzZXQgKFJTVCkgcGFja2V0cyBzZW50IGZyb20gYSBjbGllbnQgdG8gYSB0YXJnZXQuXG4gICAqXG4gICAqIFRoZXNlIHJlc2V0cyBhcmUgZ2VuZXJhdGVkIGJ5IHRoZSBjbGllbnQgYW5kIGZvcndhcmRlZCBieSB0aGUgbG9hZCBiYWxhbmNlci5cbiAgICpcbiAgICogQGRlZmF1bHQgU3VtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljVGNwQ2xpZW50UmVzZXRDb3VudChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnVENQX0NsaWVudF9SZXNldF9Db3VudCcsIHtcbiAgICAgIHN0YXRpc3RpYzogJ1N1bScsXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB0b3RhbCBudW1iZXIgb2YgcmVzZXQgKFJTVCkgcGFja2V0cyBnZW5lcmF0ZWQgYnkgdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY1RjcEVsYlJlc2V0Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1RDUF9FTEJfUmVzZXRfQ291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgLi4ucHJvcHNcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgdG90YWwgbnVtYmVyIG9mIHJlc2V0IChSU1QpIHBhY2tldHMgc2VudCBmcm9tIGEgdGFyZ2V0IHRvIGEgY2xpZW50LlxuICAgKlxuICAgKiBUaGVzZSByZXNldHMgYXJlIGdlbmVyYXRlZCBieSB0aGUgdGFyZ2V0IGFuZCBmb3J3YXJkZWQgYnkgdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFN1bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY1RjcFRhcmdldFJlc2V0Q291bnQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1RDUF9UYXJnZXRfUmVzZXRfQ291bnQnLCB7XG4gICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgICAgLi4ucHJvcHNcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIEEgbmV0d29yayBsb2FkIGJhbGFuY2VyXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU5ldHdvcmtMb2FkQmFsYW5jZXIgZXh0ZW5kcyBJTG9hZEJhbGFuY2VyVjIsIGVjMi5JVnBjRW5kcG9pbnRTZXJ2aWNlTG9hZEJhbGFuY2VyIHtcblxuICAvKipcbiAgICogVGhlIFZQQyB0aGlzIGxvYWQgYmFsYW5jZXIgaGFzIGJlZW4gY3JlYXRlZCBpbiAoaWYgYXZhaWxhYmxlKVxuICAgKi9cbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIEFkZCBhIGxpc3RlbmVyIHRvIHRoaXMgbG9hZCBiYWxhbmNlclxuICAgKlxuICAgKiBAcmV0dXJucyBUaGUgbmV3bHkgY3JlYXRlZCBsaXN0ZW5lclxuICAgKi9cbiAgYWRkTGlzdGVuZXIoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2VOZXR3b3JrTGlzdGVuZXJQcm9wcyk6IE5ldHdvcmtMaXN0ZW5lcjtcbn1cbiJdfQ==