"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
/**
 * Define a Network Target Group
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props, {
            protocol: enums_1.Protocol.TCP,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing listener
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`
            ].join(' '));
        }
        return ret;
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmV0d29yay10YXJnZXQtZ3JvdXAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJuZXR3b3JrLXRhcmdldC1ncm91cC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHFDQUFxQztBQUNyQyxtRUFDNkc7QUFDN0csMkNBQTJDO0FBQzNDLGlEQUE2RDtBQStCN0Q7O0dBRUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLG1DQUFlO0lBbUJyRCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQThCO1FBQzFFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRTtZQUN0QixRQUFRLEVBQUUsZ0JBQVEsQ0FBQyxHQUFHO1lBQ3RCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLEVBQUUsQ0FBQztRQUVwQixJQUFJLEtBQUssQ0FBQyxlQUFlLElBQUksSUFBSSxFQUFFO1lBQ2pDLElBQUksQ0FBQyxZQUFZLENBQUMsMkJBQTJCLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUMxRjtRQUVELElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBL0JEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLHlCQUF5QixDQUFDLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBQ3BHLE9BQU8sSUFBSSwwQkFBMEIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNsRixPQUFPLGtCQUFrQixDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQW1CRDs7T0FFRztJQUNJLFNBQVMsQ0FBQyxHQUFHLE9BQXFDO1FBQ3ZELEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzVCLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN2RCxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDcEM7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGdCQUFnQixDQUFDLFFBQTBCO1FBQ2hELElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyx5QkFBeUI7UUFDbEMsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RkFBd0YsQ0FBQyxDQUFDO1NBQzNHO1FBQ0QsT0FBTyxtREFBK0IsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ3hFLENBQUM7SUFFUyxRQUFRO1FBQ2hCLE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUU3QixNQUFNLFdBQVcsR0FBZ0IsSUFBSSxDQUFDLFdBQVcsSUFBSSxFQUFFLENBQUM7UUFFeEQsTUFBTSxnQkFBZ0IsR0FBRyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNsQyxJQUFJLFdBQVcsQ0FBQyxRQUFRLEVBQUU7WUFDeEIsTUFBTSxPQUFPLEdBQUcsV0FBVyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUNqRCxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7Z0JBQzNFLEdBQUcsQ0FBQyxJQUFJLENBQUMsMEJBQTBCLE9BQU8seURBQXlELGdCQUFnQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDcEk7U0FDRjtRQUVELElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFO1lBQ3pCLE9BQU8sR0FBRyxDQUFDO1NBQ1o7UUFFRCxJQUFJLENBQUMsMEJBQTBCLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUM5RCxHQUFHLENBQUMsSUFBSSxDQUFDLDBCQUEwQixXQUFXLENBQUMsUUFBUSx1Q0FBdUMsMEJBQTBCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN6STtRQUNELElBQUksV0FBVyxDQUFDLElBQUksSUFBSSxDQUFDLCtCQUErQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDdkYsR0FBRyxDQUFDLElBQUksQ0FBQztnQkFDUCxJQUFJLFdBQVcsQ0FBQyxRQUFRLG1EQUFtRDtnQkFDM0UsbUJBQW1CLCtCQUErQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRzthQUNqRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1NBQ2Q7UUFDRCxJQUFJLFdBQVcsQ0FBQyxPQUFPLElBQUksV0FBVyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsS0FBSyx5QkFBeUIsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDOUcsR0FBRyxDQUFDLElBQUksQ0FBQztnQkFDUCx5RkFBeUY7Z0JBQ3pGLFlBQVkseUJBQXlCLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsV0FBVyxDQUFDLFFBQVEsU0FBUyxXQUFXLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxFQUFFO2FBQzFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDZDtRQUVELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztDQUNGO0FBbkdELGdEQW1HQztBQW9CRDs7R0FFRztBQUNILE1BQU0sMEJBQTJCLFNBQVEsa0NBQXVCO0lBQ3ZELGdCQUFnQixDQUFDLFNBQTJCO1FBQ2pELGdEQUFnRDtJQUNsRCxDQUFDO0lBRU0sU0FBUyxDQUFDLEdBQUcsT0FBcUM7UUFDdkQsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDNUIsTUFBTSxNQUFNLEdBQUcsTUFBTSxDQUFDLDBCQUEwQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3ZELElBQUksTUFBTSxDQUFDLFVBQVUsS0FBSyxTQUFTLEVBQUU7Z0JBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsd0dBQXdHLENBQUMsQ0FBQzthQUMzSDtTQUNGO0lBQ0gsQ0FBQztDQUNGO0FBZUQsTUFBTSwwQkFBMEIsR0FBRyxDQUFDLGdCQUFRLENBQUMsSUFBSSxFQUFFLGdCQUFRLENBQUMsS0FBSyxFQUFFLGdCQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDakYsTUFBTSwrQkFBK0IsR0FBRyxDQUFDLGdCQUFRLENBQUMsSUFBSSxFQUFFLGdCQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDeEUsTUFBTSx5QkFBeUIsR0FBdUM7SUFDcEUsQ0FBQyxnQkFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7SUFDbEIsQ0FBQyxnQkFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUU7SUFDcEIsQ0FBQyxnQkFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUU7Q0FDbkIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEJhc2VUYXJnZXRHcm91cFByb3BzLCBIZWFsdGhDaGVjaywgSVRhcmdldEdyb3VwLCBsb2FkQmFsYW5jZXJOYW1lRnJvbUxpc3RlbmVyQXJuLCBMb2FkQmFsYW5jZXJUYXJnZXRQcm9wcyxcbiAgICAgICAgIFRhcmdldEdyb3VwQXR0cmlidXRlcywgVGFyZ2V0R3JvdXBCYXNlLCBUYXJnZXRHcm91cEltcG9ydFByb3BzIH0gZnJvbSAnLi4vc2hhcmVkL2Jhc2UtdGFyZ2V0LWdyb3VwJztcbmltcG9ydCB7IFByb3RvY29sIH0gZnJvbSAnLi4vc2hhcmVkL2VudW1zJztcbmltcG9ydCB7IEltcG9ydGVkVGFyZ2V0R3JvdXBCYXNlIH0gZnJvbSAnLi4vc2hhcmVkL2ltcG9ydGVkJztcbmltcG9ydCB7IElOZXR3b3JrTGlzdGVuZXIgfSBmcm9tICcuL25ldHdvcmstbGlzdGVuZXInO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IE5ldHdvcmsgVGFyZ2V0IEdyb3VwXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTmV0d29ya1RhcmdldEdyb3VwUHJvcHMgZXh0ZW5kcyBCYXNlVGFyZ2V0R3JvdXBQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgcG9ydCBvbiB3aGljaCB0aGUgbGlzdGVuZXIgbGlzdGVucyBmb3IgcmVxdWVzdHMuXG4gICAqL1xuICByZWFkb25seSBwb3J0OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIFByb3h5IFByb3RvY29sIHZlcnNpb24gMiBpcyBlbmFibGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJveHlQcm90b2NvbFYyPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIHRhcmdldHMgdG8gYWRkIHRvIHRoaXMgdGFyZ2V0IGdyb3VwLlxuICAgKlxuICAgKiBDYW4gYmUgYEluc3RhbmNlYCwgYElQQWRkcmVzc2AsIG9yIGFueSBzZWxmLXJlZ2lzdGVyaW5nIGxvYWQgYmFsYW5jaW5nXG4gICAqIHRhcmdldC4gSWYgeW91IHVzZSBlaXRoZXIgYEluc3RhbmNlYCBvciBgSVBBZGRyZXNzYCBhcyB0YXJnZXRzLCBhbGxcbiAgICogdGFyZ2V0IG11c3QgYmUgb2YgdGhlIHNhbWUgdHlwZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyB0YXJnZXRzLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFyZ2V0cz86IElOZXR3b3JrTG9hZEJhbGFuY2VyVGFyZ2V0W107XG59XG5cbi8qKlxuICogRGVmaW5lIGEgTmV0d29yayBUYXJnZXQgR3JvdXBcbiAqL1xuZXhwb3J0IGNsYXNzIE5ldHdvcmtUYXJnZXRHcm91cCBleHRlbmRzIFRhcmdldEdyb3VwQmFzZSBpbXBsZW1lbnRzIElOZXR3b3JrVGFyZ2V0R3JvdXAge1xuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIHRhcmdldCBncm91cFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzKTogSU5ldHdvcmtUYXJnZXRHcm91cCB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZE5ldHdvcmtUYXJnZXRHcm91cChzY29wZSwgaWQsIGF0dHJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgbGlzdGVuZXJcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBmcm9tVGFyZ2V0R3JvdXBBdHRyaWJ1dGVzYCBpbnN0ZWFkXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGltcG9ydChzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFRhcmdldEdyb3VwSW1wb3J0UHJvcHMpOiBJTmV0d29ya1RhcmdldEdyb3VwIHtcbiAgICByZXR1cm4gTmV0d29ya1RhcmdldEdyb3VwLmZyb21UYXJnZXRHcm91cEF0dHJpYnV0ZXMoc2NvcGUsIGlkLCBwcm9wcyk7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IGxpc3RlbmVyczogSU5ldHdvcmtMaXN0ZW5lcltdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTmV0d29ya1RhcmdldEdyb3VwUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzLCB7XG4gICAgICBwcm90b2NvbDogUHJvdG9jb2wuVENQLFxuICAgICAgcG9ydDogcHJvcHMucG9ydCxcbiAgICB9KTtcblxuICAgIHRoaXMubGlzdGVuZXJzID0gW107XG5cbiAgICBpZiAocHJvcHMucHJveHlQcm90b2NvbFYyICE9IG51bGwpIHtcbiAgICAgIHRoaXMuc2V0QXR0cmlidXRlKCdwcm94eV9wcm90b2NvbF92Mi5lbmFibGVkJywgcHJvcHMucHJveHlQcm90b2NvbFYyID8gJ3RydWUnIDogJ2ZhbHNlJyk7XG4gICAgfVxuXG4gICAgdGhpcy5hZGRUYXJnZXQoLi4uKHByb3BzLnRhcmdldHMgfHwgW10pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBsb2FkIGJhbGFuY2luZyB0YXJnZXQgdG8gdGhpcyB0YXJnZXQgZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBhZGRUYXJnZXQoLi4udGFyZ2V0czogSU5ldHdvcmtMb2FkQmFsYW5jZXJUYXJnZXRbXSkge1xuICAgIGZvciAoY29uc3QgdGFyZ2V0IG9mIHRhcmdldHMpIHtcbiAgICAgIGNvbnN0IHJlc3VsdCA9IHRhcmdldC5hdHRhY2hUb05ldHdvcmtUYXJnZXRHcm91cCh0aGlzKTtcbiAgICAgIHRoaXMuYWRkTG9hZEJhbGFuY2VyVGFyZ2V0KHJlc3VsdCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVyIGEgbGlzdGVuZXIgdGhhdCBpcyBsb2FkIGJhbGFuY2luZyB0byB0aGlzIHRhcmdldCBncm91cC5cbiAgICpcbiAgICogRG9uJ3QgY2FsbCB0aGlzIGRpcmVjdGx5LiBJdCB3aWxsIGJlIGNhbGxlZCBieSBsaXN0ZW5lcnMuXG4gICAqL1xuICBwdWJsaWMgcmVnaXN0ZXJMaXN0ZW5lcihsaXN0ZW5lcjogSU5ldHdvcmtMaXN0ZW5lcikge1xuICAgIHRoaXMubG9hZEJhbGFuY2VyQXR0YWNoZWREZXBlbmRlbmNpZXMuYWRkKGxpc3RlbmVyKTtcbiAgICB0aGlzLmxpc3RlbmVycy5wdXNoKGxpc3RlbmVyKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGdWxsIG5hbWUgb2YgZmlyc3QgbG9hZCBiYWxhbmNlclxuICAgKi9cbiAgcHVibGljIGdldCBmaXJzdExvYWRCYWxhbmNlckZ1bGxOYW1lKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMubGlzdGVuZXJzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgVGFyZ2V0R3JvdXAgbmVlZHMgdG8gYmUgYXR0YWNoZWQgdG8gYSBMb2FkQmFsYW5jZXIgYmVmb3JlIHlvdSBjYW4gY2FsbCB0aGlzIG1ldGhvZCcpO1xuICAgIH1cbiAgICByZXR1cm4gbG9hZEJhbGFuY2VyTmFtZUZyb21MaXN0ZW5lckFybih0aGlzLmxpc3RlbmVyc1swXS5saXN0ZW5lckFybik7XG4gIH1cblxuICBwcm90ZWN0ZWQgdmFsaWRhdGUoKTogc3RyaW5nW10gIHtcbiAgICBjb25zdCByZXQgPSBzdXBlci52YWxpZGF0ZSgpO1xuXG4gICAgY29uc3QgaGVhbHRoQ2hlY2s6IEhlYWx0aENoZWNrID0gdGhpcy5oZWFsdGhDaGVjayB8fCB7fTtcblxuICAgIGNvbnN0IGFsbG93ZWRJbnRlcnZhbHMgPSBbMTAsIDMwXTtcbiAgICBpZiAoaGVhbHRoQ2hlY2suaW50ZXJ2YWwpIHtcbiAgICAgIGNvbnN0IHNlY29uZHMgPSBoZWFsdGhDaGVjay5pbnRlcnZhbC50b1NlY29uZHMoKTtcbiAgICAgIGlmICghY2RrLlRva2VuLmlzVW5yZXNvbHZlZChzZWNvbmRzKSAmJiAhYWxsb3dlZEludGVydmFscy5pbmNsdWRlcyhzZWNvbmRzKSkge1xuICAgICAgICByZXQucHVzaChgSGVhbHRoIGNoZWNrIGludGVydmFsICcke3NlY29uZHN9JyBub3Qgc3VwcG9ydGVkLiBNdXN0IGJlIG9uZSBvZiB0aGUgZm9sbG93aW5nIHZhbHVlcyAnJHthbGxvd2VkSW50ZXJ2YWxzLmpvaW4oJywnKX0nLmApO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmICghaGVhbHRoQ2hlY2sucHJvdG9jb2wpIHtcbiAgICAgIHJldHVybiByZXQ7XG4gICAgfVxuXG4gICAgaWYgKCFOTEJfSEVBTFRIX0NIRUNLX1BST1RPQ09MUy5pbmNsdWRlcyhoZWFsdGhDaGVjay5wcm90b2NvbCkpIHtcbiAgICAgIHJldC5wdXNoKGBIZWFsdGggY2hlY2sgcHJvdG9jb2wgJyR7aGVhbHRoQ2hlY2sucHJvdG9jb2x9JyBpcyBub3Qgc3VwcG9ydGVkLiBNdXN0IGJlIG9uZSBvZiBbJHtOTEJfSEVBTFRIX0NIRUNLX1BST1RPQ09MUy5qb2luKCcsICcpfV1gKTtcbiAgICB9XG4gICAgaWYgKGhlYWx0aENoZWNrLnBhdGggJiYgIU5MQl9QQVRIX0hFQUxUSF9DSEVDS19QUk9UT0NPTFMuaW5jbHVkZXMoaGVhbHRoQ2hlY2sucHJvdG9jb2wpKSB7XG4gICAgICByZXQucHVzaChbXG4gICAgICAgIGAnJHtoZWFsdGhDaGVjay5wcm90b2NvbH0nIGhlYWx0aCBjaGVja3MgZG8gbm90IHN1cHBvcnQgdGhlIHBhdGggcHJvcGVydHkuYCxcbiAgICAgICAgYE11c3QgYmUgb25lIG9mIFske05MQl9QQVRIX0hFQUxUSF9DSEVDS19QUk9UT0NPTFMuam9pbignLCAnKX1dYFxuICAgICAgXS5qb2luKCcgJykpO1xuICAgIH1cbiAgICBpZiAoaGVhbHRoQ2hlY2sudGltZW91dCAmJiBoZWFsdGhDaGVjay50aW1lb3V0LnRvU2Vjb25kcygpICE9PSBOTEJfSEVBTFRIX0NIRUNLX1RJTUVPVVRTW2hlYWx0aENoZWNrLnByb3RvY29sXSkge1xuICAgICAgcmV0LnB1c2goW1xuICAgICAgICAnQ3VzdG9tIGhlYWx0aCBjaGVjayB0aW1lb3V0cyBhcmUgbm90IHN1cHBvcnRlZCBmb3IgTmV0d29yayBMb2FkIEJhbGFuY2VyIGhlYWx0aCBjaGVja3MuJyxcbiAgICAgICAgYEV4cGVjdGVkICR7TkxCX0hFQUxUSF9DSEVDS19USU1FT1VUU1toZWFsdGhDaGVjay5wcm90b2NvbF19IHNlY29uZHMgZm9yICR7aGVhbHRoQ2hlY2sucHJvdG9jb2x9LCBnb3QgJHtoZWFsdGhDaGVjay50aW1lb3V0LnRvU2Vjb25kcygpfWBcbiAgICAgIF0uam9pbignICcpKTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmV0O1xuICB9XG59XG5cbi8qKlxuICogQSBuZXR3b3JrIHRhcmdldCBncm91cFxuICovXG4vLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bm8tZW1wdHktaW50ZXJmYWNlXG5leHBvcnQgaW50ZXJmYWNlIElOZXR3b3JrVGFyZ2V0R3JvdXAgZXh0ZW5kcyBJVGFyZ2V0R3JvdXAge1xuICAvKipcbiAgICogUmVnaXN0ZXIgYSBsaXN0ZW5lciB0aGF0IGlzIGxvYWQgYmFsYW5jaW5nIHRvIHRoaXMgdGFyZ2V0IGdyb3VwLlxuICAgKlxuICAgKiBEb24ndCBjYWxsIHRoaXMgZGlyZWN0bHkuIEl0IHdpbGwgYmUgY2FsbGVkIGJ5IGxpc3RlbmVycy5cbiAgICovXG4gIHJlZ2lzdGVyTGlzdGVuZXIobGlzdGVuZXI6IElOZXR3b3JrTGlzdGVuZXIpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBBZGQgYSBsb2FkIGJhbGFuY2luZyB0YXJnZXQgdG8gdGhpcyB0YXJnZXQgZ3JvdXBcbiAgICovXG4gIGFkZFRhcmdldCguLi50YXJnZXRzOiBJTmV0d29ya0xvYWRCYWxhbmNlclRhcmdldFtdKTogdm9pZDtcbn1cblxuLyoqXG4gKiBBbiBpbXBvcnRlZCBuZXR3b3JrIHRhcmdldCBncm91cFxuICovXG5jbGFzcyBJbXBvcnRlZE5ldHdvcmtUYXJnZXRHcm91cCBleHRlbmRzIEltcG9ydGVkVGFyZ2V0R3JvdXBCYXNlIGltcGxlbWVudHMgSU5ldHdvcmtUYXJnZXRHcm91cCB7XG4gIHB1YmxpYyByZWdpc3Rlckxpc3RlbmVyKF9saXN0ZW5lcjogSU5ldHdvcmtMaXN0ZW5lcikge1xuICAgIC8vIE5vdGhpbmcgdG8gZG8sIHdlIGtub3cgbm90aGluZyBvZiBvdXIgbWVtYmVyc1xuICB9XG5cbiAgcHVibGljIGFkZFRhcmdldCguLi50YXJnZXRzOiBJTmV0d29ya0xvYWRCYWxhbmNlclRhcmdldFtdKSB7XG4gICAgZm9yIChjb25zdCB0YXJnZXQgb2YgdGFyZ2V0cykge1xuICAgICAgY29uc3QgcmVzdWx0ID0gdGFyZ2V0LmF0dGFjaFRvTmV0d29ya1RhcmdldEdyb3VwKHRoaXMpO1xuICAgICAgaWYgKHJlc3VsdC50YXJnZXRKc29uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYWRkIGEgbm9uLXNlbGYgcmVnaXN0ZXJpbmcgdGFyZ2V0IHRvIGFuIGltcG9ydGVkIFRhcmdldEdyb3VwLiBDcmVhdGUgYSBuZXcgVGFyZ2V0R3JvdXAgaW5zdGVhZC4nKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgZm9yIGNvbnN0cnVjdHMgdGhhdCBjYW4gYmUgdGFyZ2V0cyBvZiBhbiBuZXR3b3JrIGxvYWQgYmFsYW5jZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJTmV0d29ya0xvYWRCYWxhbmNlclRhcmdldCB7XG4gIC8qKlxuICAgKiBBdHRhY2ggbG9hZC1iYWxhbmNlZCB0YXJnZXQgdG8gYSBUYXJnZXRHcm91cFxuICAgKlxuICAgKiBNYXkgcmV0dXJuIEpTT04gdG8gZGlyZWN0bHkgYWRkIHRvIHRoZSBbVGFyZ2V0c10gbGlzdCwgb3IgcmV0dXJuIHVuZGVmaW5lZFxuICAgKiBpZiB0aGUgdGFyZ2V0IHdpbGwgcmVnaXN0ZXIgaXRzZWxmIHdpdGggdGhlIGxvYWQgYmFsYW5jZXIuXG4gICAqL1xuICBhdHRhY2hUb05ldHdvcmtUYXJnZXRHcm91cCh0YXJnZXRHcm91cDogSU5ldHdvcmtUYXJnZXRHcm91cCk6IExvYWRCYWxhbmNlclRhcmdldFByb3BzO1xufVxuXG5jb25zdCBOTEJfSEVBTFRIX0NIRUNLX1BST1RPQ09MUyA9IFtQcm90b2NvbC5IVFRQLCBQcm90b2NvbC5IVFRQUywgUHJvdG9jb2wuVENQXTtcbmNvbnN0IE5MQl9QQVRIX0hFQUxUSF9DSEVDS19QUk9UT0NPTFMgPSBbUHJvdG9jb2wuSFRUUCwgUHJvdG9jb2wuSFRUUFNdO1xuY29uc3QgTkxCX0hFQUxUSF9DSEVDS19USU1FT1VUUzoge1twcm90b2NvbCBpbiBQcm90b2NvbF0/OiBudW1iZXJ9ID0gIHtcbiAgW1Byb3RvY29sLkhUVFBdOiA2LFxuICBbUHJvdG9jb2wuSFRUUFNdOiAxMCxcbiAgW1Byb3RvY29sLlRDUF06IDEwLFxufTtcbiJdfQ==