"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        if (baseProps.deletionProtection) {
            this.setAttribute('deletion_protection.enabled', 'true');
        }
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Region is required to enable ELBv2 access logging`);
        }
        const account = ELBV2_ACCOUNTS[region];
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + "/" : "")}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
// https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html#access-logging-bucket-permissions
const ELBV2_ACCOUNTS = {
    'us-east-1': '127311923021',
    'us-east-2': '033677994240',
    'us-west-1': '027434742980',
    'us-west-2': '797873946194',
    'ca-central-1': '985666609251',
    'eu-central-1': '054676820928',
    'eu-west-1': '156460612806',
    'eu-west-2': '652711504416',
    'eu-west-3': '009996457667',
    'eu-north-1': '897822967062',
    'ap-east-1': '754344448648',
    'ap-northeast-1': '582318560864',
    'ap-northeast-2': '600734575887',
    'ap-northeast-3': '383597477331',
    'ap-southeast-1': '114774131450',
    'ap-southeast-2': '783225319266',
    'ap-south-1': '718504428378',
    'me-south-1': '076674570225',
    'sa-east-1': '507241528517',
    'us-gov-west-1': '048591011584',
    'us-gov-east-1': '190560391635',
    'cn-north-1': '638102146993',
    'cn-northwest-1': '037604701340',
};
//# sourceMappingURL=data:application/json;base64,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