"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends cdk.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.stringValue({ produce: () => this.targetType }),
            targets: cdk.Lazy.anyValue({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.stringValue({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckIntervalSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds()
            }),
            healthCheckPath: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds()
            }),
            healthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes
                } : undefined }),
            ...additionalProps
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error(`TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.`);
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            this.node.addWarning(`When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).`);
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push(`'vpc' is required for a non-Lambda TargetGroup`);
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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