"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// tslint:disable-next-line:max-line-length
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxtQ0FBOEM7QUFJOUM7O0dBRUc7QUFDSCxTQUFnQixnQkFBZ0IsQ0FBQyxVQUFzQjtJQUNyRCxNQUFNLEdBQUcsR0FBVSxFQUFFLENBQUM7SUFDdEIsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDckQsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztTQUMxQjtLQUNGO0lBQ0QsT0FBTyxHQUFHLENBQUM7QUFDYixDQUFDO0FBUkQsNENBUUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLHNCQUFzQixDQUFDLEtBQTBCO0lBQy9ELFFBQVEsS0FBSyxFQUFFO1FBQ2IsS0FBSywyQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUN6QyxLQUFLLDJCQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sR0FBRyxDQUFDO1FBQzNDO1lBQ0UsTUFBTSxJQUFJLEtBQUssQ0FBQywwQkFBMEIsS0FBSyxFQUFFLENBQUMsQ0FBQztLQUN0RDtBQUNILENBQUM7QUFQRCx3REFPQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0Isc0JBQXNCLENBQUMsSUFBWTtJQUNqRCxRQUFRLElBQUksRUFBRTtRQUNaLEtBQUssRUFBRSxDQUFDO1FBQ1IsS0FBSyxJQUFJLENBQUM7UUFDVixLQUFLLElBQUksQ0FBQztRQUNWLEtBQUssSUFBSTtZQUNQLE9BQU8sMkJBQW1CLENBQUMsSUFBSSxDQUFDO1FBRWxDLEtBQUssR0FBRyxDQUFDO1FBQ1QsS0FBSyxJQUFJO1lBQ1AsT0FBTywyQkFBbUIsQ0FBQyxLQUFLLENBQUM7UUFFbkM7WUFDRSxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxJQUFJLDRCQUE0QixDQUFDLENBQUM7S0FDOUY7QUFDSCxDQUFDO0FBZkQsd0RBZUM7QUFFRDs7R0FFRztBQUNILDJDQUEyQztBQUMzQyxTQUFnQix3QkFBd0IsQ0FBQyxRQUF5QyxFQUFFLElBQXdCO0lBQzFHLElBQUksUUFBUSxLQUFLLFNBQVMsSUFBSSxJQUFJLEtBQUssU0FBUyxFQUFFO1FBQ2hELE9BQU8sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUM7S0FDL0I7SUFFRCxJQUFJLFFBQVEsS0FBSyxTQUFTLEVBQUU7UUFBRSxRQUFRLEdBQUcsc0JBQXNCLENBQUMsSUFBSyxDQUFDLENBQUM7S0FBRTtJQUN6RSxJQUFJLElBQUksS0FBSyxTQUFTLEVBQUU7UUFBRSxJQUFJLEdBQUcsc0JBQXNCLENBQUMsUUFBUyxDQUFDLENBQUM7S0FBRTtJQUVyRSxPQUFPLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQzFCLENBQUM7QUFURCw0REFTQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsV0FBVyxDQUFJLENBQWdCLEVBQUUsR0FBTTtJQUNyRCxPQUFPLENBQUMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO0FBQ25DLENBQUM7QUFGRCxrQ0FFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFwcGxpY2F0aW9uUHJvdG9jb2wgfSBmcm9tIFwiLi9lbnVtc1wiO1xuXG5leHBvcnQgdHlwZSBBdHRyaWJ1dGVzID0ge1trZXk6IHN0cmluZ106IHN0cmluZyB8IHVuZGVmaW5lZH07XG5cbi8qKlxuICogUmVuZGVyIGFuIGF0dHJpYnV0ZSBkaWN0IHRvIGEgbGlzdCBvZiB7IGtleSwgdmFsdWUgfSBwYWlyc1xuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyQXR0cmlidXRlcyhhdHRyaWJ1dGVzOiBBdHRyaWJ1dGVzKSB7XG4gIGNvbnN0IHJldDogYW55W10gPSBbXTtcbiAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoYXR0cmlidXRlcykpIHtcbiAgICBpZiAodmFsdWUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0LnB1c2goeyBrZXksIHZhbHVlIH0pO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmV0O1xufVxuXG4vKipcbiAqIFJldHVybiB0aGUgYXBwcm9wcmlhdGUgZGVmYXVsdCBwb3J0IGZvciBhIGdpdmVuIHByb3RvY29sXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkZWZhdWx0UG9ydEZvclByb3RvY29sKHByb3RvOiBBcHBsaWNhdGlvblByb3RvY29sKTogbnVtYmVyIHtcbiAgc3dpdGNoIChwcm90bykge1xuICAgIGNhc2UgQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQOiByZXR1cm4gODA7XG4gICAgY2FzZSBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTOiByZXR1cm4gNDQzO1xuICAgIGRlZmF1bHQ6XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVucmVjb2duaXplZCBwcm90b2NvbDogJHtwcm90b31gKTtcbiAgfVxufVxuXG4vKipcbiAqIFJldHVybiB0aGUgYXBwcm9wcmlhdGUgZGVmYXVsdCBwcm90b2NvbCBmb3IgYSBnaXZlbiBwb3J0XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkZWZhdWx0UHJvdG9jb2xGb3JQb3J0KHBvcnQ6IG51bWJlcik6IEFwcGxpY2F0aW9uUHJvdG9jb2wge1xuICBzd2l0Y2ggKHBvcnQpIHtcbiAgICBjYXNlIDgwOlxuICAgIGNhc2UgODAwMDpcbiAgICBjYXNlIDgwMDg6XG4gICAgY2FzZSA4MDgwOlxuICAgICAgcmV0dXJuIEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUDtcblxuICAgIGNhc2UgNDQzOlxuICAgIGNhc2UgODQ0MzpcbiAgICAgIHJldHVybiBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTO1xuXG4gICAgZGVmYXVsdDpcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRG9uJ3Qga25vdyBkZWZhdWx0IHByb3RvY29sIGZvciBwb3J0OiAke3BvcnR9OyBwbGVhc2Ugc3VwcGx5IGEgcHJvdG9jb2xgKTtcbiAgfVxufVxuXG4vKipcbiAqIEdpdmVuIGEgcHJvdG9jb2wgYW5kIGEgcG9ydCwgdHJ5IHRvIGd1ZXNzIHRoZSBvdGhlciBvbmUgaWYgaXQncyB1bmRlZmluZWRcbiAqL1xuLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuZXhwb3J0IGZ1bmN0aW9uIGRldGVybWluZVByb3RvY29sQW5kUG9ydChwcm90b2NvbDogQXBwbGljYXRpb25Qcm90b2NvbCB8IHVuZGVmaW5lZCwgcG9ydDogbnVtYmVyIHwgdW5kZWZpbmVkKTogW0FwcGxpY2F0aW9uUHJvdG9jb2wgfCB1bmRlZmluZWQsIG51bWJlciB8IHVuZGVmaW5lZF0ge1xuICBpZiAocHJvdG9jb2wgPT09IHVuZGVmaW5lZCAmJiBwb3J0ID09PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4gW3VuZGVmaW5lZCwgdW5kZWZpbmVkXTtcbiAgfVxuXG4gIGlmIChwcm90b2NvbCA9PT0gdW5kZWZpbmVkKSB7IHByb3RvY29sID0gZGVmYXVsdFByb3RvY29sRm9yUG9ydChwb3J0ISk7IH1cbiAgaWYgKHBvcnQgPT09IHVuZGVmaW5lZCkgeyBwb3J0ID0gZGVmYXVsdFBvcnRGb3JQcm90b2NvbChwcm90b2NvbCEpOyB9XG5cbiAgcmV0dXJuIFtwcm90b2NvbCwgcG9ydF07XG59XG5cbi8qKlxuICogSGVscGVyIGZ1bmN0aW9uIHRvIGRlZmF1bHQgdW5kZWZpbmVkIGlucHV0IHByb3BzXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpZlVuZGVmaW5lZDxUPih4OiBUIHwgdW5kZWZpbmVkLCBkZWY6IFQpIHtcbiAgcmV0dXJuIHggIT09IHVuZGVmaW5lZCA/IHggOiBkZWY7XG59XG4iXX0=