"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
const helpers_1 = require("../helpers");
class ResourceWithLBDependency extends cdk.CfnResource {
    constructor(scope, id, targetGroup) {
        super(scope, id, { type: 'Test::Resource' });
        this.node.addDependency(targetGroup.loadBalancerAttached);
    }
}
module.exports = {
    'Listener guesses protocol from port'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            certificateArns: ['bla'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS'
        }));
        test.done();
    },
    'Listener guesses port from protocol'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            protocol: elbv2.ApplicationProtocol.HTTP,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 80
        }));
        test.done();
    },
    'Listener default to open'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        loadBalancer.addListener('MyListener', {
            port: 80,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    Description: 'Allow from anyone on port 80',
                    CidrIp: "0.0.0.0/0",
                    FromPort: 80,
                    IpProtocol: "tcp",
                    ToPort: 80
                }
            ]
        }));
        test.done();
    },
    'HTTPS listener requires certificate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        const errors = cdk.ConstructNode.validate(stack.node);
        test.deepEqual(errors.map(e => e.message), ['HTTPS Listener needs at least one certificate (call addCertificates)']);
        test.done();
    },
    'HTTPS listener can add certificate after construction'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        listener.addCertificateArns("Arns", ['cert']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Certificates: [
                { CertificateArn: "cert" }
            ],
        }));
        test.done();
    },
    'Can configure targetType on TargetGroups'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', {
            vpc,
            port: 80,
            targetType: elbv2.TargetType.IP
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            TargetType: 'ip'
        }));
        test.done();
    },
    'Can configure name on TargetGroups'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', {
            vpc,
            port: 80,
            targetGroupName: 'foo'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            Name: 'foo'
        }));
        test.done();
    },
    'Can add target groups with and without conditions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        listener.addTargetGroups('Default', {
            targetGroups: [group]
        });
        listener.addTargetGroups('WithPath', {
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [group]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: "TargetGroup3D7CD9B8" },
                    Type: "forward"
                }
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'path-pattern',
                    Values: ['/hello']
                }
            ],
            Actions: [
                {
                    TargetGroupArn: { Ref: "TargetGroup3D7CD9B8" },
                    Type: "forward"
                }
            ],
        }));
        test.done();
    },
    'Can implicitly create target groups with and without conditions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        listener.addTargets('Targets', {
            port: 80,
            targets: [new elbv2.InstanceTarget('i-12345')]
        });
        listener.addTargets('WithPath', {
            priority: 10,
            pathPattern: '/hello',
            port: 80,
            targets: [new elbv2.InstanceTarget('i-5678')]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: "LBListenerTargetsGroup76EF81E8" },
                    Type: "forward"
                }
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            VpcId: { Ref: "Stack8A423254" },
            Port: 80,
            Protocol: "HTTP",
            Targets: [
                { Id: "i-12345" }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    TargetGroupArn: { Ref: "LBListenerWithPathGroupE889F9E5" },
                    Type: "forward"
                }
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            VpcId: { Ref: "Stack8A423254" },
            Port: 80,
            Protocol: "HTTP",
            Targets: [
                { Id: "i-5678" }
            ]
        }));
        test.done();
    },
    'Add certificate to constructed listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 443 });
        // WHEN
        listener.addCertificateArns('Arns', ['cert']);
        listener.addTargets('Targets', { port: 8080, targets: [new elbv2.IpTarget('1.2.3.4')] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Certificates: [
                { CertificateArn: "cert" }
            ],
        }));
        test.done();
    },
    'Add certificate to imported listener'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        const listener2 = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack2, 'Listener', {
            listenerArn: 'listener-arn',
            defaultPort: 443,
            securityGroupId: 'security-group-id'
        });
        // WHEN
        listener2.addCertificateArns('Arns', ['cert']);
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [
                { CertificateArn: "cert" }
            ],
        }));
        test.done();
    },
    'Enable stickiness for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)]
        });
        group.enableCookieStickiness(cdk.Duration.hours(1));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            TargetGroupAttributes: [
                {
                    Key: "stickiness.enabled",
                    Value: "true"
                },
                {
                    Key: "stickiness.type",
                    Value: "lb_cookie"
                },
                {
                    Key: "stickiness.lb_cookie.duration_seconds",
                    Value: "3600"
                }
            ]
        }));
        test.done();
    },
    'Enable health check for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)]
        });
        group.configureHealthCheck({
            unhealthyThresholdCount: 3,
            timeout: cdk.Duration.hours(1),
            interval: cdk.Duration.seconds(30),
            path: '/test',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            UnhealthyThresholdCount: 3,
            HealthCheckIntervalSeconds: 30,
            HealthCheckPath: "/test",
            HealthCheckTimeoutSeconds: 3600,
        }));
        test.done();
    },
    'validation error if invalid health check protocol'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)]
        });
        group.configureHealthCheck({
            unhealthyThresholdCount: 3,
            timeout: cdk.Duration.hours(1),
            interval: cdk.Duration.seconds(30),
            path: '/test',
            protocol: elbv2.Protocol.TCP
        });
        // THEN
        const validationErrors = group.validate();
        test.deepEqual(validationErrors, ["Health check protocol 'TCP' is not supported. Must be one of [HTTP, HTTPS]"]);
        test.done();
    },
    'Can call addTargetGroups on imported listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const listener = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack, 'Listener', {
            listenerArn: 'ieks',
            securityGroupId: 'sg-12345'
        });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        listener.addTargetGroups('Gruuup', {
            priority: 30,
            hostHeader: 'example.com',
            targetGroups: [group]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            ListenerArn: 'ieks',
            Priority: 30,
            Actions: [
                {
                    TargetGroupArn: { Ref: "TargetGroup3D7CD9B8" },
                    Type: "forward"
                }
            ],
        }));
        test.done();
    },
    'Can depend on eventual listener via TargetGroup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', { vpc });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        new ResourceWithLBDependency(stack, 'SomeResource', group);
        loadBalancer.addListener('Listener', {
            port: 80,
            defaultTargetGroups: [group]
        });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                SomeResource: {
                    Type: "Test::Resource",
                    DependsOn: ["LoadBalancerListenerE1A099B9"]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'Exercise metrics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        lb.addListener('SomeListener', {
            port: 80,
            defaultTargetGroups: [group]
        });
        // WHEN
        const metrics = new Array();
        metrics.push(group.metricHttpCodeTarget(elbv2.HttpCodeTarget.TARGET_3XX_COUNT));
        metrics.push(group.metricIpv6RequestCount());
        metrics.push(group.metricUnhealthyHostCount());
        metrics.push(group.metricUnhealthyHostCount());
        metrics.push(group.metricRequestCount());
        metrics.push(group.metricTargetConnectionErrorCount());
        metrics.push(group.metricTargetResponseTime());
        metrics.push(group.metricTargetTLSNegotiationErrorCount());
        for (const metric of metrics) {
            test.equal('AWS/ApplicationELB', metric.namespace);
            const loadBalancerArn = { Ref: "LBSomeListenerCA01F1A0" };
            test.deepEqual(stack.resolve(metric.dimensions), {
                TargetGroup: { 'Fn::GetAtt': ['TargetGroup3D7CD9B8', 'TargetGroupFullName'] },
                LoadBalancer: { 'Fn::Join': ['',
                        [{ 'Fn::Select': [1, { 'Fn::Split': ['/', loadBalancerArn] }] },
                            '/',
                            { 'Fn::Select': [2, { 'Fn::Split': ['/', loadBalancerArn] }] },
                            '/',
                            { 'Fn::Select': [3, { 'Fn::Split': ['/', loadBalancerArn] }] }
                        ]
                    ]
                }
            });
        }
        test.done();
    },
    'Can add dependency on ListenerRule via TargetGroup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', { vpc });
        const group1 = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup1', { vpc, port: 80 });
        const group2 = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup2', { vpc, port: 80 });
        const listener = loadBalancer.addListener('Listener', {
            port: 80,
            defaultTargetGroups: [group1]
        });
        // WHEN
        new ResourceWithLBDependency(stack, 'SomeResource', group2);
        listener.addTargetGroups('SecondGroup', {
            pathPattern: '/bla',
            priority: 10,
            targetGroups: [group2]
        });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                SomeResource: {
                    Type: "Test::Resource",
                    DependsOn: ["LoadBalancerListenerSecondGroupRuleF5FDC196"]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'Can add fixed responses'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc
        });
        const listener = lb.addListener('Listener', {
            port: 80
        });
        // WHEN
        listener.addFixedResponse('Default', {
            contentType: elbv2.ContentType.TEXT_PLAIN,
            messageBody: 'Not Found',
            statusCode: '404'
        });
        listener.addFixedResponse('Hello', {
            priority: 10,
            pathPattern: '/hello',
            statusCode: '503'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    FixedResponseConfig: {
                        ContentType: 'text/plain',
                        MessageBody: 'Not Found',
                        StatusCode: '404'
                    },
                    Type: 'fixed-response'
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    FixedResponseConfig: {
                        StatusCode: '503'
                    },
                    Type: 'fixed-response'
                }
            ]
        }));
        test.done();
    },
    'Can add redirect responses'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc
        });
        const listener = lb.addListener('Listener', {
            port: 80
        });
        // WHEN
        listener.addRedirectResponse('Default', {
            statusCode: 'HTTP_301',
            port: '443',
            protocol: 'HTTPS'
        });
        listener.addRedirectResponse('Hello', {
            priority: 10,
            pathPattern: '/hello',
            path: '/new/#{path}',
            statusCode: 'HTTP_302'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    RedirectConfig: {
                        Port: '443',
                        Protocol: 'HTTPS',
                        StatusCode: 'HTTP_301'
                    },
                    Type: 'redirect'
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    RedirectConfig: {
                        Path: '/new/#{path}',
                        StatusCode: 'HTTP_302'
                    },
                    Type: 'redirect'
                }
            ]
        }));
        test.done();
    },
    'Can configure deregistration_delay for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', {
            vpc,
            port: 80,
            deregistrationDelay: cdk.Duration.seconds(30)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            TargetGroupAttributes: [
                {
                    Key: "deregistration_delay.timeout_seconds",
                    Value: "30"
                }
            ]
        }));
        test.done();
    },
    'Throws with bad fixed responses': {
        'status code'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc
            });
            const listener = lb.addListener('Listener', {
                port: 80
            });
            // THEN
            test.throws(() => listener.addFixedResponse('Default', {
                statusCode: '301'
            }), /`statusCode`/);
            test.done();
        },
        'message body'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc
            });
            const listener = lb.addListener('Listener', {
                port: 80
            });
            // THEN
            test.throws(() => listener.addFixedResponse('Default', {
                messageBody: 'a'.repeat(1025),
                statusCode: '500'
            }), /`messageBody`/);
            test.done();
        }
    },
    'Throws with bad redirect responses': {
        'status code'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc
            });
            const listener = lb.addListener('Listener', {
                port: 80
            });
            // THEN
            test.throws(() => listener.addRedirectResponse('Default', {
                statusCode: '301'
            }), /`statusCode`/);
            test.done();
        },
        'protocol'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc
            });
            const listener = lb.addListener('Listener', {
                port: 80
            });
            // THEN
            test.throws(() => listener.addRedirectResponse('Default', {
                protocol: 'tcp',
                statusCode: 'HTTP_301'
            }), /`protocol`/);
            test.done();
        }
    },
    'Throws when specifying both target groups and fixed reponse'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc
        });
        const listener = lb.addListener('Listener', {
            port: 80
        });
        // THEN
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule', {
            listener,
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 })],
            fixedResponse: {
                statusCode: '500'
            }
        }), /'targetGroups,fixedResponse'.*/);
        test.done();
    },
    'Throws when specifying priority 0'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc
        });
        const listener = lb.addListener('Listener', {
            port: 80
        });
        // THEN
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule', {
            listener,
            priority: 0,
            pathPattern: '/hello',
            fixedResponse: {
                statusCode: '500'
            }
        }), Error, 'Priority must have value greater than or equal to 1');
        test.done();
    },
    'Throws when specifying both target groups and redirect reponse'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc
        });
        const listener = lb.addListener('Listener', {
            port: 80
        });
        // THEN
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule', {
            listener,
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 })],
            redirectResponse: {
                statusCode: 'HTTP_301'
            }
        }), /'targetGroups,redirectResponse'.*/);
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule2', {
            listener,
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [new elbv2.ApplicationTargetGroup(stack, 'TargetGroup2', { vpc, port: 80 })],
            fixedResponse: {
                statusCode: '500'
            },
            redirectResponse: {
                statusCode: 'HTTP_301'
            }
        }), /'targetGroups,fixedResponse,redirectResponse'.*/);
        test.done();
    },
    'Imported listener with imported security group and allowAllOutbound set to false'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const listener = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack, 'Listener', {
            listenerArn: 'listener-arn',
            defaultPort: 443,
            securityGroup: ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'security-group-id', {
                allowAllOutbound: false,
            }),
        });
        // WHEN
        listener.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'security-group-id'
        }));
        test.done();
    },
    'Can pass multiple certificate arns to application listener constructor'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert2' }],
        }));
        test.done();
    },
    'Can use certificate wrapper class'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            certificates: [elbv2.ListenerCertificate.fromArn('cert1'), elbv2.ListenerCertificate.fromArn('cert2')],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert2' }],
        }));
        test.done();
    },
    'Can add additional certificates via addCertficateArns to application listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        listener.addCertificateArns("ListenerCertificateX", ['cert3']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert2' }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert3' }],
        }));
        test.done();
    },
    'Can add multiple path patterns to listener rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        listener.addTargets('Target1', {
            priority: 10,
            pathPatterns: ['/test/path/1', '/test/path/2']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'path-pattern',
                    Values: ['/test/path/1', '/test/path/2']
                }
            ]
        }));
        test.done();
    },
    'Cannot add pathPattern and pathPatterns to listener rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        test.throws(() => listener.addTargets('Target1', {
            priority: 10,
            pathPatterns: ['/test/path/1', '/test/path/2'],
            pathPattern: '/test/path/3'
        }), Error, `At least one of 'hostHeader', 'pathPattern' or 'pathPatterns' is required when defining a load balancing rule.`);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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