"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
const helpers_1 = require("../helpers");
const LB_SECURITY_GROUP = { "Fn::GetAtt": ["LBSecurityGroup8A41EA2B", "GroupId"] };
const IMPORTED_LB_SECURITY_GROUP = { "Fn::ImportValue": "Stack:ExportsOutputFnGetAttLBSecurityGroup8A41EA2BGroupId851EE1F6" };
function expectSameStackSGRules(stack) {
    expectSGRules(stack, LB_SECURITY_GROUP);
}
function expectedImportedSGRules(stack) {
    expectSGRules(stack, IMPORTED_LB_SECURITY_GROUP);
}
function expectSGRules(stack, lbGroup) {
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
        GroupId: lbGroup,
        IpProtocol: "tcp",
        Description: "Load balancer to target",
        DestinationSecurityGroupId: { "Fn::GetAtt": ["TargetSGDB98152D", "GroupId"] },
        FromPort: 8008,
        ToPort: 8008
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: "tcp",
        Description: "Load balancer to target",
        FromPort: 8008,
        GroupId: { "Fn::GetAtt": ["TargetSGDB98152D", "GroupId"] },
        SourceSecurityGroupId: lbGroup,
        ToPort: 8008
    }));
}
class TestFixture {
    constructor(createListener) {
        this.app = new cdk.App();
        this.stack = new cdk.Stack(this.app, 'Stack');
        this.vpc = new ec2.Vpc(this.stack, 'VPC', {
            maxAzs: 2
        });
        this.lb = new elbv2.ApplicationLoadBalancer(this.stack, 'LB', { vpc: this.vpc });
        createListener = createListener === undefined ? true : createListener;
        if (createListener) {
            this._listener = this.lb.addListener('Listener', { port: 80, open: false });
        }
    }
    get listener() {
        if (this._listener === undefined) {
            throw new Error('Did not create a listener');
        }
        return this._listener;
    }
}
module.exports = {
    'security groups are automatically opened bidi for default rule'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const target = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        // WHEN
        fixture.listener.addTargets('TargetGroup', {
            port: 8008,
            targets: [target]
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'security groups are automatically opened bidi for additional rule'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const target1 = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'DefaultTarget', fixture.vpc);
        const target2 = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        // WHEN
        fixture.listener.addTargets('TargetGroup1', {
            port: 80,
            targets: [target1]
        });
        fixture.listener.addTargetGroups('Rule', {
            priority: 10,
            hostHeader: 'example.com',
            targetGroups: [new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup2', {
                    vpc: fixture.vpc,
                    port: 8008,
                    targets: [target2]
                })]
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'adding the same targets twice also works'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const target = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        // WHEN
        const group = new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup', {
            vpc: fixture.vpc,
            port: 8008,
            targets: [target]
        });
        fixture.listener.addTargetGroups('Default', {
            targetGroups: [group]
        });
        fixture.listener.addTargetGroups('WithPath', {
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [group]
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'same result if target is added to group after assigning to listener'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const group = new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup', {
            vpc: fixture.vpc,
            port: 8008
        });
        fixture.listener.addTargetGroups('Default', {
            targetGroups: [group]
        });
        // WHEN
        const target = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        group.addTarget(target);
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'ingress is added to child stack SG instead of parent stack'(test) {
        // GIVEN
        const fixture = new TestFixture(true);
        const parentGroup = new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup', {
            vpc: fixture.vpc,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc)],
        });
        // listener requires at least one rule for ParentStack to create
        fixture.listener.addTargetGroups('Default', { targetGroups: [parentGroup] });
        const childStack = new cdk.Stack(fixture.app, 'childStack');
        // WHEN
        const childGroup = new elbv2.ApplicationTargetGroup(childStack, 'TargetGroup', {
            // We're assuming the 2nd VPC is peered to the 1st, or something.
            vpc: fixture.vpc,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(childStack, 'Target', fixture.vpc)],
        });
        new elbv2.ApplicationListenerRule(childStack, 'ListenerRule', {
            listener: fixture.listener,
            targetGroups: [childGroup],
            priority: 100,
            hostHeader: 'www.foo.com'
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        expectedImportedSGRules(childStack);
        test.done();
    },
    'SG peering works on exported/imported load balancer'(test) {
        // GIVEN
        const fixture = new TestFixture(false);
        const stack2 = new cdk.Stack(fixture.app, 'stack2');
        const vpc2 = new ec2.Vpc(stack2, 'VPC');
        const group = new elbv2.ApplicationTargetGroup(stack2, 'TargetGroup', {
            // We're assuming the 2nd VPC is peered to the 1st, or something.
            vpc: vpc2,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack2, 'Target', vpc2)],
        });
        // WHEN
        const lb2 = elbv2.ApplicationLoadBalancer.fromApplicationLoadBalancerAttributes(stack2, 'LB', {
            loadBalancerArn: fixture.lb.loadBalancerArn,
            securityGroupId: fixture.lb.connections.securityGroups[0].securityGroupId,
            securityGroupAllowsAllOutbound: false,
        });
        const listener2 = lb2.addListener('YetAnotherListener', { port: 80 });
        listener2.addTargetGroups('Default', { targetGroups: [group] });
        // THEN
        expectedImportedSGRules(stack2);
        test.done();
    },
    'SG peering works on exported/imported listener'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const stack2 = new cdk.Stack(fixture.app, 'stack2');
        const vpc2 = new ec2.Vpc(stack2, 'VPC');
        const group = new elbv2.ApplicationTargetGroup(stack2, 'TargetGroup', {
            // We're assuming the 2nd VPC is peered to the 1st, or something.
            vpc: vpc2,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack2, 'Target', vpc2)],
        });
        fixture.listener.addTargets('default', { port: 80 });
        // WHEN
        const listener2 = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack2, 'YetAnotherListener', {
            defaultPort: 8008,
            securityGroupId: fixture.listener.connections.securityGroups[0].securityGroupId,
            listenerArn: fixture.listener.listenerArn,
            securityGroupAllowsAllOutbound: false,
        });
        listener2.addTargetGroups('Default', {
            // Must be a non-default target
            priority: 10,
            hostHeader: 'example.com',
            targetGroups: [group]
        });
        // THEN
        expectedImportedSGRules(stack2);
        test.done();
    },
    'default port peering works on constructed listener'(test) {
        // GIVEN
        const fixture = new TestFixture();
        fixture.listener.addTargets('Default', { port: 8080, targets: [new elbv2.InstanceTarget('i-12345')] });
        // WHEN
        fixture.listener.connections.allowDefaultPortFromAnyIpv4('Open to the world');
        // THEN
        assert_1.expect(fixture.stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "Open to the world",
                    FromPort: 80,
                    IpProtocol: "tcp",
                    ToPort: 80
                }
            ],
        }));
        test.done();
    },
    'default port peering works on imported listener'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const listener2 = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack2, 'YetAnotherListener', {
            listenerArn: 'listener-arn',
            securityGroupId: 'imported-security-group-id',
            defaultPort: 8080
        });
        listener2.connections.allowDefaultPortFromAnyIpv4('Open to the world');
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            CidrIp: "0.0.0.0/0",
            Description: "Open to the world",
            IpProtocol: "tcp",
            FromPort: 8080,
            ToPort: 8080,
            GroupId: 'imported-security-group-id'
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zZWN1cml0eS1ncm91cHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnNlY3VyaXR5LWdyb3Vwcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQXVEO0FBQ3ZELHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFFckMsbUNBQW1DO0FBQ25DLHdDQUF1RDtBQW1QdkQsTUFBTSxpQkFBaUIsR0FBRyxFQUFFLFlBQVksRUFBRSxDQUFFLHlCQUF5QixFQUFFLFNBQVMsQ0FBRSxFQUFFLENBQUM7QUFDckYsTUFBTSwwQkFBMEIsR0FBRyxFQUFFLGlCQUFpQixFQUFFLG1FQUFtRSxFQUFFLENBQUM7QUFFOUgsU0FBUyxzQkFBc0IsQ0FBQyxLQUFnQjtJQUM5QyxhQUFhLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7QUFDMUMsQ0FBQztBQUVELFNBQVMsdUJBQXVCLENBQUMsS0FBZ0I7SUFDL0MsYUFBYSxDQUFDLEtBQUssRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO0FBQ25ELENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FBQyxLQUFnQixFQUFFLE9BQVk7SUFDbkQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLCtCQUErQixFQUFFO1FBQzdELE9BQU8sRUFBRSxPQUFPO1FBQ2hCLFVBQVUsRUFBRSxLQUFLO1FBQ2pCLFdBQVcsRUFBRSx5QkFBeUI7UUFDdEMsMEJBQTBCLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxrQkFBa0IsRUFBRSxTQUFTLENBQUUsRUFBRTtRQUMvRSxRQUFRLEVBQUUsSUFBSTtRQUNkLE1BQU0sRUFBRSxJQUFJO0tBQ2IsQ0FBQyxDQUFDLENBQUM7SUFDSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0NBQWdDLEVBQUU7UUFDOUQsVUFBVSxFQUFFLEtBQUs7UUFDakIsV0FBVyxFQUFFLHlCQUF5QjtRQUN0QyxRQUFRLEVBQUUsSUFBSTtRQUNkLE9BQU8sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGtCQUFrQixFQUFFLFNBQVMsQ0FBRSxFQUFFO1FBQzVELHFCQUFxQixFQUFFLE9BQU87UUFDOUIsTUFBTSxFQUFFLElBQUk7S0FDYixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUM7QUFFRCxNQUFNLFdBQVc7SUFPZixZQUFZLGNBQXdCO1FBQ2xDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDekIsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUN4QyxNQUFNLEVBQUUsQ0FBQztTQUNWLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxFQUFFLEdBQUcsSUFBSSxLQUFLLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFFakYsY0FBYyxHQUFHLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDO1FBQ3RFLElBQUksY0FBYyxFQUFFO1lBQ2xCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztTQUM3RTtJQUNILENBQUM7SUFFRCxJQUFXLFFBQVE7UUFDakIsSUFBSSxJQUFJLENBQUMsU0FBUyxLQUFLLFNBQVMsRUFBRTtZQUFFLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLENBQUMsQ0FBQztTQUFFO1FBQ25GLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQztJQUN4QixDQUFDO0NBQ0Y7QUF4U0QsaUJBQVM7SUFDUCxnRUFBZ0UsQ0FBQyxJQUFVO1FBQ3pFLFFBQVE7UUFDUixNQUFNLE9BQU8sR0FBRyxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBQ2xDLE1BQU0sTUFBTSxHQUFHLElBQUksbUNBQXlCLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRW5GLE9BQU87UUFDUCxPQUFPLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxhQUFhLEVBQUU7WUFDekMsSUFBSSxFQUFFLElBQUk7WUFDVixPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUM7U0FDbEIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV0QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUVBQW1FLENBQUMsSUFBVTtRQUM1RSxRQUFRO1FBQ1IsTUFBTSxPQUFPLEdBQUcsSUFBSSxXQUFXLEVBQUUsQ0FBQztRQUNsQyxNQUFNLE9BQU8sR0FBRyxJQUFJLG1DQUF5QixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMzRixNQUFNLE9BQU8sR0FBRyxJQUFJLG1DQUF5QixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVwRixPQUFPO1FBQ1AsT0FBTyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsY0FBYyxFQUFFO1lBQzFDLElBQUksRUFBRSxFQUFFO1lBQ1IsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDO1NBQ25CLENBQUMsQ0FBQztRQUVILE9BQU8sQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRTtZQUN2QyxRQUFRLEVBQUUsRUFBRTtZQUNaLFVBQVUsRUFBRSxhQUFhO1lBQ3pCLFlBQVksRUFBRSxDQUFDLElBQUksS0FBSyxDQUFDLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO29CQUM3RSxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7b0JBQ2hCLElBQUksRUFBRSxJQUFJO29CQUNWLE9BQU8sRUFBRSxDQUFDLE9BQU8sQ0FBQztpQkFDbkIsQ0FBQyxDQUFDO1NBQ0osQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLHNCQUFzQixDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV0QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMENBQTBDLENBQUMsSUFBVTtRQUNuRCxRQUFRO1FBQ1IsTUFBTSxPQUFPLEdBQUcsSUFBSSxXQUFXLEVBQUUsQ0FBQztRQUNsQyxNQUFNLE1BQU0sR0FBRyxJQUFJLG1DQUF5QixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVuRixPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDM0UsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO1lBQ2hCLElBQUksRUFBRSxJQUFJO1lBQ1YsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDO1NBQ2xCLENBQUMsQ0FBQztRQUVILE9BQU8sQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLFNBQVMsRUFBRTtZQUMxQyxZQUFZLEVBQUUsQ0FBQyxLQUFLLENBQUM7U0FDdEIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsVUFBVSxFQUFFO1lBQzNDLFFBQVEsRUFBRSxFQUFFO1lBQ1osV0FBVyxFQUFFLFFBQVE7WUFDckIsWUFBWSxFQUFFLENBQUMsS0FBSyxDQUFDO1NBQ3RCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFdEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFFQUFxRSxDQUFDLElBQVU7UUFDOUUsUUFBUTtRQUNSLE1BQU0sT0FBTyxHQUFHLElBQUksV0FBVyxFQUFFLENBQUM7UUFDbEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDM0UsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO1lBQ2hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsU0FBUyxFQUFFO1lBQzFDLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQztTQUN0QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxtQ0FBeUIsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbkYsS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV4QixPQUFPO1FBQ1Asc0JBQXNCLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXRDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0REFBNEQsQ0FBQyxJQUFVO1FBQ3JFLFFBQVE7UUFDUixNQUFNLE9BQU8sR0FBRyxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV0QyxNQUFNLFdBQVcsR0FBRyxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtZQUNqRixHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7WUFDaEIsSUFBSSxFQUFFLElBQUk7WUFDVixPQUFPLEVBQUUsQ0FBQyxJQUFJLG1DQUF5QixDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUMvRSxDQUFDLENBQUM7UUFFSCxnRUFBZ0U7UUFDaEUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsU0FBUyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRTdFLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRTVELE9BQU87UUFDUCxNQUFNLFVBQVUsR0FBRyxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxVQUFVLEVBQUUsYUFBYSxFQUFFO1lBQzdFLGlFQUFpRTtZQUNqRSxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7WUFDaEIsSUFBSSxFQUFFLElBQUk7WUFDVixPQUFPLEVBQUUsQ0FBQyxJQUFJLG1DQUF5QixDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQzVFLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLFVBQVUsRUFBRSxjQUFjLEVBQUU7WUFDNUQsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRO1lBQzFCLFlBQVksRUFBRSxDQUFDLFVBQVUsQ0FBQztZQUMxQixRQUFRLEVBQUUsR0FBRztZQUNiLFVBQVUsRUFBRSxhQUFhO1NBQzFCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEMsdUJBQXVCLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFcEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFEQUFxRCxDQUFDLElBQVU7UUFDOUQsUUFBUTtRQUNSLE1BQU0sT0FBTyxHQUFHLElBQUksV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3ZDLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3BELE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRTtZQUNwRSxpRUFBaUU7WUFDakUsR0FBRyxFQUFFLElBQUk7WUFDVCxJQUFJLEVBQUUsSUFBSTtZQUNWLE9BQU8sRUFBRSxDQUFDLElBQUksbUNBQXlCLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNqRSxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixDQUFDLHFDQUFxQyxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUU7WUFDNUYsZUFBZSxFQUFFLE9BQU8sQ0FBQyxFQUFFLENBQUMsZUFBZTtZQUMzQyxlQUFlLEVBQUUsT0FBTyxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWU7WUFDekUsOEJBQThCLEVBQUUsS0FBSztTQUN0QyxDQUFDLENBQUM7UUFDSCxNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsV0FBVyxDQUFDLG9CQUFvQixFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDdEUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFaEUsT0FBTztRQUNQLHVCQUF1QixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRWhDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnREFBZ0QsQ0FBQyxJQUFVO1FBQ3pELFFBQVE7UUFDUixNQUFNLE9BQU8sR0FBRyxJQUFJLFdBQVcsRUFBRSxDQUFDO1FBQ2xDLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3BELE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRTtZQUNwRSxpRUFBaUU7WUFDakUsR0FBRyxFQUFFLElBQUk7WUFDVCxJQUFJLEVBQUUsSUFBSTtZQUNWLE9BQU8sRUFBRSxDQUFDLElBQUksbUNBQXlCLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNqRSxDQUFDLENBQUM7UUFDSCxPQUFPLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVyRCxPQUFPO1FBQ1AsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDLGlDQUFpQyxDQUFDLE1BQU0sRUFBRSxvQkFBb0IsRUFBRTtZQUMxRyxXQUFXLEVBQUUsSUFBSTtZQUNqQixlQUFlLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWU7WUFDL0UsV0FBVyxFQUFFLE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVztZQUN6Qyw4QkFBOEIsRUFBRSxLQUFLO1NBQ3RDLENBQUMsQ0FBQztRQUNILFNBQVMsQ0FBQyxlQUFlLENBQUMsU0FBUyxFQUFFO1lBQ25DLCtCQUErQjtZQUMvQixRQUFRLEVBQUUsRUFBRTtZQUNaLFVBQVUsRUFBRSxhQUFhO1lBQ3pCLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQztTQUN0QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsdUJBQXVCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFaEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG9EQUFvRCxDQUFDLElBQVU7UUFDN0QsUUFBUTtRQUNSLE1BQU0sT0FBTyxHQUFHLElBQUksV0FBVyxFQUFFLENBQUM7UUFDbEMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsQ0FBQyxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFdkcsT0FBTztRQUNQLE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLDJCQUEyQixDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFFOUUsT0FBTztRQUNQLGVBQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDL0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCO29CQUNFLE1BQU0sRUFBRSxXQUFXO29CQUNuQixXQUFXLEVBQUUsbUJBQW1CO29CQUNoQyxRQUFRLEVBQUUsRUFBRTtvQkFDWixVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLEVBQUU7aUJBQ1g7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlEQUFpRCxDQUFDLElBQVU7UUFDMUQsUUFBUTtRQUNSLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRS9CLE9BQU87UUFDUCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUMsaUNBQWlDLENBQUMsTUFBTSxFQUFFLG9CQUFvQixFQUFFO1lBQzFHLFdBQVcsRUFBRSxjQUFjO1lBQzNCLGVBQWUsRUFBRSw0QkFBNEI7WUFDN0MsV0FBVyxFQUFFLElBQUk7U0FDbEIsQ0FBQyxDQUFDO1FBQ0gsU0FBUyxDQUFDLFdBQVcsQ0FBQywyQkFBMkIsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBRXZFLE9BQU87UUFDUCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0NBQWdDLEVBQUU7WUFDL0QsTUFBTSxFQUFFLFdBQVc7WUFDbkIsV0FBVyxFQUFFLG1CQUFtQjtZQUNoQyxVQUFVLEVBQUUsS0FBSztZQUNqQixRQUFRLEVBQUUsSUFBSTtZQUNkLE1BQU0sRUFBRSxJQUFJO1lBQ1osT0FBTyxFQUFFLDRCQUE0QjtTQUN0QyxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCAqIGFzIGVsYnYyIGZyb20gJy4uLy4uL2xpYic7XG5pbXBvcnQgeyBGYWtlU2VsZlJlZ2lzdGVyaW5nVGFyZ2V0IH0gZnJvbSAnLi4vaGVscGVycyc7XG5cbmV4cG9ydCA9IHtcbiAgJ3NlY3VyaXR5IGdyb3VwcyBhcmUgYXV0b21hdGljYWxseSBvcGVuZWQgYmlkaSBmb3IgZGVmYXVsdCBydWxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBmaXh0dXJlID0gbmV3IFRlc3RGaXh0dXJlKCk7XG4gICAgY29uc3QgdGFyZ2V0ID0gbmV3IEZha2VTZWxmUmVnaXN0ZXJpbmdUYXJnZXQoZml4dHVyZS5zdGFjaywgJ1RhcmdldCcsIGZpeHR1cmUudnBjKTtcblxuICAgIC8vIFdIRU5cbiAgICBmaXh0dXJlLmxpc3RlbmVyLmFkZFRhcmdldHMoJ1RhcmdldEdyb3VwJywge1xuICAgICAgcG9ydDogODAwOCxcbiAgICAgIHRhcmdldHM6IFt0YXJnZXRdXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0U2FtZVN0YWNrU0dSdWxlcyhmaXh0dXJlLnN0YWNrKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzZWN1cml0eSBncm91cHMgYXJlIGF1dG9tYXRpY2FsbHkgb3BlbmVkIGJpZGkgZm9yIGFkZGl0aW9uYWwgcnVsZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgZml4dHVyZSA9IG5ldyBUZXN0Rml4dHVyZSgpO1xuICAgIGNvbnN0IHRhcmdldDEgPSBuZXcgRmFrZVNlbGZSZWdpc3RlcmluZ1RhcmdldChmaXh0dXJlLnN0YWNrLCAnRGVmYXVsdFRhcmdldCcsIGZpeHR1cmUudnBjKTtcbiAgICBjb25zdCB0YXJnZXQyID0gbmV3IEZha2VTZWxmUmVnaXN0ZXJpbmdUYXJnZXQoZml4dHVyZS5zdGFjaywgJ1RhcmdldCcsIGZpeHR1cmUudnBjKTtcblxuICAgIC8vIFdIRU5cbiAgICBmaXh0dXJlLmxpc3RlbmVyLmFkZFRhcmdldHMoJ1RhcmdldEdyb3VwMScsIHtcbiAgICAgIHBvcnQ6IDgwLFxuICAgICAgdGFyZ2V0czogW3RhcmdldDFdXG4gICAgfSk7XG5cbiAgICBmaXh0dXJlLmxpc3RlbmVyLmFkZFRhcmdldEdyb3VwcygnUnVsZScsIHtcbiAgICAgIHByaW9yaXR5OiAxMCxcbiAgICAgIGhvc3RIZWFkZXI6ICdleGFtcGxlLmNvbScsXG4gICAgICB0YXJnZXRHcm91cHM6IFtuZXcgZWxidjIuQXBwbGljYXRpb25UYXJnZXRHcm91cChmaXh0dXJlLnN0YWNrLCAnVGFyZ2V0R3JvdXAyJywge1xuICAgICAgICB2cGM6IGZpeHR1cmUudnBjLFxuICAgICAgICBwb3J0OiA4MDA4LFxuICAgICAgICB0YXJnZXRzOiBbdGFyZ2V0Ml1cbiAgICAgIH0pXVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdFNhbWVTdGFja1NHUnVsZXMoZml4dHVyZS5zdGFjayk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkaW5nIHRoZSBzYW1lIHRhcmdldHMgdHdpY2UgYWxzbyB3b3JrcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgZml4dHVyZSA9IG5ldyBUZXN0Rml4dHVyZSgpO1xuICAgIGNvbnN0IHRhcmdldCA9IG5ldyBGYWtlU2VsZlJlZ2lzdGVyaW5nVGFyZ2V0KGZpeHR1cmUuc3RhY2ssICdUYXJnZXQnLCBmaXh0dXJlLnZwYyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgZ3JvdXAgPSBuZXcgZWxidjIuQXBwbGljYXRpb25UYXJnZXRHcm91cChmaXh0dXJlLnN0YWNrLCAnVGFyZ2V0R3JvdXAnLCB7XG4gICAgICB2cGM6IGZpeHR1cmUudnBjLFxuICAgICAgcG9ydDogODAwOCxcbiAgICAgIHRhcmdldHM6IFt0YXJnZXRdXG4gICAgfSk7XG5cbiAgICBmaXh0dXJlLmxpc3RlbmVyLmFkZFRhcmdldEdyb3VwcygnRGVmYXVsdCcsIHtcbiAgICAgIHRhcmdldEdyb3VwczogW2dyb3VwXVxuICAgIH0pO1xuICAgIGZpeHR1cmUubGlzdGVuZXIuYWRkVGFyZ2V0R3JvdXBzKCdXaXRoUGF0aCcsIHtcbiAgICAgIHByaW9yaXR5OiAxMCxcbiAgICAgIHBhdGhQYXR0ZXJuOiAnL2hlbGxvJyxcbiAgICAgIHRhcmdldEdyb3VwczogW2dyb3VwXVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdFNhbWVTdGFja1NHUnVsZXMoZml4dHVyZS5zdGFjayk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc2FtZSByZXN1bHQgaWYgdGFyZ2V0IGlzIGFkZGVkIHRvIGdyb3VwIGFmdGVyIGFzc2lnbmluZyB0byBsaXN0ZW5lcicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgZml4dHVyZSA9IG5ldyBUZXN0Rml4dHVyZSgpO1xuICAgIGNvbnN0IGdyb3VwID0gbmV3IGVsYnYyLkFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAoZml4dHVyZS5zdGFjaywgJ1RhcmdldEdyb3VwJywge1xuICAgICAgdnBjOiBmaXh0dXJlLnZwYyxcbiAgICAgIHBvcnQ6IDgwMDhcbiAgICB9KTtcbiAgICBmaXh0dXJlLmxpc3RlbmVyLmFkZFRhcmdldEdyb3VwcygnRGVmYXVsdCcsIHtcbiAgICAgIHRhcmdldEdyb3VwczogW2dyb3VwXVxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhcmdldCA9IG5ldyBGYWtlU2VsZlJlZ2lzdGVyaW5nVGFyZ2V0KGZpeHR1cmUuc3RhY2ssICdUYXJnZXQnLCBmaXh0dXJlLnZwYyk7XG4gICAgZ3JvdXAuYWRkVGFyZ2V0KHRhcmdldCk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0U2FtZVN0YWNrU0dSdWxlcyhmaXh0dXJlLnN0YWNrKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdpbmdyZXNzIGlzIGFkZGVkIHRvIGNoaWxkIHN0YWNrIFNHIGluc3RlYWQgb2YgcGFyZW50IHN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBmaXh0dXJlID0gbmV3IFRlc3RGaXh0dXJlKHRydWUpO1xuXG4gICAgY29uc3QgcGFyZW50R3JvdXAgPSBuZXcgZWxidjIuQXBwbGljYXRpb25UYXJnZXRHcm91cChmaXh0dXJlLnN0YWNrLCAnVGFyZ2V0R3JvdXAnLCB7XG4gICAgICB2cGM6IGZpeHR1cmUudnBjLFxuICAgICAgcG9ydDogODAwOCxcbiAgICAgIHRhcmdldHM6IFtuZXcgRmFrZVNlbGZSZWdpc3RlcmluZ1RhcmdldChmaXh0dXJlLnN0YWNrLCAnVGFyZ2V0JywgZml4dHVyZS52cGMpXSxcbiAgICB9KTtcblxuICAgIC8vIGxpc3RlbmVyIHJlcXVpcmVzIGF0IGxlYXN0IG9uZSBydWxlIGZvciBQYXJlbnRTdGFjayB0byBjcmVhdGVcbiAgICBmaXh0dXJlLmxpc3RlbmVyLmFkZFRhcmdldEdyb3VwcygnRGVmYXVsdCcsIHsgdGFyZ2V0R3JvdXBzOiBbcGFyZW50R3JvdXBdIH0pO1xuXG4gICAgY29uc3QgY2hpbGRTdGFjayA9IG5ldyBjZGsuU3RhY2soZml4dHVyZS5hcHAsICdjaGlsZFN0YWNrJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgY2hpbGRHcm91cCA9IG5ldyBlbGJ2Mi5BcHBsaWNhdGlvblRhcmdldEdyb3VwKGNoaWxkU3RhY2ssICdUYXJnZXRHcm91cCcsIHtcbiAgICAgIC8vIFdlJ3JlIGFzc3VtaW5nIHRoZSAybmQgVlBDIGlzIHBlZXJlZCB0byB0aGUgMXN0LCBvciBzb21ldGhpbmcuXG4gICAgICB2cGM6IGZpeHR1cmUudnBjLFxuICAgICAgcG9ydDogODAwOCxcbiAgICAgIHRhcmdldHM6IFtuZXcgRmFrZVNlbGZSZWdpc3RlcmluZ1RhcmdldChjaGlsZFN0YWNrLCAnVGFyZ2V0JywgZml4dHVyZS52cGMpXSxcbiAgICB9KTtcblxuICAgIG5ldyBlbGJ2Mi5BcHBsaWNhdGlvbkxpc3RlbmVyUnVsZShjaGlsZFN0YWNrLCAnTGlzdGVuZXJSdWxlJywge1xuICAgICAgbGlzdGVuZXI6IGZpeHR1cmUubGlzdGVuZXIsXG4gICAgICB0YXJnZXRHcm91cHM6IFtjaGlsZEdyb3VwXSxcbiAgICAgIHByaW9yaXR5OiAxMDAsXG4gICAgICBob3N0SGVhZGVyOiAnd3d3LmZvby5jb20nXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0U2FtZVN0YWNrU0dSdWxlcyhmaXh0dXJlLnN0YWNrKTtcbiAgICBleHBlY3RlZEltcG9ydGVkU0dSdWxlcyhjaGlsZFN0YWNrKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTRyBwZWVyaW5nIHdvcmtzIG9uIGV4cG9ydGVkL2ltcG9ydGVkIGxvYWQgYmFsYW5jZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGZpeHR1cmUgPSBuZXcgVGVzdEZpeHR1cmUoZmFsc2UpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBjZGsuU3RhY2soZml4dHVyZS5hcHAsICdzdGFjazInKTtcbiAgICBjb25zdCB2cGMyID0gbmV3IGVjMi5WcGMoc3RhY2syLCAnVlBDJyk7XG4gICAgY29uc3QgZ3JvdXAgPSBuZXcgZWxidjIuQXBwbGljYXRpb25UYXJnZXRHcm91cChzdGFjazIsICdUYXJnZXRHcm91cCcsIHtcbiAgICAgIC8vIFdlJ3JlIGFzc3VtaW5nIHRoZSAybmQgVlBDIGlzIHBlZXJlZCB0byB0aGUgMXN0LCBvciBzb21ldGhpbmcuXG4gICAgICB2cGM6IHZwYzIsXG4gICAgICBwb3J0OiA4MDA4LFxuICAgICAgdGFyZ2V0czogW25ldyBGYWtlU2VsZlJlZ2lzdGVyaW5nVGFyZ2V0KHN0YWNrMiwgJ1RhcmdldCcsIHZwYzIpXSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBsYjIgPSBlbGJ2Mi5BcHBsaWNhdGlvbkxvYWRCYWxhbmNlci5mcm9tQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJBdHRyaWJ1dGVzKHN0YWNrMiwgJ0xCJywge1xuICAgICAgbG9hZEJhbGFuY2VyQXJuOiBmaXh0dXJlLmxiLmxvYWRCYWxhbmNlckFybixcbiAgICAgIHNlY3VyaXR5R3JvdXBJZDogZml4dHVyZS5sYi5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXS5zZWN1cml0eUdyb3VwSWQsXG4gICAgICBzZWN1cml0eUdyb3VwQWxsb3dzQWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgIH0pO1xuICAgIGNvbnN0IGxpc3RlbmVyMiA9IGxiMi5hZGRMaXN0ZW5lcignWWV0QW5vdGhlckxpc3RlbmVyJywgeyBwb3J0OiA4MCB9KTtcbiAgICBsaXN0ZW5lcjIuYWRkVGFyZ2V0R3JvdXBzKCdEZWZhdWx0JywgeyB0YXJnZXRHcm91cHM6IFtncm91cF0gfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0ZWRJbXBvcnRlZFNHUnVsZXMoc3RhY2syKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTRyBwZWVyaW5nIHdvcmtzIG9uIGV4cG9ydGVkL2ltcG9ydGVkIGxpc3RlbmVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBmaXh0dXJlID0gbmV3IFRlc3RGaXh0dXJlKCk7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IGNkay5TdGFjayhmaXh0dXJlLmFwcCwgJ3N0YWNrMicpO1xuICAgIGNvbnN0IHZwYzIgPSBuZXcgZWMyLlZwYyhzdGFjazIsICdWUEMnKTtcbiAgICBjb25zdCBncm91cCA9IG5ldyBlbGJ2Mi5BcHBsaWNhdGlvblRhcmdldEdyb3VwKHN0YWNrMiwgJ1RhcmdldEdyb3VwJywge1xuICAgICAgLy8gV2UncmUgYXNzdW1pbmcgdGhlIDJuZCBWUEMgaXMgcGVlcmVkIHRvIHRoZSAxc3QsIG9yIHNvbWV0aGluZy5cbiAgICAgIHZwYzogdnBjMixcbiAgICAgIHBvcnQ6IDgwMDgsXG4gICAgICB0YXJnZXRzOiBbbmV3IEZha2VTZWxmUmVnaXN0ZXJpbmdUYXJnZXQoc3RhY2syLCAnVGFyZ2V0JywgdnBjMildLFxuICAgIH0pO1xuICAgIGZpeHR1cmUubGlzdGVuZXIuYWRkVGFyZ2V0cygnZGVmYXVsdCcsIHsgcG9ydDogODAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgbGlzdGVuZXIyID0gZWxidjIuQXBwbGljYXRpb25MaXN0ZW5lci5mcm9tQXBwbGljYXRpb25MaXN0ZW5lckF0dHJpYnV0ZXMoc3RhY2syLCAnWWV0QW5vdGhlckxpc3RlbmVyJywge1xuICAgICAgZGVmYXVsdFBvcnQ6IDgwMDgsXG4gICAgICBzZWN1cml0eUdyb3VwSWQ6IGZpeHR1cmUubGlzdGVuZXIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHNbMF0uc2VjdXJpdHlHcm91cElkLFxuICAgICAgbGlzdGVuZXJBcm46IGZpeHR1cmUubGlzdGVuZXIubGlzdGVuZXJBcm4sXG4gICAgICBzZWN1cml0eUdyb3VwQWxsb3dzQWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgIH0pO1xuICAgIGxpc3RlbmVyMi5hZGRUYXJnZXRHcm91cHMoJ0RlZmF1bHQnLCB7XG4gICAgICAvLyBNdXN0IGJlIGEgbm9uLWRlZmF1bHQgdGFyZ2V0XG4gICAgICBwcmlvcml0eTogMTAsXG4gICAgICBob3N0SGVhZGVyOiAnZXhhbXBsZS5jb20nLFxuICAgICAgdGFyZ2V0R3JvdXBzOiBbZ3JvdXBdXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0ZWRJbXBvcnRlZFNHUnVsZXMoc3RhY2syKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWZhdWx0IHBvcnQgcGVlcmluZyB3b3JrcyBvbiBjb25zdHJ1Y3RlZCBsaXN0ZW5lcicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgZml4dHVyZSA9IG5ldyBUZXN0Rml4dHVyZSgpO1xuICAgIGZpeHR1cmUubGlzdGVuZXIuYWRkVGFyZ2V0cygnRGVmYXVsdCcsIHsgcG9ydDogODA4MCwgdGFyZ2V0czogW25ldyBlbGJ2Mi5JbnN0YW5jZVRhcmdldCgnaS0xMjM0NScpXSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBmaXh0dXJlLmxpc3RlbmVyLmNvbm5lY3Rpb25zLmFsbG93RGVmYXVsdFBvcnRGcm9tQW55SXB2NCgnT3BlbiB0byB0aGUgd29ybGQnKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoZml4dHVyZS5zdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cCcsIHtcbiAgICAgIFNlY3VyaXR5R3JvdXBJbmdyZXNzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBDaWRySXA6IFwiMC4wLjAuMC8wXCIsXG4gICAgICAgICAgRGVzY3JpcHRpb246IFwiT3BlbiB0byB0aGUgd29ybGRcIixcbiAgICAgICAgICBGcm9tUG9ydDogODAsXG4gICAgICAgICAgSXBQcm90b2NvbDogXCJ0Y3BcIixcbiAgICAgICAgICBUb1BvcnQ6IDgwXG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2RlZmF1bHQgcG9ydCBwZWVyaW5nIHdvcmtzIG9uIGltcG9ydGVkIGxpc3RlbmVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjazIgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgbGlzdGVuZXIyID0gZWxidjIuQXBwbGljYXRpb25MaXN0ZW5lci5mcm9tQXBwbGljYXRpb25MaXN0ZW5lckF0dHJpYnV0ZXMoc3RhY2syLCAnWWV0QW5vdGhlckxpc3RlbmVyJywge1xuICAgICAgbGlzdGVuZXJBcm46ICdsaXN0ZW5lci1hcm4nLFxuICAgICAgc2VjdXJpdHlHcm91cElkOiAnaW1wb3J0ZWQtc2VjdXJpdHktZ3JvdXAtaWQnLFxuICAgICAgZGVmYXVsdFBvcnQ6IDgwODBcbiAgICB9KTtcbiAgICBsaXN0ZW5lcjIuY29ubmVjdGlvbnMuYWxsb3dEZWZhdWx0UG9ydEZyb21BbnlJcHY0KCdPcGVuIHRvIHRoZSB3b3JsZCcpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjazIpLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgQ2lkcklwOiBcIjAuMC4wLjAvMFwiLFxuICAgICAgRGVzY3JpcHRpb246IFwiT3BlbiB0byB0aGUgd29ybGRcIixcbiAgICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgICBGcm9tUG9ydDogODA4MCxcbiAgICAgIFRvUG9ydDogODA4MCxcbiAgICAgIEdyb3VwSWQ6ICdpbXBvcnRlZC1zZWN1cml0eS1ncm91cC1pZCdcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG5cbmNvbnN0IExCX1NFQ1VSSVRZX0dST1VQID0geyBcIkZuOjpHZXRBdHRcIjogWyBcIkxCU2VjdXJpdHlHcm91cDhBNDFFQTJCXCIsIFwiR3JvdXBJZFwiIF0gfTtcbmNvbnN0IElNUE9SVEVEX0xCX1NFQ1VSSVRZX0dST1VQID0geyBcIkZuOjpJbXBvcnRWYWx1ZVwiOiBcIlN0YWNrOkV4cG9ydHNPdXRwdXRGbkdldEF0dExCU2VjdXJpdHlHcm91cDhBNDFFQTJCR3JvdXBJZDg1MUVFMUY2XCIgfTtcblxuZnVuY3Rpb24gZXhwZWN0U2FtZVN0YWNrU0dSdWxlcyhzdGFjazogY2RrLlN0YWNrKSB7XG4gIGV4cGVjdFNHUnVsZXMoc3RhY2ssIExCX1NFQ1VSSVRZX0dST1VQKTtcbn1cblxuZnVuY3Rpb24gZXhwZWN0ZWRJbXBvcnRlZFNHUnVsZXMoc3RhY2s6IGNkay5TdGFjaykge1xuICBleHBlY3RTR1J1bGVzKHN0YWNrLCBJTVBPUlRFRF9MQl9TRUNVUklUWV9HUk9VUCk7XG59XG5cbmZ1bmN0aW9uIGV4cGVjdFNHUnVsZXMoc3RhY2s6IGNkay5TdGFjaywgbGJHcm91cDogYW55KSB7XG4gIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEVncmVzcycsIHtcbiAgICBHcm91cElkOiBsYkdyb3VwLFxuICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgRGVzY3JpcHRpb246IFwiTG9hZCBiYWxhbmNlciB0byB0YXJnZXRcIixcbiAgICBEZXN0aW5hdGlvblNlY3VyaXR5R3JvdXBJZDogeyBcIkZuOjpHZXRBdHRcIjogWyBcIlRhcmdldFNHREI5ODE1MkRcIiwgXCJHcm91cElkXCIgXSB9LFxuICAgIEZyb21Qb3J0OiA4MDA4LFxuICAgIFRvUG9ydDogODAwOFxuICB9KSk7XG4gIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgSXBQcm90b2NvbDogXCJ0Y3BcIixcbiAgICBEZXNjcmlwdGlvbjogXCJMb2FkIGJhbGFuY2VyIHRvIHRhcmdldFwiLFxuICAgIEZyb21Qb3J0OiA4MDA4LFxuICAgIEdyb3VwSWQ6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJUYXJnZXRTR0RCOTgxNTJEXCIsIFwiR3JvdXBJZFwiIF0gfSxcbiAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IGxiR3JvdXAsXG4gICAgVG9Qb3J0OiA4MDA4XG4gIH0pKTtcbn1cblxuY2xhc3MgVGVzdEZpeHR1cmUge1xuICBwdWJsaWMgcmVhZG9ubHkgYXBwOiBjZGsuQXBwO1xuICBwdWJsaWMgcmVhZG9ubHkgc3RhY2s6IGNkay5TdGFjaztcbiAgcHVibGljIHJlYWRvbmx5IHZwYzogZWMyLlZwYztcbiAgcHVibGljIHJlYWRvbmx5IGxiOiBlbGJ2Mi5BcHBsaWNhdGlvbkxvYWRCYWxhbmNlcjtcbiAgcHVibGljIHJlYWRvbmx5IF9saXN0ZW5lcjogZWxidjIuQXBwbGljYXRpb25MaXN0ZW5lciB8IHVuZGVmaW5lZDtcblxuICBjb25zdHJ1Y3RvcihjcmVhdGVMaXN0ZW5lcj86IGJvb2xlYW4pIHtcbiAgICB0aGlzLmFwcCA9IG5ldyBjZGsuQXBwKCk7XG4gICAgdGhpcy5zdGFjayA9IG5ldyBjZGsuU3RhY2sodGhpcy5hcHAsICdTdGFjaycpO1xuICAgIHRoaXMudnBjID0gbmV3IGVjMi5WcGModGhpcy5zdGFjaywgJ1ZQQycsIHtcbiAgICAgIG1heEF6czogMlxuICAgIH0pO1xuICAgIHRoaXMubGIgPSBuZXcgZWxidjIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcy5zdGFjaywgJ0xCJywgeyB2cGM6IHRoaXMudnBjIH0pO1xuXG4gICAgY3JlYXRlTGlzdGVuZXIgPSBjcmVhdGVMaXN0ZW5lciA9PT0gdW5kZWZpbmVkID8gdHJ1ZSA6IGNyZWF0ZUxpc3RlbmVyO1xuICAgIGlmIChjcmVhdGVMaXN0ZW5lcikge1xuICAgICAgdGhpcy5fbGlzdGVuZXIgPSB0aGlzLmxiLmFkZExpc3RlbmVyKCdMaXN0ZW5lcicsIHsgcG9ydDogODAsIG9wZW46IGZhbHNlIH0pO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBnZXQgbGlzdGVuZXIoKTogZWxidjIuQXBwbGljYXRpb25MaXN0ZW5lciB7XG4gICAgaWYgKHRoaXMuX2xpc3RlbmVyID09PSB1bmRlZmluZWQpIHsgdGhyb3cgbmV3IEVycm9yKCdEaWQgbm90IGNyZWF0ZSBhIGxpc3RlbmVyJyk7IH1cbiAgICByZXR1cm4gdGhpcy5fbGlzdGVuZXI7XG4gIH1cbn1cbiJdfQ==