"use strict";
const assert_1 = require("@aws-cdk/assert");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
const helpers_1 = require("../helpers");
class ResourceWithLBDependency extends cdk.CfnResource {
    constructor(scope, id, targetGroup) {
        super(scope, id, { type: 'Test::Resource' });
        this.node.addDependency(targetGroup.loadBalancerAttached);
    }
}
module.exports = {
    'Trivial add listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            defaultTargetGroups: [new elbv2.NetworkTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'TCP',
            Port: 443
        }));
        test.done();
    },
    'Can add target groups'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 443 });
        const group = new elbv2.NetworkTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        listener.addTargetGroups('Default', group);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: "TargetGroup3D7CD9B8" },
                    Type: "forward"
                }
            ],
        }));
        test.done();
    },
    'Can implicitly create target groups'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 443 });
        // WHEN
        listener.addTargets('Targets', {
            port: 80,
            targets: [new elbv2.InstanceTarget('i-12345')]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: "LBListenerTargetsGroup76EF81E8" },
                    Type: "forward"
                }
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            VpcId: { Ref: "Stack8A423254" },
            Port: 80,
            Protocol: "TCP",
            Targets: [
                { Id: "i-12345" }
            ]
        }));
        test.done();
    },
    'Enable health check for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 443 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)]
        });
        group.configureHealthCheck({
            interval: cdk.Duration.seconds(30)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 30
        }));
        test.done();
    },
    'Enable taking a dependency on an NLB target group\'s load balancer'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 443 });
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)]
        });
        // WHEN
        new ResourceWithLBDependency(stack, 'MyResource', group);
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                MyResource: {
                    Type: "Test::Resource",
                    DependsOn: [
                        // 2nd dependency is there because of the structure of the construct tree.
                        // It does not harm.
                        "LBListenerGroupGroup79B304FF",
                        "LBListener49E825B4",
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'Trivial add TLS listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const cert = new acm.Certificate(stack, 'Certificate', {
            domainName: 'example.com'
        });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            protocol: elbv2.Protocol.TLS,
            certificates: [elbv2.ListenerCertificate.fromCertificateManager(cert)],
            sslPolicy: elbv2.SslPolicy.TLS12,
            defaultTargetGroups: [new elbv2.NetworkTargetGroup(stack, 'Group', { vpc, port: 80 })]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'TLS',
            Port: 443,
            Certificates: [
                { CertificateArn: { Ref: "Certificate4E7ABB08" } }
            ],
            SslPolicy: "ELBSecurityPolicy-TLS-1-2-2017-01"
        }));
        test.done();
    },
    'Invalid Protocol listener'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        test.throws(() => lb.addListener('Listener', {
            port: 443,
            protocol: elbv2.Protocol.HTTP,
            defaultTargetGroups: [new elbv2.NetworkTargetGroup(stack, 'Group', { vpc, port: 80 })]
        }), /The protocol must be one of TCP, TLS, UDP, TCP_UDP\. Found HTTP/);
        test.done();
    },
    'Invalid Listener Target Healthcheck Interval'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('PublicListener', { port: 80 });
        const targetGroup = listener.addTargets('ECS', {
            port: 80,
            healthCheck: {
                interval: cdk.Duration.seconds(60)
            }
        });
        const validationErrors = targetGroup.validate();
        const intervalError = validationErrors.find((err) => /Health check interval '60' not supported. Must be one of the following values/.test(err));
        test.notEqual(intervalError, undefined, 'Failed to return health check interval validation error');
        test.done();
    },
    'validation error if invalid health check protocol'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('PublicListener', { port: 80 });
        const targetGroup = listener.addTargets('ECS', {
            port: 80,
            healthCheck: {
                interval: cdk.Duration.seconds(60)
            }
        });
        targetGroup.configureHealthCheck({
            interval: cdk.Duration.seconds(30),
            protocol: elbv2.Protocol.UDP
        });
        // THEN
        const validationErrors = targetGroup.validate();
        test.deepEqual(validationErrors, ["Health check protocol 'UDP' is not supported. Must be one of [HTTP, HTTPS, TCP]"]);
        test.done();
    },
    'validation error if invalid path health check protocol'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('PublicListener', { port: 80 });
        const targetGroup = listener.addTargets('ECS', {
            port: 80,
            healthCheck: {
                interval: cdk.Duration.seconds(60)
            }
        });
        targetGroup.configureHealthCheck({
            interval: cdk.Duration.seconds(30),
            protocol: elbv2.Protocol.TCP,
            path: '/'
        });
        // THEN
        const validationErrors = targetGroup.validate();
        test.deepEqual(validationErrors, [
            "'TCP' health checks do not support the path property. Must be one of [HTTP, HTTPS]"
        ]);
        test.done();
    },
    'validation error if invalid timeout health check'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('PublicListener', { port: 80 });
        const targetGroup = listener.addTargets('ECS', {
            port: 80,
            healthCheck: {
                interval: cdk.Duration.seconds(60)
            }
        });
        targetGroup.configureHealthCheck({
            interval: cdk.Duration.seconds(30),
            protocol: elbv2.Protocol.HTTP,
            timeout: cdk.Duration.seconds(10),
        });
        // THEN
        const validationErrors = targetGroup.validate();
        test.deepEqual(validationErrors, [
            "Custom health check timeouts are not supported for Network Load Balancer health checks. Expected 6 seconds for HTTP, got 10"
        ]);
        test.done();
    },
    'Protocol & certs TLS listener'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        test.throws(() => lb.addListener('Listener', {
            port: 443,
            protocol: elbv2.Protocol.TLS,
            defaultTargetGroups: [new elbv2.NetworkTargetGroup(stack, 'Group', { vpc, port: 80 })]
        }), /When the protocol is set to TLS, you must specify certificates/);
        test.done();
    },
    'TLS and certs specified listener'(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        const cert = new acm.Certificate(stack, 'Certificate', {
            domainName: 'example.com'
        });
        test.throws(() => lb.addListener('Listener', {
            port: 443,
            protocol: elbv2.Protocol.TCP,
            certificates: [{ certificateArn: cert.certificateArn }],
            defaultTargetGroups: [new elbv2.NetworkTargetGroup(stack, 'Group', { vpc, port: 80 })]
        }), /Protocol must be TLS when certificates have been specified/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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