"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
module.exports = {
    'Trivial construction: internet facing'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            internetFacing: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internet-facing",
            Subnets: [
                { Ref: "StackPublicSubnet1Subnet0AD81D22" },
                { Ref: "StackPublicSubnet2Subnet3C7D2288" },
            ],
            Type: "network"
        }));
        test.done();
    },
    'Trivial construction: internal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internal",
            Subnets: [
                { Ref: "StackPrivateSubnet1Subnet47AC2BC7" },
                { Ref: "StackPrivateSubnet2SubnetA2F8EDD8" },
            ],
            Type: "network"
        }));
        test.done();
    },
    'Attributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            crossZoneEnabled: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: [
                {
                    Key: "load_balancing.cross_zone.enabled",
                    Value: "true"
                }
            ]
        }));
        test.done();
    },
    'Access logging'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, undefined, { env: { region: 'us-east-1' } });
        const vpc = new ec2.Vpc(stack, 'Stack');
        const bucket = new s3.Bucket(stack, 'AccessLoggingBucket');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.logAccessLogs(bucket);
        // THEN
        // verify that the LB attributes reference the bucket
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: [
                {
                    Key: "access_logs.s3.enabled",
                    Value: "true"
                },
                {
                    Key: "access_logs.s3.bucket",
                    Value: { Ref: "AccessLoggingBucketA6D88F29" }
                }
            ],
        }));
        // verify the bucket policy allows the ALB to put objects in the bucket
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: ["s3:PutObject*", "s3:Abort*"],
                        Effect: 'Allow',
                        Principal: { AWS: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::127311923021:root"]] } },
                        Resource: {
                            "Fn::Join": ["", [{ "Fn::GetAtt": ["AccessLoggingBucketA6D88F29", "Arn"] }, "/AWSLogs/",
                                    { Ref: "AWS::AccountId" }, "/*"]]
                        }
                    }
                ]
            }
        }));
        // verify the ALB depends on the bucket *and* the bucket policy
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            DependsOn: ['AccessLoggingBucketPolicy700D7CC6', 'AccessLoggingBucketA6D88F29']
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'access logging with prefix'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, undefined, { env: { region: 'us-east-1' } });
        const vpc = new ec2.Vpc(stack, 'Stack');
        const bucket = new s3.Bucket(stack, 'AccessLoggingBucket');
        const lb = new elbv2.NetworkLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.logAccessLogs(bucket, 'prefix-of-access-logs');
        // THEN
        // verify that the LB attributes reference the bucket
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: [
                {
                    Key: "access_logs.s3.enabled",
                    Value: "true"
                },
                {
                    Key: "access_logs.s3.bucket",
                    Value: { Ref: "AccessLoggingBucketA6D88F29" }
                },
                {
                    Key: "access_logs.s3.prefix",
                    Value: "prefix-of-access-logs"
                }
            ],
        }));
        // verify the bucket policy allows the ALB to put objects in the bucket
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: ["s3:PutObject*", "s3:Abort*"],
                        Effect: 'Allow',
                        Principal: { AWS: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::127311923021:root"]] } },
                        Resource: {
                            "Fn::Join": ["", [{ "Fn::GetAtt": ["AccessLoggingBucketA6D88F29", "Arn"] }, "/prefix-of-access-logs/AWSLogs/",
                                    { Ref: "AWS::AccountId" }, "/*"]]
                        }
                    }
                ]
            }
        }));
        test.done();
    },
    'loadBalancerName'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'ALB', {
            loadBalancerName: 'myLoadBalancer',
            vpc
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Name: 'myLoadBalancer'
        }));
        test.done();
    },
    'imported network load balancer with no vpc specified throws error when calling addTargets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const nlbArn = "arn:aws:elasticloadbalancing::000000000000::dummyloadbalancer";
        const nlb = elbv2.NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(stack, 'NLB', {
            loadBalancerArn: nlbArn,
        });
        // WHEN
        const listener = nlb.addListener('Listener', { port: 80 });
        test.throws(() => listener.addTargets('targetgroup', { port: 8080 }));
        test.done();
    },
    'imported network load balancer with vpc does not throw error when calling addTargets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const nlbArn = "arn:aws:elasticloadbalancing::000000000000::dummyloadbalancer";
        const nlb = elbv2.NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(stack, 'NLB', {
            loadBalancerArn: nlbArn,
            vpc,
        });
        // WHEN
        const listener = nlb.addListener('Listener', { port: 80 });
        test.doesNotThrow(() => listener.addTargets('targetgroup', { port: 8080 }));
        test.done();
    },
    'Trivial construction: internal with Isolated subnets only'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC', {
            subnetConfiguration: [{
                    cidrMask: 20,
                    name: 'Isolated',
                    subnetType: ec2.SubnetType.ISOLATED,
                }]
        });
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            internetFacing: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internal",
            Subnets: [
                { Ref: "VPCIsolatedSubnet1SubnetEBD00FC6" },
                { Ref: "VPCIsolatedSubnet2Subnet4B1C8CAA" },
            ],
            Type: "network"
        }));
        test.done();
    },
    'Internal with Public, Private, and Isolated subnets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC', {
            subnetConfiguration: [{
                    cidrMask: 24,
                    name: 'Public',
                    subnetType: ec2.SubnetType.PUBLIC,
                }, {
                    cidrMask: 24,
                    name: 'Private',
                    subnetType: ec2.SubnetType.PRIVATE,
                }, {
                    cidrMask: 28,
                    name: 'Isolated',
                    subnetType: ec2.SubnetType.ISOLATED,
                }
            ]
        });
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            internetFacing: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internal",
            Subnets: [
                { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
            ],
            Type: "network"
        }));
        test.done();
    },
    'Internet-facing with Public, Private, and Isolated subnets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC', {
            subnetConfiguration: [{
                    cidrMask: 24,
                    name: 'Public',
                    subnetType: ec2.SubnetType.PUBLIC,
                }, {
                    cidrMask: 24,
                    name: 'Private',
                    subnetType: ec2.SubnetType.PRIVATE,
                }, {
                    cidrMask: 28,
                    name: 'Isolated',
                    subnetType: ec2.SubnetType.ISOLATED,
                }
            ]
        });
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            internetFacing: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internet-facing",
            Subnets: [
                { Ref: "VPCPublicSubnet1SubnetB4246D30" },
                { Ref: "VPCPublicSubnet2Subnet74179F39" },
            ],
            Type: "network"
        }));
        test.done();
    },
    'Internal load balancer supplying public subnets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            internetFacing: false,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internal",
            Subnets: [
                { Ref: "VPCPublicSubnet1SubnetB4246D30" },
                { Ref: "VPCPublicSubnet2Subnet74179F39" },
            ],
            Type: "network"
        }));
        test.done();
    },
    'Internal load balancer supplying isolated subnets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC', {
            subnetConfiguration: [{
                    cidrMask: 24,
                    name: 'Public',
                    subnetType: ec2.SubnetType.PUBLIC,
                }, {
                    cidrMask: 24,
                    name: 'Private',
                    subnetType: ec2.SubnetType.PRIVATE,
                }, {
                    cidrMask: 28,
                    name: 'Isolated',
                    subnetType: ec2.SubnetType.ISOLATED,
                }
            ]
        });
        // WHEN
        new elbv2.NetworkLoadBalancer(stack, 'LB', {
            vpc,
            internetFacing: false,
            vpcSubnets: { subnetType: ec2.SubnetType.ISOLATED }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: "internal",
            Subnets: [
                { Ref: "VPCIsolatedSubnet1SubnetEBD00FC6" },
                { Ref: "VPCIsolatedSubnet2Subnet4B1C8CAA" },
            ],
            Type: "network"
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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