'''
collection of tools that facilitate the testing of pylabs3 based code
'''
import mock
import functools
import os
import sys
import imp
import ConfigParser
import inspect
from itertools import izip
from contextlib import contextmanager


class MockNotFound(Exception):
    pass


class CloudApiSpecNotFound(Exception):
    pass


class CloudApiSpecArgumentMismatch(Exception):
    pass


class CallingDirNotFound(Exception):
    pass


class EnumeratorValueMock(object):
    '''
    Represents a q.enumerators.Type.VALUE object
    '''
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return self.value

    def __eq__(self, other):
        return str(self) == str(other)


class EnumeratorMock(object):
    '''
    Represents a q.enumerators.Type object
    '''
    def __getattr__(self, attr):
        return EnumeratorValueMock(attr)


class ModuleProxy(object):
    '''
    Represents a component in the q.foo.bar chain
    '''
    def __init__(self, path, calls):
        self.path = path
        self._calls = calls

    def _is_valid_q_path(self, full_path, attr):
        if any(x.startswith(full_path) for x in self._calls):
            return True
        if attr == 'enumerators':
            return True
        return False

    def __getattr__(self, attr):
        if self.path == ['q', 'enumerators']:
            return EnumeratorMock()
        else:
            new_path = self.path + [attr]
            full_path = '.'.join(new_path)
            if full_path in self._calls:
                call = self._calls[full_path]
                if call.q_as_first_arg:  # some mocks need the q context
                    # create a new q stub with the calls
                    call = functools.partial(call, ModuleProxy(['q'], self._calls))
                return call
            elif full_path == 'q._init_called':
                return True
            elif self._is_valid_q_path(full_path, attr):
                return ModuleProxy(new_path, self._calls)
            else:
                raise MockNotFound('%s does not exist' % full_path)

    def setup_calls(self, *calls, **kwargs):
        '''
        Sets up the q_mock object

        @param call_path:      Full q.foo.bar path
        @type call_path:       string
        @param custom_mock:    Override the normal mock generation with a custom mock object
        @type custom_mock:     object
        @param q_as_first_arg: Inject the full q root object as the first function argument
        @type q_as_first_arg:  bool
        @param args:           Arguments to instantiated mock object
        @type args:            list
        @param kwargs:         Keyword arguments to the instantiated mock object
        @type kwargs:          dict
        '''
        disable_log_auto_mock = kwargs.get('disable_log_auto_mock', False)
        calls_dict = dict(calls)
        # set up auto logging mock
        if not disable_log_auto_mock and 'q.logger.log' not in calls_dict:
            def _simple_stdout_log(*args, **kwargs):
                print ' '.join(args)
            path, mock_obj = q_call('q.logger.log', side_effect=_simple_stdout_log)
            calls_dict[path] = mock_obj
        self._calls = calls_dict

    def reset(self):
        '''
        Removes all qcall mocks from the object
        '''
        self.calls = {}


def q_call(call_path, *args, **kwargs):
    '''
    Generates a new q function mock

    @param call_path:      Full q.foo.bar path
    @type call_path:       string
    @param custom_mock:    Override the normal mock generation with a custom mock object
    @type custom_mock:     object
    @param q_as_first_arg: Inject the full q root object as the first function argument
    @type q_as_first_arg:  bool
    @param args:           Arguments to instantiated mock object
    @type args:            list
    @param kwargs:         Keyword arguments to the instantiated mock object
    @type kwargs:          dict

    @return:               q_path and mock object
    @rtype:                tuple(string, object)
    '''
    default_mock = mock.Mock(*args, **kwargs)
    call_mock = kwargs.get('custom_mock', default_mock)
    call_mock.q_as_first_arg = kwargs.get('q_as_first_arg', False)
    return call_path, call_mock


# we need a global here because modules are only imported once,
# so only the first instantiation is used during execution
# it would be a little bit better if this took threads {green/task}lets into account
# (see werkzeug Local implementation)
q_implementation = ModuleProxy(['q'], {})


def q_mock(*calls, **kwargs):
    '''
    Instantiates the q mock object with all the mocked calls

    @param calls:                 List of ('q.foo.bar', mock) tuples generated by q_call
    @type calls                   q_call tuples
    @param disable_log_auto_mock: Disable implicit q.logger.log mock
    @type disable_log_auto_mock:  bool

    @return:                      Root q mock object
    @rtype:                       mock.Mock
    '''
    q_implementation.setup_calls(*calls, **kwargs)
    return q_implementation


def locally_execute_rscript(q, agentguid, scriptname, params, executionparams):
    '''
    Mocks the executeActorActionScript behaviour

    Note that this code assumes the rscript dir to be relative to the caller file path.
    It tries to execute the script with path <calling_source_file_dir>/scripts/<scriptname>.rscript
    For more parameter info, consult the executeActorActionScript docstring
    '''
    calling_frame = sys._getframe(1)
    calling_file_path = calling_frame.f_code.co_filename

    script_path = os.path.join(os.path.dirname(calling_file_path),
                               'scripts', scriptname + '.rscript')
    script_globals = {
        'params': params,
        'q': q
    }
    execfile(script_path, script_globals)
    return params


def get_calling_test_dir_path():
    '''
    Return the dir that contains the calling function code

    @return:               Directory path that contains source code of caller
    '''
    stack = inspect.stack()
    for _, path, _, _, _, _ in stack:
        basename = os.path.basename(path)
        if os.path.isfile(path) and basename.startswith('test_') and basename.endswith('.py'):
            return os.path.dirname(path)
    raise CallingDirNotFound('could not find calling test dir')


def get_qpackage_base_path():
    '''
    Returns the parent directory of the base testing directory

    @return:               Directory below the base testing directory
    '''
    calling_dir_path = get_calling_test_dir_path()
    descend_path = calling_dir_path
    descend_count = 0
    test_dir_found = False

    # descend tree until we find a test package dir
    while descend_path:
        # check if dir is still a package
        if not os.path.isfile(os.path.join(descend_path, '__init__.py')):
            raise ImportError('path %s is not a package dir' % descend_path)
        descend_path, _, current_dir = descend_path.rpartition(os.path.sep)
        descend_count += 1
        if current_dir == 'test':
            test_dir_found = True
            break
    if not test_dir_found:
        raise ImportError('can\'t find test root dir')

    base_dir_components = [calling_dir_path] + ['..'] * descend_count
    base_dir = os.path.abspath(os.path.join(*base_dir_components))
    return base_dir


def import_qpackage_src(relative_path, module_name):
    '''
    Imports a qpackage source file

    The script finds the base test directory that contains all test subdirs
    and and imports qpackage sources relative to this test folder.
    The function returns the imported module object

    @param relative_path:      Relative path from <qpackage_name>/qpackage4/<platform> to the source file
    @type relative_path:       string
    @param module_name:        Name of the module/source file to import
    @type module_name:         string
    @return:                   Loaded module
    @rtype:                    module object
    '''

    base_dir = get_qpackage_base_path()
    test_config = load_test_config()
    qpackage_source_path = test_config['generic']['qpackage_source_path']
    src_path = os.path.join(base_dir, qpackage_source_path,
                            relative_path.replace('.', os.path.sep),
                            module_name + '.py')

    # clean up previously loaded module
    # imp.load_source doesn't overwrite a module with the same name
    if module_name in sys.modules:
        del sys.modules[module_name]

    return imp.load_source(module_name, src_path)


def _find_test_config_path():
    calling_test_dir = os.path.normpath(get_calling_test_dir_path())
    test_base_dir = os.path.normpath(os.path.join(get_qpackage_base_path(), 'test'))

    current_dir = calling_test_dir
    last_dir = None
    while True:
        test_config_path = os.path.join(current_dir, '.testconfig')
        if not current_dir.startswith(test_base_dir):
            # we reached the base test dir, stop iteration
            return None
        if os.path.isfile(test_config_path):
            return test_config_path

        if current_dir == last_dir:
            # avoid endless repetition
            break
        last_dir = current_dir
        current_dir = os.path.normpath(os.path.join(current_dir, '..'))
    return None


def load_test_config(config_path=None):
    '''
    Load the test configuration from the base test directory

    @param config_path:        optional path to the test configuration
    @return:                   Test configuration as a dict of sections -> (key -> value)
    @rtype:                    dict

    '''
    if not config_path:
        config_path = _find_test_config_path()

    cf = ConfigParser.ConfigParser()
    cf.read([config_path])
    return dict((x, dict(cf.items(x))) for x in cf.sections())


class CloudApiSpec(object):
    def __init__(self, spec_class):
        self.spec_class = spec_class

    @classmethod
    def load(cls, spec, object_name):
        '''
        Load a spec entry defined in the test config

        @param spec:               Name of the section in the testconfig file
        @type spec:                string
        @param object_name:        Object to load from the testspec dir (rootobject, actor,...)
        @type object_name:         string

        @return:                   The cloud api specification object for the rootobject,actor,...
        @rtype:                    CloudApiSpec
        '''
        test_base_dir = os.path.join(get_qpackage_base_path(), 'test')
        test_config = load_test_config()[spec]

        # read config, construct filename, classname by convention
        rel_spec_path = test_config['spec_path']
        prefix = test_config.get('spec_prefix', None)
        spec_filename = (prefix.lower() + '_' + object_name if prefix else object_name) + '.py'

        module_path = os.path.join(test_base_dir, rel_spec_path, spec_filename)
        # load the spec function
        spec_class = cls._load_spec_class(module_path)
        return CloudApiSpec(spec_class)

    def gen_call_params(self, function_name, parameters):
        '''
        Generate the default parameters from the cloud api spec and add custom parameters

        @param function_name:      Name of the function to generate the parameters for
        @type function_name:       string
        @param parameters:         Extra parameters for the function call
        @type parameters:          dict

        @return:                   Combined parameters
        @rtype:                    dictionary

        '''
        all_args, default_params = self._get_args(function_name)
        # merge parameters
        params = {}
        params.update(default_params)
        params.update(parameters)

        # crossreference from spec defined parameters from spec
        all_args = set(x for x in all_args if x != 'self')
        defined_args = set(params.keys())
        if all_args != defined_args:
            message = 'expected args: %s, defined args: %s' % (' '.join(all_args), ' '.join(defined_args))
            raise CloudApiSpecArgumentMismatch(message)
        return params

    def _get_args(self, function_name):
        function = getattr(self.spec_class, function_name)
        # extract the spec and create a args->default value mapping
        spec = inspect.getargspec(function)
        nr_non_default_params = len(spec.args) - len(spec.defaults)
        default_args = dict(izip(spec.args[nr_non_default_params:], spec.defaults))
        keywords = spec.keywords if spec.keywords else []
        return spec.args + keywords, default_args

    @classmethod
    def _load_spec_class(cls, module_path, class_name=None):
                # load the spec file
        module_name = 'temp_api_import_module'
        if not os.path.isfile(module_path):
            raise CloudApiSpecNotFound('could not find cloud api spec in module %s' % module_path)

        if module_name in sys.modules:
            del sys.modules[module_name]

        module = imp.load_source(module_name, module_path)
        # find the classes
        if not class_name:
            classes = [getattr(module, obj) for obj in dir(module) if inspect.isclass(getattr(module, obj))]
            if len(classes) == 1:
                return classes[0]
            else:
                raise CloudApiSpecNotFound('could not find cloud api spec class in module %s' % module_path)
        else:
            return getattr(module, class_name)


def load_cloud_api_spec(spec_entry, object_name):
    '''
    Load a spec entry defined in the test config

    @param spec_entry:         Name of the section in the testconfig file
    @type spec_entry:          string
    @param object_name:        Object to load from the testspec dir (rootobject, actor,...)
    @type object_name:         string

    @return:                   The cloud api specification object for the rootobject,actor,...
    @rtype:                    CloudApiSpec
    '''
    return CloudApiSpec.load(spec_entry, object_name)


def setup_import_paths():
    '''
    Setup import paths defined in the testconfig file
    '''
    test_config = load_test_config()
    paths = test_config['generic'].get('python_paths', [])
    paths = [path.strip() for path in paths.split(',') if path.strip()]
    base_dir = os.path.join(get_qpackage_base_path())

    for path in reversed(paths):
        print 'adding path', path
        sys.path.insert(0, os.path.join(base_dir, path))


class PylabsPatcher(object):
    ''' Main entry point for pylabs mocking machinery '''
    def __init__(self, patch_pymonkey_import=True):
        self._q = None
        self._patch_pymonkey_import = patch_pymonkey_import

    def start(self):
        ''' Generate the basic q mock object and the pymonkey imports

        make sure to always call the stop() method!
        '''
        # set up a basic mock, without any calls
        self._q = q_mock()

        if self._patch_pymonkey_import:
            # set up the module mocks
            pymonkey_module = mock.Mock()
            pymonkey_module.q = self._q
            pymonkey_base = mock.Mock()
            pymonkey_base.q = self._q

            # register mocks as modules
            sys.modules['pymonkey'] = pymonkey_module
            sys.modules['pymonkey.InitBaseCore'] = pymonkey_base
        return self._q

    def stop(self):
        '''Undo pylabs patching '''
        # cleanup
        if self._patch_pymonkey_import:
            del sys.modules['pymonkey']
            del sys.modules['pymonkey.InitBaseCore']
        if self._q:
            self._q.reset()


@contextmanager
def patch_pymonkey_import():
    '''
    Patch pymonkey import so that from pymonkey import q returns the q mock object
    also return the mock object for configuration
    '''
    pylabs_patcher = PylabsPatcher()
    q_ = pylabs_patcher.start()
    # return q object to context manager
    yield q_

    pylabs_patcher.stop()
