"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk_1 = require("@aws-cdk/cdk");
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends cdk_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * The set of S3 bucket IDs that are allowed to send notifications to this queue.
         */
        this.notifyingBuckets = new Set();
    }
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatement(statement);
        }
    }
    /**
     * Allows using SQS queues as destinations for bucket notifications.
     * Use `bucket.onEvent(event, queue)` to subscribe.
     * @param bucketArn The ARN of the notifying bucket.
     * @param bucketId A unique ID for the notifying bucket.
     */
    asBucketNotificationDestination(bucketArn, bucketId) {
        if (!this.notifyingBuckets.has(bucketId)) {
            this.addToResourcePolicy(new iam.PolicyStatement()
                .addServicePrincipal('s3.amazonaws.com')
                .addAction('sqs:SendMessage')
                .addResource(this.queueArn)
                .addCondition('ArnLike', { 'aws:SourceArn': bucketArn }));
            // if this queue is encrypted, we need to allow S3 to read messages since that's how
            // it verifies that the notification destination configuration is valid.
            // by setting allowNoOp to false, we ensure that only custom keys that we can actually
            // control access to can be used here as described in:
            // https://docs.aws.amazon.com/AmazonS3/latest/dev/ways-to-add-notification-config-to-bucket.html
            if (this.encryptionMasterKey) {
                this.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement()
                    .addServicePrincipal('s3.amazonaws.com')
                    .addAction('kms:GenerateDataKey')
                    .addAction('kms:Decrypt')
                    .addAllResources(), /* allowNoOp */ false);
            }
            this.notifyingBuckets.add(bucketId);
        }
        return {
            arn: this.queueArn,
            type: s3n.BucketNotificationDestinationType.Queue,
            dependencies: [this.policy]
        };
    }
    /**
     * Allow using SQS queues as lifecycle hook targets
     */
    asLifecycleHookTarget(lifecycleHook) {
        this.grantSendMessages(lifecycleHook.role);
        return { notificationTargetArn: this.queueArn };
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:ChangeMessageVisibilityBatch
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:DeleteMessageBatch
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        return this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:ChangeMessageVisibilityBatch', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:DeleteMessageBatch', 'sqs:GetQueueAttributes');
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:SendMessageBatch
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        return this.grant(grantee, 'sqs:SendMessage', 'sqs:SendMessageBatch', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
//# sourceMappingURL=data:application/json;base64,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