"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("@aws-cdk/aws-kms");
const cdk_1 = require("@aws-cdk/cdk");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["Unencrypted"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KmsManaged"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["Kms"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: props.queueName,
            ...this.determineFifoProps(props),
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelaySec,
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriodSec,
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTimeSec,
            visibilityTimeout: props.visibilityTimeoutSec,
        });
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueArn = queue.queueArn;
        this.queueName = queue.queueName;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.Unencrypted;
            if (encryption !== QueueEncryption.Kms && props.encryptionMasterKey) {
                encryption = QueueEncryption.Kms; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.Unencrypted) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KmsManaged) {
                const masterKey = kms.EncryptionKey.import(this, 'Key', {
                    keyArn: 'alias/aws/sqs'
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuseSec
                    }
                };
            }
            if (encryption === QueueEncryption.Kms) {
                const masterKey = props.encryptionMasterKey || new kms.EncryptionKey(this, 'Key', {
                    description: `Created by ${this.node.path}`
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuseSec
                    }
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing queue
     */
    static import(scope, id, props) {
        return new ImportedQueue(scope, id, props);
    }
    /**
     * Export a queue
     */
    export() {
        return {
            queueArn: new cdk_1.CfnOutput(this, 'QueueArn', { value: this.queueArn }).makeImportValue().toString(),
            queueUrl: new cdk_1.CfnOutput(this, 'QueueUrl', { value: this.queueUrl }).makeImportValue().toString(),
            keyArn: this.encryptionMasterKey
                ? new cdk_1.CfnOutput(this, 'KeyArn', { value: this.encryptionMasterKey.keyArn }).makeImportValue().toString()
                : undefined
        };
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        if (typeof fifoQueue === 'undefined' && typeof props.queueName === 'string' && props.queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof props.queueName === 'string') {
            if (fifoQueue && !props.queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && props.queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
/**
 * A queue that has been imported
 */
class ImportedQueue extends queue_base_1.QueueBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.autoCreatePolicy = false;
        this.queueArn = props.queueArn;
        this.queueUrl = props.queueUrl;
        this.queueName = this.node.stack.parseArn(props.queueArn).resource;
        if (props.keyArn) {
            this.encryptionMasterKey = kms.EncryptionKey.import(this, 'Key', {
                keyArn: props.keyArn
            });
        }
    }
    /**
     * Export a queue
     */
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,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