"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk_1 = require("@aws-cdk/cdk");
const sqs = require("../lib");
const lib_1 = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new cdk_1.Stack();
    const queue = new lib_1.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        "PolicyDocument": {
            "Statement": [
                {
                    "Action": expectedActions,
                    "Effect": "Allow",
                    "Resource": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            ],
            "Version": "2012-10-17"
        }
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new cdk_1.Stack();
        new sqs.Queue(stack, 'Queue');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue"
                }
            }
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new cdk_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "DLQ581697C4": {
                    "Type": "AWS::SQS::Queue"
                },
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "RedrivePolicy": {
                            "deadLetterTargetArn": {
                                "Fn::GetAtt": [
                                    "DLQ581697C4",
                                    "Arn"
                                ]
                            },
                            "maxReceiveCount": 3
                        }
                    }
                }
            }
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new cdk_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement().addAllResources().addActions('sqs:*').addPrincipal(new iam.ArnPrincipal('arn')));
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyQueueE6CA6235": {
                    "Type": "AWS::SQS::Queue"
                },
                "MyQueuePolicy6BBEDDAC": {
                    "Type": "AWS::SQS::QueuePolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:*",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": "arn"
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Queues": [
                            {
                                "Ref": "MyQueueE6CA6235"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'exporting and importing works'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue');
        // WHEN
        const ref = queue.export();
        const imports = sqs.Queue.import(stack, 'Imported', ref);
        // THEN
        // "import" returns an IQueue bound to `Fn::ImportValue`s.
        test.deepEqual(stack.node.resolve(imports.queueArn), { 'Fn::ImportValue': 'Stack:QueueQueueArn8CF496D5' });
        test.deepEqual(stack.node.resolve(imports.queueUrl), { 'Fn::ImportValue': 'Stack:QueueQueueUrlC30FF916' });
        // the exporting stack has Outputs for QueueARN and QueueURL
        const outputs = assert_1.SynthUtils.toCloudFormation(stack).Outputs;
        // tslint:disable-next-line:max-line-length
        test.deepEqual(outputs.QueueQueueArn8CF496D5, { Value: { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] }, Export: { Name: 'Stack:QueueQueueArn8CF496D5' } });
        test.deepEqual(outputs.QueueQueueUrlC30FF916, { Value: { Ref: 'Queue4A7E3555' }, Export: { Name: 'Stack:QueueQueueUrlC30FF916' } });
        test.done();
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:ChangeMessageVisibilityBatch', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:DeleteMessageBatch', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:SendMessageBatch', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'hello', 'world'), 'hello', 'world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new cdk_1.Stack();
            const queue = lib_1.Queue.import(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url'
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "sqs:PurgeQueue",
                                "sqs:GetQueueAttributes",
                                "sqs:GetQueueUrl"
                            ],
                            "Effect": "Allow",
                            "Resource": "arn:aws:sqs:us-east-1:123456789012:queue1"
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        }
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new cdk_1.Stack();
            const key = new kms.EncryptionKey(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": { "Fn::GetAtt": ["CustomKey1E6D0D07", "Arn"] }
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new cdk_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.Kms });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": {
                    "Fn::GetAtt": [
                        "QueueKey39FCBAE6",
                        "Arn"
                    ]
                }
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new cdk_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KmsManaged });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "Queue4A7E3555": {
                        "Type": "AWS::SQS::Queue",
                        "Properties": {
                            "KmsMasterKeyId": "alias/aws/sqs"
                        }
                    }
                }
            });
            test.done();
        },
        'export should produce outputs the key arn and return import-values for these outputs': {
            'with custom key'(test) {
                const stack = new cdk_1.Stack();
                const customKey = new sqs.Queue(stack, 'QueueWithCustomKey', { encryption: sqs.QueueEncryption.Kms });
                const exportCustom = customKey.export();
                test.deepEqual(stack.node.resolve(exportCustom), {
                    queueArn: { 'Fn::ImportValue': 'Stack:QueueWithCustomKeyQueueArnD326BB9B' },
                    queueUrl: { 'Fn::ImportValue': 'Stack:QueueWithCustomKeyQueueUrlF07DDC70' },
                    keyArn: { 'Fn::ImportValue': 'Stack:QueueWithCustomKeyKeyArn537F6E42' }
                });
                test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
                    "QueueWithCustomKeyQueueArnD326BB9B": {
                        "Value": {
                            "Fn::GetAtt": [
                                "QueueWithCustomKeyB3E22087",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "Stack:QueueWithCustomKeyQueueArnD326BB9B"
                        }
                    },
                    "QueueWithCustomKeyQueueUrlF07DDC70": {
                        "Value": {
                            "Ref": "QueueWithCustomKeyB3E22087"
                        },
                        "Export": {
                            "Name": "Stack:QueueWithCustomKeyQueueUrlF07DDC70"
                        }
                    },
                    "QueueWithCustomKeyKeyArn537F6E42": {
                        "Value": {
                            "Fn::GetAtt": [
                                "QueueWithCustomKeyD80425F1",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "Stack:QueueWithCustomKeyKeyArn537F6E42"
                        }
                    }
                });
                test.done();
            },
            'with managed key'(test) {
                const stack = new cdk_1.Stack();
                const managedKey = new sqs.Queue(stack, 'QueueWithManagedKey', { encryption: sqs.QueueEncryption.KmsManaged });
                const exportManaged = managedKey.export();
                test.deepEqual(stack.node.resolve(exportManaged), {
                    queueArn: { 'Fn::ImportValue': 'Stack:QueueWithManagedKeyQueueArn8798A14E' },
                    queueUrl: { 'Fn::ImportValue': 'Stack:QueueWithManagedKeyQueueUrlD735C981' },
                    keyArn: { 'Fn::ImportValue': 'Stack:QueueWithManagedKeyKeyArn9C42A85D' }
                });
                test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
                    "QueueWithManagedKeyQueueArn8798A14E": {
                        "Value": {
                            "Fn::GetAtt": [
                                "QueueWithManagedKeyE1B747A1",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "Stack:QueueWithManagedKeyQueueArn8798A14E"
                        }
                    },
                    "QueueWithManagedKeyQueueUrlD735C981": {
                        "Value": {
                            "Ref": "QueueWithManagedKeyE1B747A1"
                        },
                        "Export": {
                            "Name": "Stack:QueueWithManagedKeyQueueUrlD735C981"
                        }
                    },
                    "QueueWithManagedKeyKeyArn9C42A85D": {
                        "Value": "alias/aws/sqs",
                        "Export": {
                            "Name": "Stack:QueueWithManagedKeyKeyArn9C42A85D"
                        }
                    }
                });
                test.done();
            }
        }
    },
    'bucket notifications': {
        'queues can be used as destinations'(test) {
            const stack = new cdk_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue');
            const bucket = new s3.Bucket(stack, 'Bucket');
            bucket.onObjectRemoved(queue);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "sqs:SendMessage",
                            "Condition": {
                                "ArnLike": {
                                    "aws:SourceArn": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "Arn"
                                        ]
                                    }
                                }
                            },
                            "Effect": "Allow",
                            "Principal": {
                                "Service": {
                                    "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]]
                                }
                            },
                            "Resource": {
                                "Fn::GetAtt": [
                                    "Queue4A7E3555",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Queues": [
                    {
                        "Ref": "Queue4A7E3555"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
                "BucketName": {
                    "Ref": "Bucket83908E77"
                },
                "NotificationConfiguration": {
                    "QueueConfigurations": [
                        {
                            "Events": [
                                "s3:ObjectRemoved:*"
                            ],
                            "QueueArn": {
                                "Fn::GetAtt": [
                                    "Queue4A7E3555",
                                    "Arn"
                                ]
                            }
                        }
                    ]
                }
            }));
            // make sure the queue policy is added as a dependency to the bucket
            // notifications resource so it will be created first.
            test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Resources.BucketNotifications8F2E257D.DependsOn, ['QueuePolicy25439813']);
            test.done();
        },
        'if the queue is encrypted with a custom kms key, the key resource policy is updated to allow s3 to read messages'(test) {
            const stack = new cdk_1.Stack();
            const bucket = new s3.Bucket(stack, 'Bucket');
            const queue = new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.Kms });
            bucket.onObjectCreated(queue);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                "KeyPolicy": {
                    "Statement": [
                        {
                            "Action": [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                "Ref": "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            "Resource": "*"
                        },
                        {
                            "Action": [
                                "kms:GenerateDataKey",
                                "kms:Decrypt"
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "Service": {
                                    "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]]
                                }
                            },
                            "Resource": "*"
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Description": "Created by Queue"
            }));
            test.done();
        },
        'fails if trying to subscribe to a queue with managed kms encryption'(test) {
            const stack = new cdk_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KmsManaged });
            const bucket = new s3.Bucket(stack, 'Bucket');
            test.throws(() => bucket.onObjectRemoved(queue), 'Unable to add statement to IAM resource policy for KMS key: "alias/aws/sqs"');
            test.done();
        }
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new lib_1.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.node.resolve(topic.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            periodSec: 300,
            statistic: 'Sum'
        });
        test.deepEqual(stack.node.resolve(topic.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            periodSec: 300,
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zcXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnNxcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQW1FO0FBQ25FLHdDQUF5QztBQUN6Qyx3Q0FBeUM7QUFDekMsc0NBQXVDO0FBQ3ZDLHNDQUFxQztBQUVyQyw4QkFBK0I7QUFDL0IsZ0NBQStCO0FBZ2dCL0IsU0FBUyxTQUFTLENBQUMsTUFBcUQsRUFBRSxHQUFHLGVBQXlCO0lBQ3BHLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQzFDLE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFFOUMsTUFBTSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUV6QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7UUFDaEQsZ0JBQWdCLEVBQUU7WUFDaEIsV0FBVyxFQUFFO2dCQUNYO29CQUNFLFFBQVEsRUFBRSxlQUFlO29CQUN6QixRQUFRLEVBQUUsT0FBTztvQkFDakIsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixpQkFBaUI7NEJBQ2pCLEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7YUFDRjtZQUNELFNBQVMsRUFBRSxZQUFZO1NBQ3hCO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDO0FBcGhCRCxpQkFBUztJQUNQLG9CQUFvQixDQUFDLElBQVU7UUFDN0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRTlCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLGVBQWUsRUFBRTtvQkFDakIsTUFBTSxFQUFFLGlCQUFpQjtpQkFDeEI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCwwQkFBMEIsQ0FBQyxJQUFVO1FBQ25DLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLGVBQWUsRUFBRSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsZUFBZSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV2RixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxhQUFhLEVBQUU7b0JBQ2YsTUFBTSxFQUFFLGlCQUFpQjtpQkFDeEI7Z0JBQ0QsZUFBZSxFQUFFO29CQUNqQixNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osZUFBZSxFQUFFOzRCQUNqQixxQkFBcUIsRUFBRTtnQ0FDckIsWUFBWSxFQUFFO29DQUNkLGFBQWE7b0NBQ2IsS0FBSztpQ0FDSjs2QkFDRjs0QkFDRCxpQkFBaUIsRUFBRSxDQUFDO3lCQUNuQjtxQkFDRjtpQkFDQTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtEQUErRCxDQUFDLElBQVU7UUFDeEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzlDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckksZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNwQixXQUFXLEVBQUU7Z0JBQ1gsaUJBQWlCLEVBQUU7b0JBQ25CLE1BQU0sRUFBRSxpQkFBaUI7aUJBQ3hCO2dCQUNELHVCQUF1QixFQUFFO29CQUN6QixNQUFNLEVBQUUsdUJBQXVCO29CQUMvQixZQUFZLEVBQUU7d0JBQ1osZ0JBQWdCLEVBQUU7NEJBQ2xCLFdBQVcsRUFBRTtnQ0FDWDtvQ0FDQSxRQUFRLEVBQUUsT0FBTztvQ0FDakIsUUFBUSxFQUFFLE9BQU87b0NBQ2pCLFdBQVcsRUFBRTt3Q0FDWCxLQUFLLEVBQUUsS0FBSztxQ0FDYjtvQ0FDRCxVQUFVLEVBQUUsR0FBRztpQ0FDZDs2QkFDRjs0QkFDRCxTQUFTLEVBQUUsWUFBWTt5QkFDdEI7d0JBQ0QsUUFBUSxFQUFFOzRCQUNWO2dDQUNFLEtBQUssRUFBRSxpQkFBaUI7NkJBQ3pCO3lCQUNBO3FCQUNGO2lCQUNBO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0JBQStCLENBQUMsSUFBVTtRQUN4QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRTVDLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDM0IsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUV6RCxPQUFPO1FBRVAsMERBQTBEO1FBQzFELElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsNkJBQTZCLEVBQUUsQ0FBQyxDQUFDO1FBQzNHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsNkJBQTZCLEVBQUUsQ0FBQyxDQUFDO1FBRTNHLDREQUE0RDtRQUM1RCxNQUFNLE9BQU8sR0FBRyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUMzRCwyQ0FBMkM7UUFDM0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMscUJBQXFCLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxlQUFlLEVBQUUsS0FBSyxDQUFFLEVBQUUsRUFBRSxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsNkJBQTZCLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDeEosSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMscUJBQXFCLEVBQUUsRUFBRSxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLDZCQUE2QixFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRXBJLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxRQUFRLEVBQUU7UUFDUixzQkFBc0IsQ0FBQyxJQUFVO1lBQy9CLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsRUFDM0Msb0JBQW9CLEVBQ3BCLDZCQUE2QixFQUM3QixrQ0FBa0MsRUFDbEMsaUJBQWlCLEVBQ2pCLG1CQUFtQixFQUNuQix3QkFBd0IsRUFDeEIsd0JBQXdCLENBQ3pCLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsbUJBQW1CLENBQUMsSUFBVTtZQUM1QixTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLEVBQ3hDLGlCQUFpQixFQUNqQixzQkFBc0IsRUFDdEIsd0JBQXdCLEVBQ3hCLGlCQUFpQixDQUNsQixDQUFDO1lBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELFlBQVksQ0FBQyxJQUFVO1lBQ3JCLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQ2pDLGdCQUFnQixFQUNoQix3QkFBd0IsRUFDeEIsaUJBQWlCLENBQ2xCLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNEJBQTRCLENBQUMsSUFBVTtZQUNyQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDLEVBQzlDLE9BQU8sRUFDUCxPQUFPLENBQ1IsQ0FBQztZQUNGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxxQ0FBcUMsQ0FBQyxJQUFVO1lBQzlDLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxLQUFLLEdBQUcsV0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUMxQyxRQUFRLEVBQUUsMkNBQTJDO2dCQUNyRCxRQUFRLEVBQUUsbUJBQW1CO2FBQzlCLENBQUMsQ0FBQztZQUVILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFFekMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUV2QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELGdCQUFnQixFQUFFO29CQUNoQixXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsUUFBUSxFQUFFO2dDQUNSLGdCQUFnQjtnQ0FDaEIsd0JBQXdCO2dDQUN4QixpQkFBaUI7NkJBQ2xCOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixVQUFVLEVBQUUsMkNBQTJDO3lCQUN4RDtxQkFDRjtvQkFDRCxTQUFTLEVBQUUsWUFBWTtpQkFDeEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELGtCQUFrQixFQUFFO1FBQ2xCLG9EQUFvRCxDQUFDLElBQVU7WUFDN0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztZQUUxQixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBQ3RELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsbUJBQW1CLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUUxRSxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLENBQUMsQ0FBQztZQUMxQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7Z0JBQy9DLGdCQUFnQixFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsbUJBQW1CLEVBQUUsS0FBSyxDQUFFLEVBQUU7YUFDbkUsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsbUZBQW1GLENBQUMsSUFBVTtZQUM1RixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1lBRTFCLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUV2RSxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQztZQUNoRCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7Z0JBQy9DLGdCQUFnQixFQUFFO29CQUNsQixZQUFZLEVBQUU7d0JBQ1osa0JBQWtCO3dCQUNsQixLQUFLO3FCQUNOO2lCQUNBO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQseUNBQXlDLENBQUMsSUFBVTtZQUNsRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1lBRTFCLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztZQUM5RSxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNwQixXQUFXLEVBQUU7b0JBQ2IsZUFBZSxFQUFFO3dCQUNmLE1BQU0sRUFBRSxpQkFBaUI7d0JBQ3pCLFlBQVksRUFBRTs0QkFDZCxnQkFBZ0IsRUFBRSxlQUFlO3lCQUNoQztxQkFDRjtpQkFDQTthQUNGLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxzRkFBc0YsRUFBRTtZQUV0RixpQkFBaUIsQ0FBQyxJQUFVO2dCQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO2dCQUUxQixNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLG9CQUFvQixFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztnQkFFdEcsTUFBTSxZQUFZLEdBQUcsU0FBUyxDQUFDLE1BQU0sRUFBRSxDQUFDO2dCQUV4QyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxFQUFFO29CQUMvQyxRQUFRLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSwwQ0FBMEMsRUFBRTtvQkFDM0UsUUFBUSxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsMENBQTBDLEVBQUU7b0JBQzNFLE1BQU0sRUFBRSxFQUFFLGlCQUFpQixFQUFFLHdDQUF3QyxFQUFFO2lCQUN4RSxDQUFDLENBQUM7Z0JBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sRUFBRTtvQkFDekQsb0NBQW9DLEVBQUU7d0JBQ3RDLE9BQU8sRUFBRTs0QkFDUCxZQUFZLEVBQUU7Z0NBQ2QsNEJBQTRCO2dDQUM1QixLQUFLOzZCQUNKO3lCQUNGO3dCQUNELFFBQVEsRUFBRTs0QkFDUixNQUFNLEVBQUUsMENBQTBDO3lCQUNuRDtxQkFDQTtvQkFDRCxvQ0FBb0MsRUFBRTt3QkFDdEMsT0FBTyxFQUFFOzRCQUNQLEtBQUssRUFBRSw0QkFBNEI7eUJBQ3BDO3dCQUNELFFBQVEsRUFBRTs0QkFDUixNQUFNLEVBQUUsMENBQTBDO3lCQUNuRDtxQkFDQTtvQkFDRCxrQ0FBa0MsRUFBRTt3QkFDcEMsT0FBTyxFQUFFOzRCQUNQLFlBQVksRUFBRTtnQ0FDZCw0QkFBNEI7Z0NBQzVCLEtBQUs7NkJBQ0o7eUJBQ0Y7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLE1BQU0sRUFBRSx3Q0FBd0M7eUJBQ2pEO3FCQUNBO2lCQUNGLENBQUMsQ0FBQztnQkFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1lBRUQsa0JBQWtCLENBQUMsSUFBVTtnQkFDM0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztnQkFFMUIsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7Z0JBRS9HLE1BQU0sYUFBYSxHQUFHLFVBQVUsQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFFMUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRTtvQkFDaEQsUUFBUSxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsMkNBQTJDLEVBQUU7b0JBQzVFLFFBQVEsRUFBRSxFQUFFLGlCQUFpQixFQUFFLDJDQUEyQyxFQUFFO29CQUM1RSxNQUFNLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSx5Q0FBeUMsRUFBRTtpQkFDekUsQ0FBQyxDQUFDO2dCQUVILElBQUksQ0FBQyxTQUFTLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLEVBQUU7b0JBQ3pELHFDQUFxQyxFQUFFO3dCQUN2QyxPQUFPLEVBQUU7NEJBQ1AsWUFBWSxFQUFFO2dDQUNkLDZCQUE2QjtnQ0FDN0IsS0FBSzs2QkFDSjt5QkFDRjt3QkFDRCxRQUFRLEVBQUU7NEJBQ1IsTUFBTSxFQUFFLDJDQUEyQzt5QkFDcEQ7cUJBQ0E7b0JBQ0QscUNBQXFDLEVBQUU7d0JBQ3ZDLE9BQU8sRUFBRTs0QkFDUCxLQUFLLEVBQUUsNkJBQTZCO3lCQUNyQzt3QkFDRCxRQUFRLEVBQUU7NEJBQ1IsTUFBTSxFQUFFLDJDQUEyQzt5QkFDcEQ7cUJBQ0E7b0JBQ0QsbUNBQW1DLEVBQUU7d0JBQ3JDLE9BQU8sRUFBRSxlQUFlO3dCQUN4QixRQUFRLEVBQUU7NEJBQ1IsTUFBTSxFQUFFLHlDQUF5Qzt5QkFDbEQ7cUJBQ0E7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7U0FFRjtLQUVGO0lBRUQsc0JBQXNCLEVBQUU7UUFFdEIsb0NBQW9DLENBQUMsSUFBVTtZQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDNUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztZQUU5QyxNQUFNLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRTlCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtnQkFDckQsZ0JBQWdCLEVBQUU7b0JBQ2xCLFdBQVcsRUFBRTt3QkFDWDs0QkFDQSxRQUFRLEVBQUUsaUJBQWlCOzRCQUMzQixXQUFXLEVBQUU7Z0NBQ1gsU0FBUyxFQUFFO29DQUNYLGVBQWUsRUFBRTt3Q0FDZixZQUFZLEVBQUU7NENBQ2QsZ0JBQWdCOzRDQUNoQixLQUFLO3lDQUNKO3FDQUNGO2lDQUNBOzZCQUNGOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixXQUFXLEVBQUU7Z0NBQ1gsU0FBUyxFQUFFO29DQUNULFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7aUNBQ3JEOzZCQUNGOzRCQUNELFVBQVUsRUFBRTtnQ0FDVixZQUFZLEVBQUU7b0NBQ2QsZUFBZTtvQ0FDZixLQUFLO2lDQUNKOzZCQUNGO3lCQUNBO3FCQUNGO29CQUNELFNBQVMsRUFBRSxZQUFZO2lCQUN0QjtnQkFDRCxRQUFRLEVBQUU7b0JBQ1Y7d0JBQ0UsS0FBSyxFQUFFLGVBQWU7cUJBQ3ZCO2lCQUNBO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsK0JBQStCLEVBQUU7Z0JBQzdELFlBQVksRUFBRTtvQkFDZCxLQUFLLEVBQUUsZ0JBQWdCO2lCQUN0QjtnQkFDRCwyQkFBMkIsRUFBRTtvQkFDN0IscUJBQXFCLEVBQUU7d0JBQ3JCOzRCQUNBLFFBQVEsRUFBRTtnQ0FDUixvQkFBb0I7NkJBQ3JCOzRCQUNELFVBQVUsRUFBRTtnQ0FDVixZQUFZLEVBQUU7b0NBQ2QsZUFBZTtvQ0FDZixLQUFLO2lDQUNKOzZCQUNGO3lCQUNBO3FCQUNGO2lCQUNBO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixvRUFBb0U7WUFDcEUsc0RBQXNEO1lBQ3RELElBQUksQ0FBQyxTQUFTLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUMsMkJBQTJCLENBQUMsU0FBUyxFQUFFLENBQUUscUJBQXFCLENBQUUsQ0FBQyxDQUFDO1lBRTlILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxrSEFBa0gsQ0FBQyxJQUFVO1lBRTNILE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztZQUM5QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFFckYsTUFBTSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUU5QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZUFBZSxFQUFFO2dCQUM3QyxXQUFXLEVBQUU7b0JBQ2IsV0FBVyxFQUFFO3dCQUNYOzRCQUNBLFFBQVEsRUFBRTtnQ0FDUixhQUFhO2dDQUNiLGVBQWU7Z0NBQ2YsYUFBYTtnQ0FDYixXQUFXO2dDQUNYLFVBQVU7Z0NBQ1YsYUFBYTtnQ0FDYixhQUFhO2dDQUNiLGNBQWM7Z0NBQ2QsVUFBVTtnQ0FDVixhQUFhO2dDQUNiLHlCQUF5QjtnQ0FDekIsdUJBQXVCOzZCQUN4Qjs0QkFDRCxRQUFRLEVBQUUsT0FBTzs0QkFDakIsV0FBVyxFQUFFO2dDQUNYLEtBQUssRUFBRTtvQ0FDUCxVQUFVLEVBQUU7d0NBQ1YsRUFBRTt3Q0FDRjs0Q0FDQSxNQUFNOzRDQUNOO2dEQUNFLEtBQUssRUFBRSxnQkFBZ0I7NkNBQ3hCOzRDQUNELFFBQVE7NENBQ1I7Z0RBQ0UsS0FBSyxFQUFFLGdCQUFnQjs2Q0FDeEI7NENBQ0QsT0FBTzt5Q0FDTjtxQ0FDRjtpQ0FDQTs2QkFDRjs0QkFDRCxVQUFVLEVBQUUsR0FBRzt5QkFDZDt3QkFDRDs0QkFDQSxRQUFRLEVBQUU7Z0NBQ1IscUJBQXFCO2dDQUNyQixhQUFhOzZCQUNkOzRCQUNELFFBQVEsRUFBRSxPQUFPOzRCQUNqQixXQUFXLEVBQUU7Z0NBQ1gsU0FBUyxFQUFFO29DQUNULFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7aUNBQ3JEOzZCQUNGOzRCQUNELFVBQVUsRUFBRSxHQUFHO3lCQUNkO3FCQUNGO29CQUNELFNBQVMsRUFBRSxZQUFZO2lCQUN0QjtnQkFDRCxhQUFhLEVBQUUsa0JBQWtCO2FBQ2xDLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHFFQUFxRSxDQUFDLElBQVU7WUFDOUUsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7WUFDNUYsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztZQUM5QyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLEVBQUUsNkVBQTZFLENBQUMsQ0FBQztZQUNoSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBRUY7SUFFRCxjQUFjLENBQUMsSUFBVTtRQUN2QixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFeEMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLDBCQUEwQixFQUFFLENBQUMsRUFBRTtZQUNyRSxVQUFVLEVBQUUsRUFBQyxTQUFTLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxlQUFlLEVBQUUsV0FBVyxDQUFFLEVBQUUsRUFBQztZQUMzRSxTQUFTLEVBQUUsU0FBUztZQUNwQixVQUFVLEVBQUUsc0JBQXNCO1lBQ2xDLFNBQVMsRUFBRSxHQUFHO1lBQ2QsU0FBUyxFQUFFLEtBQUs7U0FDakIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMscUJBQXFCLEVBQUUsQ0FBQyxFQUFFO1lBQ2hFLFVBQVUsRUFBRSxFQUFDLFNBQVMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGVBQWUsRUFBRSxXQUFXLENBQUUsRUFBRSxFQUFDO1lBQzNFLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLFVBQVUsRUFBRSxpQkFBaUI7WUFDN0IsU0FBUyxFQUFFLEdBQUc7WUFDZCxTQUFTLEVBQUUsU0FBUztTQUNyQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlLCBTeW50aFV0aWxzIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCBpYW0gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtaWFtJyk7XG5pbXBvcnQga21zID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWttcycpO1xuaW1wb3J0IHMzID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLXMzJyk7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2Nkayc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHNxcyA9IHJlcXVpcmUoJy4uL2xpYicpO1xuaW1wb3J0IHsgUXVldWUgfSBmcm9tICcuLi9saWInO1xuXG4vLyB0c2xpbnQ6ZGlzYWJsZTpvYmplY3QtbGl0ZXJhbC1rZXktcXVvdGVzXG5cbmV4cG9ydCA9IHtcbiAgJ2RlZmF1bHQgcHJvcGVydGllcycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJyk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIlF1ZXVlNEE3RTM1NTVcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlNRUzo6UXVldWVcIlxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ3dpdGggYSBkZWFkIGxldHRlciBxdWV1ZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgZGxxID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ0RMUScpO1xuICAgIG5ldyBzcXMuUXVldWUoc3RhY2ssICdRdWV1ZScsIHsgZGVhZExldHRlclF1ZXVlOiB7IHF1ZXVlOiBkbHEsIG1heFJlY2VpdmVDb3VudDogMyB9IH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHtcbiAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgICAgXCJETFE1ODE2OTdDNFwiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6U1FTOjpRdWV1ZVwiXG4gICAgICAgIH0sXG4gICAgICAgIFwiUXVldWU0QTdFMzU1NVwiOiB7XG4gICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6U1FTOjpRdWV1ZVwiLFxuICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiUmVkcml2ZVBvbGljeVwiOiB7XG4gICAgICAgICAgXCJkZWFkTGV0dGVyVGFyZ2V0QXJuXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIkRMUTU4MTY5N0M0XCIsXG4gICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIm1heFJlY2VpdmVDb3VudFwiOiAzXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhZGRUb1BvbGljeSB3aWxsIGF1dG9tYXRpY2FsbHkgY3JlYXRlIGEgcG9saWN5IGZvciB0aGlzIHF1ZXVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdNeVF1ZXVlJyk7XG4gICAgcXVldWUuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCgpLmFkZEFsbFJlc291cmNlcygpLmFkZEFjdGlvbnMoJ3NxczoqJykuYWRkUHJpbmNpcGFsKG5ldyBpYW0uQXJuUHJpbmNpcGFsKCdhcm4nKSkpO1xuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7XG4gICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlRdWV1ZUU2Q0E2MjM1XCI6IHtcbiAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTUVM6OlF1ZXVlXCJcbiAgICAgICAgfSxcbiAgICAgICAgXCJNeVF1ZXVlUG9saWN5NkJCRUREQUNcIjoge1xuICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlNRUzo6UXVldWVQb2xpY3lcIixcbiAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIlBvbGljeURvY3VtZW50XCI6IHtcbiAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICBcIkFjdGlvblwiOiBcInNxczoqXCIsXG4gICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICBcIlByaW5jaXBhbFwiOiB7XG4gICAgICAgICAgICAgIFwiQVdTXCI6IFwiYXJuXCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIlJlc291cmNlXCI6IFwiKlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUXVldWVzXCI6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIlJlZlwiOiBcIk15UXVldWVFNkNBNjIzNVwiXG4gICAgICAgICAgfVxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2V4cG9ydGluZyBhbmQgaW1wb3J0aW5nIHdvcmtzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcmVmID0gcXVldWUuZXhwb3J0KCk7XG4gICAgY29uc3QgaW1wb3J0cyA9IHNxcy5RdWV1ZS5pbXBvcnQoc3RhY2ssICdJbXBvcnRlZCcsIHJlZik7XG5cbiAgICAvLyBUSEVOXG5cbiAgICAvLyBcImltcG9ydFwiIHJldHVybnMgYW4gSVF1ZXVlIGJvdW5kIHRvIGBGbjo6SW1wb3J0VmFsdWVgcy5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5ub2RlLnJlc29sdmUoaW1wb3J0cy5xdWV1ZUFybiksIHsgJ0ZuOjpJbXBvcnRWYWx1ZSc6ICdTdGFjazpRdWV1ZVF1ZXVlQXJuOENGNDk2RDUnIH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLm5vZGUucmVzb2x2ZShpbXBvcnRzLnF1ZXVlVXJsKSwgeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrOlF1ZXVlUXVldWVVcmxDMzBGRjkxNicgfSk7XG5cbiAgICAvLyB0aGUgZXhwb3J0aW5nIHN0YWNrIGhhcyBPdXRwdXRzIGZvciBRdWV1ZUFSTiBhbmQgUXVldWVVUkxcbiAgICBjb25zdCBvdXRwdXRzID0gU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKS5PdXRwdXRzO1xuICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICB0ZXN0LmRlZXBFcXVhbChvdXRwdXRzLlF1ZXVlUXVldWVBcm44Q0Y0OTZENSwgeyBWYWx1ZTogeyAnRm46OkdldEF0dCc6IFsgJ1F1ZXVlNEE3RTM1NTUnLCAnQXJuJyBdIH0sIEV4cG9ydDogeyBOYW1lOiAnU3RhY2s6UXVldWVRdWV1ZUFybjhDRjQ5NkQ1JyB9IH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKG91dHB1dHMuUXVldWVRdWV1ZVVybEMzMEZGOTE2LCB7IFZhbHVlOiB7IFJlZjogJ1F1ZXVlNEE3RTM1NTUnIH0sIEV4cG9ydDogeyBOYW1lOiAnU3RhY2s6UXVldWVRdWV1ZVVybEMzMEZGOTE2JyB9IH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2dyYW50cyc6IHtcbiAgICAnZ3JhbnRDb25zdW1lTWVzc2FnZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCgocSwgcCkgPT4gcS5ncmFudENvbnN1bWVNZXNzYWdlcyhwKSxcbiAgICAgICAgJ3NxczpSZWNlaXZlTWVzc2FnZScsXG4gICAgICAgICdzcXM6Q2hhbmdlTWVzc2FnZVZpc2liaWxpdHknLFxuICAgICAgICAnc3FzOkNoYW5nZU1lc3NhZ2VWaXNpYmlsaXR5QmF0Y2gnLFxuICAgICAgICAnc3FzOkdldFF1ZXVlVXJsJyxcbiAgICAgICAgJ3NxczpEZWxldGVNZXNzYWdlJyxcbiAgICAgICAgJ3NxczpEZWxldGVNZXNzYWdlQmF0Y2gnLFxuICAgICAgICAnc3FzOkdldFF1ZXVlQXR0cmlidXRlcycsXG4gICAgICApO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdncmFudFNlbmRNZXNzYWdlcycodGVzdDogVGVzdCkge1xuICAgICAgdGVzdEdyYW50KChxLCBwKSA9PiBxLmdyYW50U2VuZE1lc3NhZ2VzKHApLFxuICAgICAgICAnc3FzOlNlbmRNZXNzYWdlJyxcbiAgICAgICAgJ3NxczpTZW5kTWVzc2FnZUJhdGNoJyxcbiAgICAgICAgJ3NxczpHZXRRdWV1ZUF0dHJpYnV0ZXMnLFxuICAgICAgICAnc3FzOkdldFF1ZXVlVXJsJyxcbiAgICAgICk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2dyYW50UHVyZ2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCgocSwgcCkgPT4gcS5ncmFudFB1cmdlKHApLFxuICAgICAgICAnc3FzOlB1cmdlUXVldWUnLFxuICAgICAgICAnc3FzOkdldFF1ZXVlQXR0cmlidXRlcycsXG4gICAgICAgICdzcXM6R2V0UXVldWVVcmwnLFxuICAgICAgKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZ3JhbnQoKSBpcyBnZW5lcmFsIHB1cnBvc2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3RHcmFudCgocSwgcCkgPT4gcS5ncmFudChwLCAnaGVsbG8nLCAnd29ybGQnKSxcbiAgICAgICAgJ2hlbGxvJyxcbiAgICAgICAgJ3dvcmxkJ1xuICAgICAgKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZ3JhbnRzIGFsc28gd29yayBvbiBpbXBvcnRlZCBxdWV1ZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCBxdWV1ZSA9IFF1ZXVlLmltcG9ydChzdGFjaywgJ0ltcG9ydCcsIHtcbiAgICAgICAgcXVldWVBcm46ICdhcm46YXdzOnNxczp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOnF1ZXVlMScsXG4gICAgICAgIHF1ZXVlVXJsOiAnaHR0cHM6Ly9xdWV1ZS11cmwnXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgdXNlciA9IG5ldyBpYW0uVXNlcihzdGFjaywgJ1VzZXInKTtcblxuICAgICAgcXVldWUuZ3JhbnRQdXJnZSh1c2VyKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogW1xuICAgICAgICAgICAgICAgIFwic3FzOlB1cmdlUXVldWVcIixcbiAgICAgICAgICAgICAgICBcInNxczpHZXRRdWV1ZUF0dHJpYnV0ZXNcIixcbiAgICAgICAgICAgICAgICBcInNxczpHZXRRdWV1ZVVybFwiXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiBcImFybjphd3M6c3FzOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6cXVldWUxXCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9XG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH1cbiAgfSxcblxuICAncXVldWUgZW5jcnlwdGlvbic6IHtcbiAgICAnZW5jcnlwdGlvbk1hc3RlcktleSBjYW4gYmUgc2V0IHRvIGEgY3VzdG9tIEtNUyBrZXknKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGtleSA9IG5ldyBrbXMuRW5jcnlwdGlvbktleShzdGFjaywgJ0N1c3RvbUtleScpO1xuICAgICAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnUXVldWUnLCB7IGVuY3J5cHRpb25NYXN0ZXJLZXk6IGtleSB9KTtcblxuICAgICAgdGVzdC5zYW1lKHF1ZXVlLmVuY3J5cHRpb25NYXN0ZXJLZXksIGtleSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpTUVM6OlF1ZXVlJywge1xuICAgICAgICBcIkttc01hc3RlcktleUlkXCI6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJDdXN0b21LZXkxRTZEMEQwN1wiLCBcIkFyblwiIF0gfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2Ega21zIGtleSB3aWxsIGJlIGFsbG9jYXRlZCBpZiBlbmNyeXB0aW9uID0ga21zIGJ1dCBhIG1hc3RlciBrZXkgaXMgbm90IHNwZWNpZmllZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywgeyBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLkttcyB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6S01TOjpLZXknKSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpTUVM6OlF1ZXVlJywge1xuICAgICAgICBcIkttc01hc3RlcktleUlkXCI6IHtcbiAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICBcIlF1ZXVlS2V5MzlGQ0JBRTZcIixcbiAgICAgICAgICBcIkFyblwiXG4gICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2l0IGlzIHBvc3NpYmxlIHRvIHVzZSBhIG1hbmFnZWQga21zIGtleScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgbmV3IHNxcy5RdWV1ZShzdGFjaywgJ1F1ZXVlJywgeyBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLkttc01hbmFnZWQgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiUXVldWU0QTdFMzU1NVwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTUVM6OlF1ZXVlXCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIkttc01hc3RlcktleUlkXCI6IFwiYWxpYXMvYXdzL3Nxc1wiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdleHBvcnQgc2hvdWxkIHByb2R1Y2Ugb3V0cHV0cyB0aGUga2V5IGFybiBhbmQgcmV0dXJuIGltcG9ydC12YWx1ZXMgZm9yIHRoZXNlIG91dHB1dHMnOiB7XG5cbiAgICAgICd3aXRoIGN1c3RvbSBrZXknKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgICBjb25zdCBjdXN0b21LZXkgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnUXVldWVXaXRoQ3VzdG9tS2V5JywgeyBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLkttcyB9KTtcblxuICAgICAgICBjb25zdCBleHBvcnRDdXN0b20gPSBjdXN0b21LZXkuZXhwb3J0KCk7XG5cbiAgICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2subm9kZS5yZXNvbHZlKGV4cG9ydEN1c3RvbSksIHtcbiAgICAgICAgICBxdWV1ZUFybjogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrOlF1ZXVlV2l0aEN1c3RvbUtleVF1ZXVlQXJuRDMyNkJCOUInIH0sXG4gICAgICAgICAgcXVldWVVcmw6IHsgJ0ZuOjpJbXBvcnRWYWx1ZSc6ICdTdGFjazpRdWV1ZVdpdGhDdXN0b21LZXlRdWV1ZVVybEYwN0REQzcwJyB9LFxuICAgICAgICAgIGtleUFybjogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrOlF1ZXVlV2l0aEN1c3RvbUtleUtleUFybjUzN0Y2RTQyJyB9XG4gICAgICAgIH0pO1xuXG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykuT3V0cHV0cywge1xuICAgICAgICAgIFwiUXVldWVXaXRoQ3VzdG9tS2V5UXVldWVBcm5EMzI2QkI5QlwiOiB7XG4gICAgICAgICAgXCJWYWx1ZVwiOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJRdWV1ZVdpdGhDdXN0b21LZXlCM0UyMjA4N1wiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6UXVldWVXaXRoQ3VzdG9tS2V5UXVldWVBcm5EMzI2QkI5QlwiXG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJRdWV1ZVdpdGhDdXN0b21LZXlRdWV1ZVVybEYwN0REQzcwXCI6IHtcbiAgICAgICAgICBcIlZhbHVlXCI6IHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiUXVldWVXaXRoQ3VzdG9tS2V5QjNFMjIwODdcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6UXVldWVXaXRoQ3VzdG9tS2V5UXVldWVVcmxGMDdEREM3MFwiXG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJRdWV1ZVdpdGhDdXN0b21LZXlLZXlBcm41MzdGNkU0MlwiOiB7XG4gICAgICAgICAgXCJWYWx1ZVwiOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJRdWV1ZVdpdGhDdXN0b21LZXlEODA0MjVGMVwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6UXVldWVXaXRoQ3VzdG9tS2V5S2V5QXJuNTM3RjZFNDJcIlxuICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICd3aXRoIG1hbmFnZWQga2V5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgICAgY29uc3QgbWFuYWdlZEtleSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdRdWV1ZVdpdGhNYW5hZ2VkS2V5JywgeyBlbmNyeXB0aW9uOiBzcXMuUXVldWVFbmNyeXB0aW9uLkttc01hbmFnZWQgfSk7XG5cbiAgICAgICAgY29uc3QgZXhwb3J0TWFuYWdlZCA9IG1hbmFnZWRLZXkuZXhwb3J0KCk7XG5cbiAgICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2subm9kZS5yZXNvbHZlKGV4cG9ydE1hbmFnZWQpLCB7XG4gICAgICAgICAgcXVldWVBcm46IHsgJ0ZuOjpJbXBvcnRWYWx1ZSc6ICdTdGFjazpRdWV1ZVdpdGhNYW5hZ2VkS2V5UXVldWVBcm44Nzk4QTE0RScgfSxcbiAgICAgICAgICBxdWV1ZVVybDogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrOlF1ZXVlV2l0aE1hbmFnZWRLZXlRdWV1ZVVybEQ3MzVDOTgxJyB9LFxuICAgICAgICAgIGtleUFybjogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrOlF1ZXVlV2l0aE1hbmFnZWRLZXlLZXlBcm45QzQyQTg1RCcgfVxuICAgICAgICB9KTtcblxuICAgICAgICB0ZXN0LmRlZXBFcXVhbChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLk91dHB1dHMsIHtcbiAgICAgICAgICBcIlF1ZXVlV2l0aE1hbmFnZWRLZXlRdWV1ZUFybjg3OThBMTRFXCI6IHtcbiAgICAgICAgICBcIlZhbHVlXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlF1ZXVlV2l0aE1hbmFnZWRLZXlFMUI3NDdBMVwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6UXVldWVXaXRoTWFuYWdlZEtleVF1ZXVlQXJuODc5OEExNEVcIlxuICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUXVldWVXaXRoTWFuYWdlZEtleVF1ZXVlVXJsRDczNUM5ODFcIjoge1xuICAgICAgICAgIFwiVmFsdWVcIjoge1xuICAgICAgICAgICAgXCJSZWZcIjogXCJRdWV1ZVdpdGhNYW5hZ2VkS2V5RTFCNzQ3QTFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJFeHBvcnRcIjoge1xuICAgICAgICAgICAgXCJOYW1lXCI6IFwiU3RhY2s6UXVldWVXaXRoTWFuYWdlZEtleVF1ZXVlVXJsRDczNUM5ODFcIlxuICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUXVldWVXaXRoTWFuYWdlZEtleUtleUFybjlDNDJBODVEXCI6IHtcbiAgICAgICAgICBcIlZhbHVlXCI6IFwiYWxpYXMvYXdzL3Nxc1wiLFxuICAgICAgICAgIFwiRXhwb3J0XCI6IHtcbiAgICAgICAgICAgIFwiTmFtZVwiOiBcIlN0YWNrOlF1ZXVlV2l0aE1hbmFnZWRLZXlLZXlBcm45QzQyQTg1RFwiXG4gICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfSk7XG5cbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9XG5cbiAgICB9XG5cbiAgfSxcblxuICAnYnVja2V0IG5vdGlmaWNhdGlvbnMnOiB7XG5cbiAgICAncXVldWVzIGNhbiBiZSB1c2VkIGFzIGRlc3RpbmF0aW9ucycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgICAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnUXVldWUnKTtcbiAgICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdCdWNrZXQnKTtcblxuICAgICAgYnVja2V0Lm9uT2JqZWN0UmVtb3ZlZChxdWV1ZSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OlNRUzo6UXVldWVQb2xpY3knLCB7XG4gICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAge1xuICAgICAgICAgIFwiQWN0aW9uXCI6IFwic3FzOlNlbmRNZXNzYWdlXCIsXG4gICAgICAgICAgXCJDb25kaXRpb25cIjoge1xuICAgICAgICAgICAgXCJBcm5MaWtlXCI6IHtcbiAgICAgICAgICAgIFwiYXdzOlNvdXJjZUFyblwiOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiQnVja2V0ODM5MDhFNzdcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgXCJQcmluY2lwYWxcIjoge1xuICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wiczMuXCIsIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfV1dXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIlJlc291cmNlXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlF1ZXVlNEE3RTM1NTVcIixcbiAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgICAgXCJRdWV1ZXNcIjogW1xuICAgICAgICB7XG4gICAgICAgICAgXCJSZWZcIjogXCJRdWV1ZTRBN0UzNTU1XCJcbiAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9KSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdDdXN0b206OlMzQnVja2V0Tm90aWZpY2F0aW9ucycsIHtcbiAgICAgICAgXCJCdWNrZXROYW1lXCI6IHtcbiAgICAgICAgXCJSZWZcIjogXCJCdWNrZXQ4MzkwOEU3N1wiXG4gICAgICAgIH0sXG4gICAgICAgIFwiTm90aWZpY2F0aW9uQ29uZmlndXJhdGlvblwiOiB7XG4gICAgICAgIFwiUXVldWVDb25maWd1cmF0aW9uc1wiOiBbXG4gICAgICAgICAge1xuICAgICAgICAgIFwiRXZlbnRzXCI6IFtcbiAgICAgICAgICAgIFwiczM6T2JqZWN0UmVtb3ZlZDoqXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIFwiUXVldWVBcm5cIjoge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiUXVldWU0QTdFMzU1NVwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICAvLyBtYWtlIHN1cmUgdGhlIHF1ZXVlIHBvbGljeSBpcyBhZGRlZCBhcyBhIGRlcGVuZGVuY3kgdG8gdGhlIGJ1Y2tldFxuICAgICAgLy8gbm90aWZpY2F0aW9ucyByZXNvdXJjZSBzbyBpdCB3aWxsIGJlIGNyZWF0ZWQgZmlyc3QuXG4gICAgICB0ZXN0LmRlZXBFcXVhbChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLlJlc291cmNlcy5CdWNrZXROb3RpZmljYXRpb25zOEYyRTI1N0QuRGVwZW5kc09uLCBbICdRdWV1ZVBvbGljeTI1NDM5ODEzJyBdKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpZiB0aGUgcXVldWUgaXMgZW5jcnlwdGVkIHdpdGggYSBjdXN0b20ga21zIGtleSwgdGhlIGtleSByZXNvdXJjZSBwb2xpY3kgaXMgdXBkYXRlZCB0byBhbGxvdyBzMyB0byByZWFkIG1lc3NhZ2VzJyh0ZXN0OiBUZXN0KSB7XG5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnQnVja2V0Jyk7XG4gICAgICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdRdWV1ZScsIHsgZW5jcnlwdGlvbjogc3FzLlF1ZXVlRW5jcnlwdGlvbi5LbXMgfSk7XG5cbiAgICAgIGJ1Y2tldC5vbk9iamVjdENyZWF0ZWQocXVldWUpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpLTVM6OktleScsIHtcbiAgICAgICAgXCJLZXlQb2xpY3lcIjoge1xuICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAge1xuICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgIFwia21zOkNyZWF0ZSpcIixcbiAgICAgICAgICAgIFwia21zOkRlc2NyaWJlKlwiLFxuICAgICAgICAgICAgXCJrbXM6RW5hYmxlKlwiLFxuICAgICAgICAgICAgXCJrbXM6TGlzdCpcIixcbiAgICAgICAgICAgIFwia21zOlB1dCpcIixcbiAgICAgICAgICAgIFwia21zOlVwZGF0ZSpcIixcbiAgICAgICAgICAgIFwia21zOlJldm9rZSpcIixcbiAgICAgICAgICAgIFwia21zOkRpc2FibGUqXCIsXG4gICAgICAgICAgICBcImttczpHZXQqXCIsXG4gICAgICAgICAgICBcImttczpEZWxldGUqXCIsXG4gICAgICAgICAgICBcImttczpTY2hlZHVsZUtleURlbGV0aW9uXCIsXG4gICAgICAgICAgICBcImttczpDYW5jZWxLZXlEZWxldGlvblwiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgXCJQcmluY2lwYWxcIjoge1xuICAgICAgICAgICAgXCJBV1NcIjoge1xuICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXCI6aWFtOjpcIixcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBcIjpyb290XCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJSZXNvdXJjZVwiOiBcIipcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgICAgIFwia21zOkdlbmVyYXRlRGF0YUtleVwiLFxuICAgICAgICAgICAgXCJrbXM6RGVjcnlwdFwiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgXCJQcmluY2lwYWxcIjoge1xuICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wiczMuXCIsIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfV1dXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIlJlc291cmNlXCI6IFwiKlwiXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgICAgXCJEZXNjcmlwdGlvblwiOiBcIkNyZWF0ZWQgYnkgUXVldWVcIlxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2ZhaWxzIGlmIHRyeWluZyB0byBzdWJzY3JpYmUgdG8gYSBxdWV1ZSB3aXRoIG1hbmFnZWQga21zIGVuY3J5cHRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCBxdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdRdWV1ZScsIHsgZW5jcnlwdGlvbjogc3FzLlF1ZXVlRW5jcnlwdGlvbi5LbXNNYW5hZ2VkIH0pO1xuICAgICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0J1Y2tldCcpO1xuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gYnVja2V0Lm9uT2JqZWN0UmVtb3ZlZChxdWV1ZSksICdVbmFibGUgdG8gYWRkIHN0YXRlbWVudCB0byBJQU0gcmVzb3VyY2UgcG9saWN5IGZvciBLTVMga2V5OiBcImFsaWFzL2F3cy9zcXNcIicpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfVxuXG4gIH0sXG5cbiAgJ3Rlc3QgbWV0cmljcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB0b3BpYyA9IG5ldyBRdWV1ZShzdGFjaywgJ1F1ZXVlJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2subm9kZS5yZXNvbHZlKHRvcGljLm1ldHJpY051bWJlck9mTWVzc2FnZXNTZW50KCkpLCB7XG4gICAgICBkaW1lbnNpb25zOiB7UXVldWVOYW1lOiB7ICdGbjo6R2V0QXR0JzogWyAnUXVldWU0QTdFMzU1NScsICdRdWV1ZU5hbWUnIF0gfX0sXG4gICAgICBuYW1lc3BhY2U6ICdBV1MvU1FTJyxcbiAgICAgIG1ldHJpY05hbWU6ICdOdW1iZXJPZk1lc3NhZ2VzU2VudCcsXG4gICAgICBwZXJpb2RTZWM6IDMwMCxcbiAgICAgIHN0YXRpc3RpYzogJ1N1bSdcbiAgICB9KTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLm5vZGUucmVzb2x2ZSh0b3BpYy5tZXRyaWNTZW50TWVzc2FnZVNpemUoKSksIHtcbiAgICAgIGRpbWVuc2lvbnM6IHtRdWV1ZU5hbWU6IHsgJ0ZuOjpHZXRBdHQnOiBbICdRdWV1ZTRBN0UzNTU1JywgJ1F1ZXVlTmFtZScgXSB9fSxcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9TUVMnLFxuICAgICAgbWV0cmljTmFtZTogJ1NlbnRNZXNzYWdlU2l6ZScsXG4gICAgICBwZXJpb2RTZWM6IDMwMCxcbiAgICAgIHN0YXRpc3RpYzogJ0F2ZXJhZ2UnXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfVxufTtcblxuZnVuY3Rpb24gdGVzdEdyYW50KGFjdGlvbjogKHE6IFF1ZXVlLCBwcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsKSA9PiB2b2lkLCAuLi5leHBlY3RlZEFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHF1ZXVlID0gbmV3IFF1ZXVlKHN0YWNrLCAnTXlRdWV1ZScpO1xuICBjb25zdCBwcmluY2lwYWwgPSBuZXcgaWFtLlVzZXIoc3RhY2ssICdVc2VyJyk7XG5cbiAgYWN0aW9uKHF1ZXVlLCBwcmluY2lwYWwpO1xuXG4gIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICB7XG4gICAgICAgICAgXCJBY3Rpb25cIjogZXhwZWN0ZWRBY3Rpb25zLFxuICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICBcIlJlc291cmNlXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiTXlRdWV1ZUU2Q0E2MjM1XCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICB9XG4gIH0pKTtcbn1cbiJdfQ==