import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import iam = require('@aws-cdk/aws-iam');
import { Construct, IResource, RemovalPolicy, Resource } from '@aws-cdk/cdk';
import { LogStream } from './log-stream';
import { MetricFilter } from './metric-filter';
import { IFilterPattern } from './pattern';
import { ILogSubscriptionDestination, SubscriptionFilter } from './subscription-filter';
export interface ILogGroup extends IResource {
    /**
     * The ARN of this log group
     * @attribute
     */
    readonly logGroupArn: string;
    /**
     * The name of this log group
     * @attribute
     */
    readonly logGroupName: string;
    /**
     * Create a new Log Stream for this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the LogStream
     */
    addStream(id: string, props?: StreamOptions): LogStream;
    /**
     * Create a new Subscription Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the SubscriptionFilter
     */
    addSubscriptionFilter(id: string, props: SubscriptionFilterOptions): SubscriptionFilter;
    /**
     * Create a new Metric Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the MetricFilter
     */
    addMetricFilter(id: string, props: MetricFilterOptions): MetricFilter;
    /**
     * Extract a metric from structured log events in the LogGroup
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield')
     * @param metricNamespace Namespace to emit the metric under
     * @param metricName Name to emit the metric under
     * @returns A Metric object representing the extracted metric
     */
    extractMetric(jsonField: string, metricNamespace: string, metricName: string): cloudwatch.Metric;
    /**
     * Give permissions to write to create and write to streams in this log group
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Give the indicated permissions on this log group and all streams
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * An CloudWatch Log Group
 */
declare abstract class LogGroupBase extends Resource implements ILogGroup {
    /**
     * The ARN of this log group
     */
    abstract readonly logGroupArn: string;
    /**
     * The name of this log group
     */
    abstract readonly logGroupName: string;
    /**
     * Create a new Log Stream for this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the LogStream
     */
    addStream(id: string, props?: StreamOptions): LogStream;
    /**
     * Create a new Subscription Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the SubscriptionFilter
     */
    addSubscriptionFilter(id: string, props: SubscriptionFilterOptions): SubscriptionFilter;
    /**
     * Create a new Metric Filter on this Log Group
     *
     * @param id Unique identifier for the construct in its parent
     * @param props Properties for creating the MetricFilter
     */
    addMetricFilter(id: string, props: MetricFilterOptions): MetricFilter;
    /**
     * Extract a metric from structured log events in the LogGroup
     *
     * Creates a MetricFilter on this LogGroup that will extract the value
     * of the indicated JSON field in all records where it occurs.
     *
     * The metric will be available in CloudWatch Metrics under the
     * indicated namespace and name.
     *
     * @param jsonField JSON field to extract (example: '$.myfield')
     * @param metricNamespace Namespace to emit the metric under
     * @param metricName Name to emit the metric under
     * @returns A Metric object representing the extracted metric
     */
    extractMetric(jsonField: string, metricNamespace: string, metricName: string): cloudwatch.Metric;
    /**
     * Give permissions to write to create and write to streams in this log group
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Give the indicated permissions on this log group and all streams
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * How long, in days, the log contents will be retained.
 */
export declare enum RetentionDays {
    /**
     * 1 day
     */
    OneDay = 1,
    /**
     * 3 days
     */
    ThreeDays = 3,
    /**
     * 5 days
     */
    FiveDays = 5,
    /**
     * 1 week
     */
    OneWeek = 7,
    /**
     * 2 weeks
     */
    TwoWeeks = 14,
    /**
     * 1 month
     */
    OneMonth = 30,
    /**
     * 2 months
     */
    TwoMonths = 60,
    /**
     * 3 months
     */
    ThreeMonths = 90,
    /**
     * 4 months
     */
    FourMonths = 120,
    /**
     * 5 months
     */
    FiveMonths = 150,
    /**
     * 6 months
     */
    SixMonths = 180,
    /**
     * 1 year
     */
    OneYear = 365,
    /**
     * 13 months
     */
    ThirteenMonths = 400,
    /**
     * 18 months
     */
    EighteenMonths = 545,
    /**
     * 2 years
     */
    TwoYears = 731,
    /**
     * 5 years
     */
    FiveYears = 1827,
    /**
     * 10 years
     */
    TenYears = 3653
}
/**
 * Properties for a LogGroup
 */
export interface LogGroupProps {
    /**
     * Name of the log group.
     *
     * @default Automatically generated
     */
    readonly logGroupName?: string;
    /**
     * How long, in days, the log contents will be retained.
     *
     * To retain all logs, set this value to Infinity.
     *
     * @default 731 days (2 years)
     */
    readonly retentionDays?: RetentionDays;
    /**
     * Determine the removal policy of this log group.
     *
     * Normally you want to retain the log group so you can diagnose issues
     * from logs even after a deployment that no longer includes the log group.
     * In that case, use the normal date-based retention policy to age out your
     * logs.
     *
     * @default RemovalPolicy.Retain
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Define a CloudWatch Log Group
 */
export declare class LogGroup extends LogGroupBase {
    /**
     * Import an existing LogGroup
     */
    static fromLogGroupArn(scope: Construct, id: string, logGroupArn: string): ILogGroup;
    /**
     * The ARN of this log group
     */
    readonly logGroupArn: string;
    /**
     * The name of this log group
     */
    readonly logGroupName: string;
    constructor(scope: Construct, id: string, props?: LogGroupProps);
}
/**
 * Properties for a new LogStream created from a LogGroup
 */
export interface StreamOptions {
    /**
     * The name of the log stream to create.
     *
     * The name must be unique within the log group.
     *
     * @default Automatically generated
     */
    readonly logStreamName?: string;
}
/**
 * Properties for a new SubscriptionFilter created from a LogGroup
 */
export interface SubscriptionFilterOptions {
    /**
     * The destination to send the filtered events to.
     *
     * For example, a Kinesis stream or a Lambda function.
     */
    readonly destination: ILogSubscriptionDestination;
    /**
     * Log events matching this pattern will be sent to the destination.
     */
    readonly filterPattern: IFilterPattern;
}
/**
 * Properties for a MetricFilter created from a LogGroup
 */
export interface MetricFilterOptions {
    /**
     * Pattern to search for log events.
     */
    readonly filterPattern: IFilterPattern;
    /**
     * The namespace of the metric to emit.
     */
    readonly metricNamespace: string;
    /**
     * The name of the metric to emit.
     */
    readonly metricName: string;
    /**
     * The value to emit for the metric.
     *
     * Can either be a literal number (typically "1"), or the name of a field in the structure
     * to take the value from the matched event. If you are using a field value, the field
     * value must have been matched using the pattern.
     *
     * If you want to specify a field from a matched JSON structure, use '$.fieldName',
     * and make sure the field is in the pattern (if only as '$.fieldName = *').
     *
     * If you want to specify a field from a matched space-delimited structure,
     * use '$fieldName'.
     *
     * @default "1"
     */
    readonly metricValue?: string;
    /**
     * The value to emit if the pattern does not match a particular event.
     *
     * @default No metric emitted.
     */
    readonly defaultValue?: number;
}
export {};
