"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'fixed retention'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retentionDays: lib_1.RetentionDays.OneWeek
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 7
        }));
        test.done();
    },
    'default retention'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 731
        }));
        test.done();
    },
    'infinite retention/dont delete log group by default'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retentionDays: Infinity
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'will delete log group if asked to'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retentionDays: Infinity,
            removalPolicy: cdk_1.RemovalPolicy.Destroy
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Delete"
                }
            }
        }));
        test.done();
    },
    'export/import'(test) {
        // GIVEN
        const stack2 = new cdk_1.Stack();
        // WHEN
        const imported = lib_1.LogGroup.fromLogGroupArn(stack2, 'lg', 'arn:aws:logs:us-east-1:123456789012:log-group:my-log-group');
        imported.addStream('MakeMeAStream');
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::Logs::LogStream', {
            LogGroupName: "my-log-group"
        }));
        test.done();
    },
    'extractMetric'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: "{ $.myField = \"*\" }",
            LogGroupName: { Ref: "LogGroupF5B46931" },
            MetricTransformations: [
                {
                    MetricName: "Field",
                    MetricNamespace: "MyService",
                    MetricValue: "$.myField"
                }
            ]
        }));
        test.equal(metric.namespace, 'MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'extractMetric allows passing in namespaces with "/"'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyNamespace/MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: "{ $.myField = \"*\" }",
            MetricTransformations: [
                {
                    MetricName: "Field",
                    MetricNamespace: "MyNamespace/MyService",
                    MetricValue: "$.myField"
                }
            ]
        }));
        test.equal(metric.namespace, 'MyNamespace/MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'grant'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        const user = new iam.User(stack, 'User');
        // WHEN
        lg.grantWrite(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: ["logs:CreateLogStream", "logs:PutLogEvents"],
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["LogGroupF5B46931", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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