"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Schedule for scheduled event rules
 */
class Schedule {
    constructor() {
    }
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that Cloudwatch Events will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error(`Cannot supply both 'day' and 'weekDay', use at most one`);
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
}
exports.Schedule = Schedule;
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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