"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(id, resource) {
        this.id = id;
        this.resource = resource;
    }
    bind() {
        return {
            id: this.id || '',
            arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
            targetResource: this.resource,
        };
    }
}
module.exports = {
    'default rule'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(10 minutes)",
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'can get rule name'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        new cdk.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                RuleName: rule.ruleName
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            RuleName: { Ref: 'MyRuleA44AB831' }
        }));
        test.done();
    },
    'rule with physical name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            ruleName: 'PhysicalName',
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Name: 'PhysicalName'
        }));
        test.done();
    },
    'eventPattern is rendered properly'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                account: ['account1', 'account2'],
                detail: {
                    foo: [1, 2],
                },
                detailType: ['detailType1'],
                id: ['id1', 'id2'],
                region: ['region1', 'region2', 'region3'],
                resources: ['r1'],
                source: ['src1', 'src2'],
                time: ['t1'],
                version: ['0']
            }
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "EventPattern": {
                            account: ['account1', 'account2'],
                            detail: { foo: [1, 2] },
                            'detail-type': ['detailType1'],
                            id: ['id1', 'id2'],
                            region: ['region1', 'region2', 'region3'],
                            resources: ['r1'],
                            source: ['src1', 'src2'],
                            time: ['t1'],
                            version: ['0']
                        },
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'fails synthesis if neither eventPattern nor scheudleExpression are specified'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        new rule_1.Rule(stack, 'Rule');
        test.throws(() => app.synth(), /Either 'eventPattern' or 'schedule' must be defined/);
        test.done();
    },
    'addEventPattern can be used to add filters'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule');
        rule.addEventPattern({
            account: ['12345'],
            detail: {
                foo: ['hello']
            }
        });
        rule.addEventPattern({
            source: ['aws.source'],
            detail: {
                foo: ['bar'],
                goo: {
                    hello: ['world']
                }
            }
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "EventPattern": {
                            "account": [
                                "12345"
                            ],
                            "detail": {
                                "foo": [
                                    "hello",
                                    "bar"
                                ],
                                "goo": {
                                    "hello": [
                                        "world"
                                    ]
                                }
                            },
                            "source": [
                                "aws.source"
                            ]
                        },
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'targets can be added via props or addTarget with input transformer'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
            })
        };
        const t2 = {
            bind: () => ({
                id: '',
                arn: 'ARN2',
                input: lib_1.RuleTargetInput.fromText(`This is ${lib_1.EventField.fromPath('$.detail.bla')}`),
            })
        };
        const rule = new rule_1.Rule(stack, 'EventRule', {
            targets: [t1],
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
        });
        rule.addTarget(t2);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "EventRule5A491D2C": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(5 minutes)",
                        "State": "ENABLED",
                        "Targets": [
                            {
                                "Arn": "ARN1",
                                "Id": "Target0",
                                "KinesisParameters": {
                                    "PartitionKeyPath": "partitionKeyPath"
                                }
                            },
                            {
                                "Arn": "ARN2",
                                "Id": "Target1",
                                "InputTransformer": {
                                    "InputPathsMap": {
                                        "detail-bla": "$.detail.bla"
                                    },
                                    "InputTemplate": "\"This is <detail-bla>\""
                                },
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'input template can contain tokens'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '', arn: 'ARN2', input: lib_1.RuleTargetInput.fromText('Hello, "world"')
            })
        });
        // tokens are used here (FnConcat), but this is a text template so we
        // expect it to be wrapped in double quotes automatically for us.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join('', ['a', 'b']).toString()),
            })
        });
        // jsonTemplate can be used to format JSON documents with replacements
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN3',
                input: lib_1.RuleTargetInput.fromObject({ foo: lib_1.EventField.fromPath('$.detail.bar') }),
            })
        });
        // tokens can also used for JSON templates.
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN4',
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join(' ', ['hello', '"world"']).toString()),
            })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "EventRule5A491D2C": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "State": "ENABLED",
                        "ScheduleExpression": "rate(1 minute)",
                        "Targets": [
                            {
                                "Arn": "ARN2",
                                "Id": "Target0",
                                "Input": '"Hello, \\"world\\""',
                            },
                            {
                                "Arn": "ARN1",
                                "Id": "Target1",
                                "Input": "\"ab\"",
                                "KinesisParameters": {
                                    "PartitionKeyPath": "partitionKeyPath"
                                }
                            },
                            {
                                "Arn": "ARN3",
                                "Id": "Target2",
                                "InputTransformer": {
                                    "InputPathsMap": {
                                        "detail-bar": "$.detail.bar"
                                    },
                                    "InputTemplate": "{\"foo\":<detail-bar>}"
                                }
                            },
                            {
                                "Arn": "ARN4",
                                "Id": "Target3",
                                "Input": '"hello \\"world\\""'
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'target can declare role which will be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        const role = new iam.Role(stack, 'SomeRole', {
            assumedBy: new iam.ServicePrincipal('nobody')
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: '',
                arn: 'ARN2',
                role,
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            "Targets": [
                {
                    "Arn": "ARN2",
                    "Id": "Target0",
                    "RoleArn": { "Fn::GetAtt": ["SomeRole6DDC54DD", "Arn"] }
                }
            ]
        }));
        test.done();
    },
    'asEventRuleTarget can use the ruleArn and a uniqueId of the rule'(test) {
        const stack = new cdk.Stack();
        let receivedRuleArn = 'FAIL';
        let receivedRuleId = 'FAIL';
        const t1 = {
            bind: (eventRule) => {
                receivedRuleArn = eventRule.ruleArn;
                receivedRuleId = eventRule.node.uniqueId;
                return {
                    id: '',
                    arn: 'ARN1',
                    kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
                };
            }
        };
        const rule = new rule_1.Rule(stack, 'EventRule');
        rule.addTarget(t1);
        test.deepEqual(stack.resolve(receivedRuleArn), stack.resolve(rule.ruleArn));
        test.deepEqual(receivedRuleId, rule.node.uniqueId);
        test.done();
    },
    'fromEventRuleArn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const importedRule = rule_1.Rule.fromEventRuleArn(stack, 'ImportedRule', 'arn:aws:events:us-east-2:123456789012:rule/example');
        // THEN
        test.deepEqual(importedRule.ruleArn, 'arn:aws:events:us-east-2:123456789012:rule/example');
        test.deepEqual(importedRule.ruleName, 'example');
        test.done();
    },
    'rule can be disabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "State": "DISABLED"
        }));
        test.done();
    },
    'can add multiple targets with the same id'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        rule.addTarget(new SomeTarget());
        rule.addTarget(new SomeTarget());
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    "Arn": "ARN1",
                    "Id": "Target0",
                    "KinesisParameters": {
                        "PartitionKeyPath": "partitionKeyPath"
                    }
                },
                {
                    "Arn": "ARN1",
                    "Id": "Target1",
                    "KinesisParameters": {
                        "PartitionKeyPath": "partitionKeyPath"
                    }
                }
            ]
        }));
        test.done();
    },
    'rule and target must be in the same region'(test) {
        const app = new cdk.App();
        const sourceStack = new cdk.Stack(app, 'SourceStack');
        const rule = new rule_1.Rule(sourceStack, 'Rule');
        const targetStack = new cdk.Stack(app, 'TargetStack', { env: { region: 'us-west-2' } });
        const resource = new cdk.Construct(targetStack, 'Resource');
        test.throws(() => {
            rule.addTarget(new SomeTarget('T', resource));
        }, /Rule and target must be in the same region/);
        test.done();
    },
    'sqsParameters are generated when they are specified in target props'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: '',
                arn: 'ARN1',
                sqsParameters: { messageGroupId: 'messageGroupId' }
            })
        };
        new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
            targets: [t1],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: [
                {
                    "Arn": "ARN1",
                    "Id": "Target0",
                    "SqsParameters": {
                        "MessageGroupId": "messageGroupId"
                    }
                }
            ]
        }));
        test.done();
    },
    'associate rule with event bus'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                detail: ['detail']
            },
            eventBus,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventBusName: {
                Ref: 'EventBus7B8748AA'
            }
        }));
        test.done();
    },
    'throws with eventBus and schedule'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        const eventBus = new lib_1.EventBus(stack, 'EventBus');
        // THEN
        test.throws(() => new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
            eventBus,
        }), /Cannot associate rule with 'eventBus' when using 'schedule'/);
        test.done();
    },
    'for cross-account targets': {
        'requires that the source stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceStack = new cdk.Stack(app, 'SourceStack');
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the source stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete account'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetStack = new cdk.Stack(app, 'TargetStack');
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete account for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the target stack specify a concrete region'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /You need to provide a concrete region for the target stack when using cross-account events/);
            test.done();
        },
        'requires that the source stack be part of an App'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(undefined, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack which uses cross-account targets must be part of a CDK app/);
            test.done();
        },
        'requires that the target stack be part of an App'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(undefined, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Target stack which uses cross-account event targets must be part of a CDK app/);
            test.done();
        },
        'requires that the source and target stacks be part of the same App'(test) {
            const sourceApp = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(sourceApp, 'SourceStack', { env: { account: sourceAccount, region: 'us-west-2' } });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetApp = new cdk.App();
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(targetApp, 'TargetStack', { env: { account: targetAccount, region: 'us-west-2' } });
            const resource = new cdk.Construct(targetStack, 'Resource');
            test.throws(() => {
                rule.addTarget(new SomeTarget('T', resource));
            }, /Event stack and target stack must belong to the same CDK app/);
            test.done();
        },
        'generates the correct rules in the source and target stacks when eventPattern is passed in the constructor'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule', {
                eventPattern: {
                    source: ['some-event'],
                },
            });
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource1 = new cdk.Construct(targetStack, 'Resource1');
            const resource2 = new cdk.Construct(targetStack, 'Resource2');
            rule.addTarget(new SomeTarget('T1', resource1));
            rule.addTarget(new SomeTarget('T2', resource2));
            assert_1.expect(sourceStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T1",
                        "Arn": {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    { "Ref": "AWS::Partition" },
                                    `:events:us-west-2:${targetAccount}:event-bus/default`,
                                ],
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T1",
                        "Arn": "ARN1",
                    },
                ],
            }));
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T2",
                        "Arn": "ARN1",
                    },
                ],
            }));
            const eventBusPolicyStack = app.node.findChild(`EventBusPolicy-${sourceAccount}-us-west-2-${targetAccount}`);
            assert_1.expect(eventBusPolicyStack).to(assert_1.haveResourceLike('AWS::Events::EventBusPolicy', {
                "Action": "events:PutEvents",
                "StatementId": "MySid",
                "Principal": sourceAccount,
            }));
            test.done();
        },
        'generates the correct rule in the target stack when addEventPattern in the source rule is used'(test) {
            const app = new cdk.App();
            const sourceAccount = '123456789012';
            const sourceStack = new cdk.Stack(app, 'SourceStack', {
                env: {
                    account: sourceAccount,
                    region: 'us-west-2',
                },
            });
            const rule = new rule_1.Rule(sourceStack, 'Rule');
            const targetAccount = '234567890123';
            const targetStack = new cdk.Stack(app, 'TargetStack', {
                env: {
                    account: targetAccount,
                    region: 'us-west-2',
                },
            });
            const resource = new cdk.Construct(targetStack, 'Resource1');
            rule.addTarget(new SomeTarget('T', resource));
            rule.addEventPattern({
                source: ['some-event'],
            });
            assert_1.expect(targetStack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "some-event",
                    ],
                },
                "State": "ENABLED",
                "Targets": [
                    {
                        "Id": "T",
                        "Arn": "ARN1",
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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