"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventBus = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
/**
 * Define an EventBridge EventBus.
 *
 * @stability stable
 * @resource AWS::Events::EventBus
 */
class EventBus extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.string({ produce: () => core_1.Names.uniqueId(this) }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: eventBusName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param eventBusArn ARN of imported event bus.
     * @stability stable
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = eventBusArn;
                this.eventBusName = parts.resourceName || '';
                this.eventBusPolicy = '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing event bus resource.
     *
     * @param scope Parent construct.
     * @param id Construct ID.
     * @param attrs Imported event bus properties.
     * @stability stable
     */
    static fromEventBusAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = attrs.eventBusArn;
                this.eventBusName = attrs.eventBusName;
                this.eventBusPolicy = attrs.eventBusPolicy;
                this.eventSourceName = attrs.eventSourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    static grantPutEvents(grantee) {
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/eventbridge/latest/userguide/permissions-reference-eventbridge.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
            }
            else if (eventBusName !== undefined && !core_1.Token.isUnresolved(eventBusName)) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
_a = JSII_RTTI_SYMBOL_1;
EventBus[_a] = { fqn: "@aws-cdk/aws-events.EventBus", version: "1.91.0" };
//# sourceMappingURL=data:application/json;base64,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