"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const fs = require("fs");
const path = require("path");
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * (experimental) The code the canary should execute.
 *
 * @experimental
 */
class Code {
    /**
     * (experimental) Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `InlineCode` with inline code.
     * @experimental
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * (experimental) Specify code from a local path.
     *
     * Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param assetPath Either a directory or a .zip file.
     * @returns `AssetCode` associated with the specified path.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * (experimental) Specify code from an s3 bucket.
     *
     * The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @returns `S3Code` associated with the specified S3 object.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
/**
 * (experimental) Canary code from an Asset.
 *
 * @experimental
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     * @experimental
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(scope, handler) {
        this.validateCanaryAsset(handler);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = `${handler.split('.')[0]}.js`;
            if (!fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', filename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${filename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
/**
 * (experimental) Canary code from an inline string.
 *
 * @experimental
 */
class InlineCode extends Code {
    /**
     * @experimental
     */
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope, handler) {
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * (experimental) S3 bucket path to the code zip file.
 *
 * @experimental
 */
class S3Code extends Code {
    /**
     * @experimental
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope, _handler) {
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
//# sourceMappingURL=data:application/json;base64,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