"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LowCostECS = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const lib = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfn_tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
class LowCostECS extends lib.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.vpc =
            props.vpc ??
                new ec2.Vpc(this, 'Vpc', {
                    natGateways: 0,
                    subnetConfiguration: [
                        {
                            name: 'PublicSubnet',
                            subnetType: ec2.SubnetType.PUBLIC,
                        },
                    ],
                });
        this.cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            containerInsights: props.containerInsights,
        });
        this.hostAutoScalingGroup = this.cluster.addCapacity('HostInstanceCapacity', {
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, {
                cachedInContext: true,
            }),
            instanceType: new ec2.InstanceType(props.hostInstanceType ?? 't2.micro'),
            spotPrice: props.hostInstanceSpotPrice,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            associatePublicIpAddress: true,
            minCapacity: 1,
            maxCapacity: 1,
        });
        if (props.securityGroup) {
            this.hostAutoScalingGroup.addSecurityGroup(props.securityGroup);
        }
        else {
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(443));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(443));
        }
        /**
         * Add managed policy to allow ssh through ssm manager
         */
        this.hostAutoScalingGroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        /**
         * Add policy to associate elastic ip on startup
         */
        this.hostAutoScalingGroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['ec2:DescribeAddresses', 'ec2:AssociateAddress'],
            resources: ['*'],
        }));
        const hostInstanceIp = new ec2.CfnEIP(this, 'HostInstanceIp');
        const tagUniqueId = lib.Names.uniqueId(hostInstanceIp);
        hostInstanceIp.tags.setTag('Name', tagUniqueId);
        const awsCliTag = props.awsCliDockerTag ?? 'latest';
        this.hostAutoScalingGroup.addUserData('INSTANCE_ID=$(curl --silent http://169.254.169.254/latest/meta-data/instance-id)', `ALLOCATION_ID=$(docker run --net=host amazon/aws-cli:${awsCliTag} ec2 describe-addresses --region ${this.hostAutoScalingGroup.env.region} --filter Name=tag:Name,Values=${tagUniqueId} --query 'Addresses[].AllocationId' --output text | head)`, `docker run --net=host amazon/aws-cli:${awsCliTag} ec2 associate-address --region ${this.hostAutoScalingGroup.env.region} --instance-id "$INSTANCE_ID" --allocation-id "$ALLOCATION_ID" --allow-reassociation`);
        this.certFileSystem = new aws_efs_1.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
            securityGroup: new ec2.SecurityGroup(this, 'FileSystemSecurityGroup', {
                vpc: this.vpc,
                allowAllOutbound: false,
            }),
            removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
        });
        this.certFileSystem.connections.allowDefaultPortTo(this.hostAutoScalingGroup);
        this.certFileSystem.connections.allowDefaultPortFrom(this.hostAutoScalingGroup);
        /**
         * ARecord to Elastic ip
         */
        const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZone', {
            domainName: props.hostedZoneDomain,
        });
        const records = props.recordDomainNames ?? [hostedZone.zoneName];
        records.forEach((record) => new route53.ARecord(this, `ARecord${record}`, {
            zone: hostedZone,
            recordName: record,
            target: route53.RecordTarget.fromIpAddresses(hostInstanceIp.ref),
        }));
        /**
         * Certbot Task Definition
         * Mounts generated certificate to EFS
         */
        const logGroup = props.logGroup ??
            new aws_logs_1.LogGroup(this, 'LogGroup', {
                retention: aws_logs_1.RetentionDays.TWO_YEARS,
                removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
            });
        const certbotTaskDefinition = new ecs.Ec2TaskDefinition(this, 'CertbotTaskDefinition');
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ListHostedZones', 'route53:GetChange'],
            resources: ['*'],
        }));
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ChangeResourceRecordSets'],
            resources: [hostedZone.hostedZoneArn],
        }));
        const certbotTag = props.certbotDockerTag ?? 'v1.29.0';
        const certbotContainer = certbotTaskDefinition.addContainer('CertbotContainer', {
            image: ecs.ContainerImage.fromRegistry(`certbot/dns-route53:${certbotTag}`),
            containerName: 'certbot',
            memoryReservationMiB: 64,
            command: [
                'certonly',
                '--verbose',
                '--preferred-challenges=dns-01',
                '--dns-route53',
                '--dns-route53-propagation-seconds=300',
                '--non-interactive',
                '--agree-tos',
                '--expand',
                '-m',
                props.email,
                '--cert-name',
                records[0],
                ...records.flatMap((domain) => ['-d', domain]),
            ],
            logging: ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: certbotTag,
            }),
        });
        this.certFileSystem.grant(certbotTaskDefinition.taskRole, 'elasticfilesystem:ClientWrite');
        certbotTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        certbotContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: false,
        });
        /**
         * Schedule Certbot certificate create/renew on Step Functions
         * Sends email notification on certbot failure
         */
        const topic = new aws_sns_1.Topic(this, 'Topic');
        new aws_sns_1.Subscription(this, 'EmailSubscription', {
            topic: topic,
            protocol: aws_sns_1.SubscriptionProtocol.EMAIL,
            endpoint: props.email,
        });
        const certbotRunTask = new sfn_tasks.EcsRunTask(this, 'CreateCertificate', {
            cluster: this.cluster,
            taskDefinition: certbotTaskDefinition,
            launchTarget: new sfn_tasks.EcsEc2LaunchTarget(),
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        });
        certbotRunTask.addCatch(new sfn_tasks.SnsPublish(this, 'SendEmailOnFailure', {
            topic: topic,
            message: sfn.TaskInput.fromJsonPathAt('$'),
        }).next(new sfn.Fail(this, 'Fail')));
        certbotRunTask.addRetry({
            interval: lib.Duration.seconds(20),
        });
        const certbotStateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: certbotRunTask,
        });
        new aws_events_1.Rule(this, 'CertbotScheduleRule', {
            schedule: aws_events_1.Schedule.rate(lib.Duration.days(props.certbotScheduleInterval ?? 60)),
            targets: [new aws_events_targets_1.SfnStateMachine(certbotStateMachine)],
        });
        /**
         * Server ECS task
         */
        const serverTaskDefinition = props.serverTaskDefinition
            ? this.createTaskDefinition(props.serverTaskDefinition)
            : this.createSampleTaskDefinition(records, logGroup);
        this.certFileSystem.grant(serverTaskDefinition.taskRole, 'elasticfilesystem:ClientMount');
        serverTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        serverTaskDefinition.defaultContainer?.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: true,
        });
        /**
         * AWS cli container to execute certbot sfn before the default container startup.
         */
        serverTaskDefinition.defaultContainer?.addContainerDependencies({
            container: serverTaskDefinition.addContainer('AWSCliContainer', {
                image: ecs.ContainerImage.fromRegistry(`amazon/aws-cli:${awsCliTag}`),
                containerName: 'aws-cli',
                memoryReservationMiB: 64,
                entryPoint: ['/bin/bash', '-c'],
                command: [
                    `set -eux
          aws configure set region ${certbotStateMachine.env.region} && \\
          aws configure set output text && \\
          EXECUTION_ARN=$(aws stepfunctions start-execution --state-machine-arn ${certbotStateMachine.stateMachineArn} --query executionArn) && \\
          until [ $(aws stepfunctions describe-execution --execution-arn "$EXECUTION_ARN" --query status) != RUNNING ];
          do
            echo "Waiting for $EXECUTION_ARN"
            sleep 10
          done`,
                ],
                essential: false,
                logging: ecs.LogDriver.awsLogs({
                    logGroup: logGroup,
                    streamPrefix: awsCliTag,
                }),
            }),
            condition: ecs.ContainerDependencyCondition.COMPLETE,
        });
        certbotStateMachine.grantExecution(serverTaskDefinition.taskRole, 'states:DescribeExecution');
        certbotStateMachine.grantStartExecution(serverTaskDefinition.taskRole);
        this.service = new ecs.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition: serverTaskDefinition,
            desiredCount: 1,
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            circuitBreaker: {
                rollback: true,
            },
            enableExecuteCommand: true,
        });
        new lib.CfnOutput(this, 'PublicIpAddress', { value: hostInstanceIp.ref });
        new lib.CfnOutput(this, 'certbotStateMachineName', { value: certbotStateMachine.stateMachineName });
        new lib.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
        new lib.CfnOutput(this, 'ServiceName', { value: this.service.serviceName });
    }
    createTaskDefinition(taskDefinitionOptions) {
        const serverTaskDefinition = new ecs.Ec2TaskDefinition(this, 'ServerTaskDefinition', taskDefinitionOptions.taskDefinition);
        taskDefinitionOptions.containers?.forEach((props, index) => {
            const container = serverTaskDefinition.addContainer(props.containerName ?? `container${index}`, props);
            container.addPortMappings(...(props.portMappings ?? []));
        });
        taskDefinitionOptions.volumes?.forEach((props) => serverTaskDefinition.addVolume(props));
        return serverTaskDefinition;
    }
    createSampleTaskDefinition(records, logGroup) {
        const nginxTaskDefinition = new ecs.Ec2TaskDefinition(this, 'NginxTaskDefinition');
        const nginxContainer = nginxTaskDefinition.addContainer('NginxContainer', {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../examples/containers/nginx')),
            containerName: 'nginx',
            memoryReservationMiB: 64,
            essential: true,
            environment: {
                SERVER_NAME: records.join(' '),
                CERT_NAME: records[0],
            },
            logging: ecs.LogDrivers.awsLogs({
                logGroup: logGroup,
                streamPrefix: 'nginx-proxy',
            }),
        });
        nginxContainer.addPortMappings({
            hostPort: 80,
            containerPort: 80,
            protocol: ecs.Protocol.TCP,
        }, {
            hostPort: 443,
            containerPort: 443,
            protocol: ecs.Protocol.TCP,
        });
        return nginxTaskDefinition;
    }
}
exports.LowCostECS = LowCostECS;
_a = JSII_RTTI_SYMBOL_1;
LowCostECS[_a] = { fqn: "low-cost-ecs.LowCostECS", version: "0.0.19" };
//# sourceMappingURL=data:application/json;base64,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