# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['licensespring', 'licensespring.api']

package_data = \
{'': ['*']}

install_requires = \
['pycryptodome>=3.14.1,<4.0.0', 'requests>=2.27.1,<3.0.0']

setup_kwargs = {
    'name': 'licensespring',
    'version': '1.0.6',
    'description': 'LicenseSpring Python Library',
    'long_description': '# LicenseSpring Python Library\n\nThe LicenseSpring Python Library provides convenient access to the LicenseSpring API from\napplications written in the Python language.\n\n## Installation\n\nInstall `licensespring` library:\n\n```\npip install licensespring\n```\n\nRequires: Python >=3.7\n\n## Hardware (Device) IDs\n\nThis library provides a preconfigured identity provider which uses [uuid.getnode()](https://docs.python.org/3/library/uuid.html#uuid.getnode) to generate unique ID per device as described:\n\n> Get the hardware address as a 48-bit positive integer. The first time this runs, it may launch a separate program, which could be quite slow. If all attempts to obtain the hardware address fail, we choose a random 48-bit number with the multicast bit (least significant bit of the first octet) set to 1 as recommended in RFC 4122. “Hardware address” means the MAC address of a network interface. On a machine with multiple network interfaces, universally administered MAC addresses (i.e. where the second least significant bit of the first octet is unset) will be preferred over locally administered MAC addresses, but with no other ordering guarantees.\n\nAll of the methods exposed by hardware identity provider:\n```python\nclass HardwareIdProvider:\n    def get_id(self):\n        return str(uuid.getnode())\n\n    def get_os_ver(self):\n        return platform.platform()\n\n    def get_hostname(self):\n        return platform.node()\n\n    def get_ip(self):\n        return socket.gethostbyname(self.get_hostname())\n\n    def get_is_vm(self):\n        return False\n\n    def get_vm_info(self):\n        return None\n\n    def get_mac_address(self):\n        return ":".join(("%012X" % uuid.getnode())[i : i + 2] for i in range(0, 12, 2))\n\n    def get_request_id(self):\n        return str(uuid.uuid4())\n```\n\nTo overwrite any of these methods extend the `HardwareIdProvider`, overwrite the methods you want and provide it when initializing the APIClient:\n```python\nclass CustomHardwareIdProvider(HardwareIdProvider):\n    def get_id(self):\n        return "_my_id_"\n\napi_client = APIClient(api_key="_your_api_key_", shared_key="_your_shared_key_", hardware_id_provider=CustomHardwareIdProvider)\n```\n\n## APIClient Usage Examples\n\n### Set app version\n```python\nimport licensespring\n\nlicensespring.app_version = "MyApp 1.0.0"\n```\n\n### Create APIClient\n```python\nfrom licensespring.api import APIClient\n\napi_client = APIClient(api_key="_your_api_key_", shared_key="_your_shared_key_")\n```\n\n### Activate key based license\n```python\nproduct = "lkprod1"\nlicense_key = "GPB7-279T-6MNK-CQLK"\nlicense_data = api_client.activate_license(product=product, license_key=license_key)\n\nprint(license_data)\n```\n\n### Activate user based license\n```python\nproduct = "uprod1"\nusername = "user1@email.com"\npassword = "nq64k1!@"\n\nlicense_data = api_client.activate_license(\n    product=product, username=username, password=password\n)\n\nprint(license_data)\n```\n\n### Deactivate key based license\n```python\nproduct = "lkprod1"\nlicense_key = "GPUB-J4PH-CGNK-C7LK"\nis_deactivated = api_client.deactivate_license(product=product, license_key=license_key)\n\nprint(is_deactivated)\n```\n\n### Deactivate user based license\n```python\nproduct = "uprod1"\nusername = "user1@email.com"\npassword = "nq64k1!@"\n\nlicense_data = api_client.deactivate_license(\n    product=product, username=username, password=password\n)\n\nprint(license_data)\n```\n\n### Check key based license\n```python\nproduct = "lkprod1"\nlicense_key = "GPBQ-DZCP-E9SK-CQLK"\n\nlicense_data = api_client.check_license(product=product, license_key=license_key)\n\nprint(license_data)\n```\n\n### Check user based license\n```python\nproduct = "uprod1"\nusername = "user2@email.com"\npassword = "1l48y#!b"\n\nlicense_data = api_client.check_license(product=product, username=username)\n\nprint(license_data)\n```\n\n### Add consumption\n```python\nproduct = "lkprod1"\nlicense_key = "GPSU-QTKQ-HSSK-C9LK"\n\n# Add 1 consumption\nconsumption_data = api_client.add_consumption(\n    product=product, license_key=license_key\n)\n\n# Add 3 consumptions\nconsumption_data = api_client.add_consumption(\n    product=product, license_key=license_key, consumptions=3\n)\n\n# Add 1 consumption, allow overages and define max overages\nconsumption_data = api_client.add_consumption(\n    product=product, license_key=license_key, allow_overages=True, max_overages=10\n)\n\nprint(consumption_data)\n```\n\n### Add feature consumption\n```python\nproduct = "lkprod1"\nlicense_key = "GPTJ-LSYZ-USEK-C8LK"\nfeature = "lkprod1cf1"\n\n# Add 1 consumption\nfeature_consumption_data = api_client.add_feature_consumption(\n    product=product, license_key=license_key, feature=feature\n)\n\n# Add 3 consumptions\nfeature_consumption_data = api_client.add_feature_consumption(\n    product=product, license_key=license_key, feature=feature, consumptions=3\n)\n\nprint(feature_consumption_data)\n```\n\n### Trial key\n```python\nproduct = "lkprod2"\n\ntrial_license_data = api_client.trial_key(product=product)\n\nprint(trial_license_data)\n```\n\n### Product details\n```python\nproduct = "lkprod1"\n\nproduct_data = api_client.product_details(product=product)\n\nprint(product_data)\n```\n\n### Track device variables\n```python\nproduct = "lkprod1"\nlicense_key = "GPUB-SZF9-AB2K-C7LK"\nvariables = {"variable_1_key": "variable_1_value", "variable_2_key": "variable_2_value"}\n\nis_tracked = api_client.track_device_variables(product=product, license_key=license_key, variables=variables)\n\nprint(is_tracked)\n```\n\n### Get device variables\n```python\nproduct = "lkprod1"\nlicense_key = "GPUB-SZF9-AB2K-C7LK"\n\ndevice_variables = api_client.get_device_variables(product=product, license_key=license_key)\n\nprint(device_variables)\n```\n\n### Floating borrow\n```python\nproduct = "lkprod1"\nlicense_key = "GPUC-NGWU-3NJK-C7LK"\n\n# Borrow for 2 hours\nborrowed_until = (datetime.utcnow() + timedelta(hours=2)).isoformat()\nfloating_borrow_data = api_client.floating_borrow(product=product, license_key=license_key, borrowed_until=borrowed_until)\n\nprint(floating_borrow_data)\n```\n\n### Floating release\n```python\nproduct = "lkprod1"\nlicense_key = "GPUC-NGWU-3NJK-C7LK"\n\nis_released = api_client.floating_release(product=product, license_key=license_key)\n\nprint(is_released)\n```\n\n### Change password\n```python\nusername = "user4@email.com"\npassword = "_old_password_"\nnew_password = "_new_password_"\n\nis_password_changed = api_client.change_password(username=username, password=password, new_password=new_password)\n\nprint(is_password_changed)\n```\n\n### Versions\n```python\nproduct = "lkprod1"\nlicense_key = "GPB7-279T-6MNK-CQLK"\n\n# Get versions for all environments\nversions_data = api_client.versions(product=product, license_key=license_key)\n\n# Get versions for mac environment\nmac_versions_data = api_client.versions(\n    product=product, license_key=license_key, env="mac"\n)\n\nprint(versions_data)\n```\n\n### Installation file\n```python\nproduct = "lkprod1"\nlicense_key = "GPB7-279T-6MNK-CQLK"\n\n# Get the latest installation file\ninstallation_file_data = api_client.installation_file(\n    product=product, license_key=license_key\n)\n\n# Get the latest installation file for linux environment\ninstallation_file_data = api_client.installation_file(\n    product=product, license_key=license_key, env="linux"\n)\n\n# Get the latest installation file for version 1.0.0\ninstallation_file_data = api_client.installation_file(\n    product=product, license_key=license_key, version="1.0.0"\n)\n\nprint(installation_file_data)\n```\n\n### Customer license users\n```python\nproduct = "uprod1"\ncustomer = \'c1@c.com\'\n\ncustomer_license_users_data = api_client.customer_license_users(\n    product=product, customer=customer\n)\n\nprint(customer_license_users_data)\n```\n\n### SSO URL\n```python\nproduct = "uprod1"\ncustomer_account_code = "ccorp"\n\nsso_url_data = api_client.sso_url(\n    product=product, customer_account_code=customer_account_code\n)\n\nprint(sso_url_data)\n```\n\n\n### SSO URL with `code` response type\n```python\nproduct = "uprod1"\ncustomer_account_code = "ccorp"\n\nsso_url_data = api_client.sso_url(\n    product=product,\n    customer_account_code=customer_account_code,\n    response_type="code",\n)\n\nprint(sso_url_data)\n```\n\n### Activate offline\n```python\nproduct = "lkprod1"\nlicense_key = "GPY7-VHX9-MDSK-C3LK"\n\n# Generate data for offline activation\nactivate_offline_data = api_client.activate_offline_dump(\n    product=product, license_key=license_key\n)\n\n# Write to file\nwith open(\'activate_offline.req\', mode=\'w\') as f:\n    print(activate_offline_data, file=f)\n\n# Activate offline\nlicense_data = api_client.activate_offline(data=activate_offline_data)\n\nprint(license_data)\n```\n\n### Activate offline load\n```python\n# Read from file\nwith open(\'./ls_activation.lic\') as file:\n    ls_activation_data = file.read()\n\nlicense_data = api_client.activate_offline_load(ls_activation_data)\n\nprint(license_data)\n```\n\n### Deactivate offline\n```python\nproduct = "lkprod1"\nlicense_key = "GPYC-X5J2-L5SK-C3LK"\n\n# Generate data for offline deactivation\ndeactivate_offline_data = api_client.deactivate_offline_dump(\n    product=product, license_key=license_key\n)\n\n# Write to file\nwith open(\'deactivate_offline.req\', mode=\'w\') as f:\n    print(deactivate_offline_data, file=f)\n\n# Deactivate offline\nis_deactivated = api_client.deactivate_offline(data=deactivate_offline_data)\n\nprint(is_deactiavted)\n```\n',
    'author': 'Toni Sredanović',
    'author_email': 'toni@licensespring.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://licensespring.com/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
