# Copyright 2023 ACCESS-NRI and contributors. See the top-level COPYRIGHT file for details.
# SPDX-License-Identifier: Apache-2.0

import dataclasses
from pathlib import Path
from unittest import mock

import intake
import pandas as pd
import pytest
import xarray as xr
from ecgtools.builder import INVALID_ASSET, TRACEBACK
from intake_esm.source import _get_xarray_open_kwargs, _open_dataset
from intake_esm.utils import OPTIONS

from access_nri_intake.source import CORE_COLUMNS, builders
from access_nri_intake.source.utils import _NCFileInfo


@pytest.mark.parametrize(
    "basedirs, builder, kwargs, num_assets, num_valid_assets, num_datasets",
    [
        (["access-om2"], "AccessOm2Builder", {}, 12, 12, 6),
        (
            ["access-cm2/by578", "access-cm2/by578a"],
            "AccessCm2Builder",
            {"ensemble": True},
            18,
            14,
            7,
        ),
        (["access-esm1-5"], "AccessEsm15Builder", {"ensemble": False}, 11, 11, 10),
        (["access-cm3"], "AccessCm3Builder", {}, 32, 31, 13),
        (["access-om3"], "AccessOm3Builder", {}, 14, 14, 7),
        (["mom6"], "Mom6Builder", {}, 27, 27, 15),
        (["roms"], "ROMSBuilder", {}, 4, 4, 1),
        (["access-esm1-6"], "AccessEsm16Builder", {"ensemble": False}, 20, 20, 7),
        (["woa"], "WoaBuilder", {}, 8, 8, 3),
        (["cmip6"], "Cmip6Builder", {"ensemble": False}, 74, 72, 14),
        (["cmip6"], "Cmip6Builder", {"ensemble": True}, 74, 72, 31),
    ],
)
@pytest.mark.filterwarnings("ignore:Time coordinate does not include bounds")
@pytest.mark.filterwarnings("ignore:Unable to parse 4 assets")
@pytest.mark.filterwarnings("ignore:Unable to parse 1 assets")
def test_builder_build(
    tmp_path,
    test_data,
    basedirs,
    builder,
    kwargs,
    num_assets,
    num_valid_assets,
    num_datasets,
):
    """
    Test the various steps of the build process
    """

    Builder = getattr(builders, builder)
    path = [str(test_data / Path(basedir)) for basedir in basedirs]
    builder = Builder(path, **kwargs)

    with pytest.raises(ValueError, match="asset list provided is None"):
        builder.valid_assets

    builder.get_assets()
    assert isinstance(builder.assets, list)
    assert len(builder.assets) == num_assets

    builder.build()
    assert isinstance(builder.df, pd.DataFrame)
    assert len(builder.df) == num_valid_assets
    assert all([col in builder.df.columns for col in CORE_COLUMNS])

    builder.save(name="test", description="test datastore", directory=str(tmp_path))

    cat = intake.open_esm_datastore(
        str(tmp_path / "test.json"),
        columns_with_iterables=builder.columns_with_iterables,
    )
    assert len(cat.df) == num_valid_assets
    assert len(cat) == num_datasets

    assert len(builder.valid_assets) == num_valid_assets

    # Make sure NCFileInfo fields are in dataframe
    for field in dataclasses.fields(_NCFileInfo):
        assert hasattr(builder.df, field.name)


@pytest.mark.parametrize(
    "filename, builder, realm, member, file_id",
    [
        (
            "access-om2/output000/ocean/ocean.nc",
            "AccessOm2Builder",
            "ocean",
            None,
            "ocean.1yr.nv:2.st_ocean:1.xt_ocean:1.yt_ocean:1",
        ),
        (
            "access-om2/output000/ice/OUTPUT/iceh.1900-01.nc",
            "AccessOm2Builder",
            "seaIce",
            None,
            "seaIce.1mon.d2:2.ni:1.nj:1",
        ),
        (
            "access-cm2/by578/history/atm/netCDF/by578a.pd201501_dai.nc",
            "AccessCm2Builder",
            "atmos",
            "by578",
            "atmos.1day.bnds:2.lat:1.lon:1",
        ),
        (
            "access-cm2/by578a/history/atm/netCDF/by578aa.pd201501_dai.nc",
            "AccessCm2Builder",
            "atmos",
            "by578a",
            "atmos.1day.bnds:2.lat:1.lon:1",
        ),
        (
            "access-cm2/by578/history/ice/iceh_d.2015-01.nc",
            "AccessCm2Builder",
            "seaIce",
            "by578",
            "seaIce.1day.d2:2.ni:1.nj:1",
        ),
        (
            "access-cm2/by578/history/ocn/ocean_daily.nc-20150630",
            "AccessCm2Builder",
            "ocean",
            "by578",
            "ocean.1day.nv:2.xt_ocean:1.yt_ocean:1",
        ),
        (
            "access-om3/output000/GMOM_JRA_WD.mom6.h.sfc_1900_01_02.nc",
            "AccessOm3Builder",
            "ocean",
            None,
            "ocean.1day.nv:2.xh:1.yh:1",
        ),
        (
            "access-om3/output000/ocean.stats.nc",
            "AccessOm3Builder",
            "ocean",
            None,
            "ocean.fx.",
        ),
        (
            "access-om3/output000/GMOM_JRA_WD.cice.h.1900-01-01.nc",
            "AccessOm3Builder",
            "seaIce",
            None,
            "seaIce.1day.nbnd:2.ni:1.nj:1",
        ),
        (
            "access-om3/output000/GMOM_JRA_WD.ww3.hi.1900-01-02-00000.nc",
            "AccessOm3Builder",
            "wave",
            None,
            "wave.fx.freq:1.nx:1.ny:1",
        ),
        # MOM6
        (
            "mom6/output000/19000101.ice_daily.nc",
            "Mom6Builder",
            "seaIce",
            None,
            "seaIce.1day.nv:2.xT:1.xTe:1.yT:1.yTe:1",
        ),
        (
            "mom6/output001/19010101.ice_month.nc",
            "Mom6Builder",
            "seaIce",
            None,
            "seaIce.1mon.nv:2.xT:1.xTe:1.yT:1.yTe:1",
        ),
        (
            "mom6/output001/19010101.ocean_annual_rho2.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1yr.nv:2.rho2_i:1.rho2_l:1.xh:1.xq:1.yh:1.yq:1",
        ),
        (
            "mom6/output000/19000101.ocean_annual_z.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1yr.nv:2.xh:1.xq:1.yh:1.yq:1.z_i:1.z_l:1",
        ),
        (
            "mom6/output000/19000101.ocean_daily.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1day.nv:2.xh:1.yh:1",
        ),
        (
            "mom6/output001/19010101.ocean_month_rho2.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1mon.nv:2.rho2_i:1.rho2_l:1.xh:1.yh:1.yq:1",
        ),
        (
            "mom6/output000/19000101.ocean_month_z.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1mon.nv:2.xh:1.xq:1.yh:1.yq:1.z_i:1.z_l:1",
        ),
        (
            "mom6/output001/19010101.ocean_month.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1mon.nv:2.xh:1.xq:1.yh:1.yq:1.zi:1.zl:1",
        ),
        (
            "mom6/output000/19000101.ocean_scalar_annual.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1yr.nv:2.scalar_axis:1",
        ),
        (
            "mom6/output001/19010101.ocean_scalar_month.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.1mon.nv:2.scalar_axis:1",
        ),
        (
            "mom6/output000/19000101.ocean_static.nc",
            "Mom6Builder",
            "ocean",
            None,
            "ocean.fx.xh:1.xq:1.yh:1.yq:1",
        ),
        (
            "roms/roms_his_0016.nc",
            "ROMSBuilder",
            "seaIce",
            None,
            "seaIce.5day.boundary:1.eta_psi:1.eta_rho:1.eta_u:1.eta_v:1.s_rho:1.s_w:1.tracer:1.xi_psi:1.xi_rho:1.xi_u:1.xi_v:1",
        ),
        (
            "access-esm1-6/output000/atmosphere/NetCDF/aiihca.pea1apr.nc",
            "AccessEsm16Builder",
            "atmos",
            None,
            "bnds:2.lat:2.lat_v:2.lon:2.lon_u:2.soil_model_level_number:2",
        ),
        (
            "access-esm1-6/output000/ocean/ocean-2d-fprec_melt_heat-1monthly-mean-ym_0101_01.nc",
            "AccessEsm16Builder",
            "ocean",
            None,
            "nv:2.xt_ocean:2.yt_ocean:2",
        ),
        (
            "woa/woa13_ts_01_mom01.nc",
            "WoaBuilder",
            "ocean",
            None,
            "ocean.fx.GRID_X_T:2.GRID_Y_T:2.ZT:2.03f617824bf731c4",
        ),
        (
            "access-cm3/1981/ocean/access-cm3.mom6.2d.Heat_PmE.1mon.mean.1981.nc",
            "AccessCm3Builder",
            "ocean",
            None,
            "ocean.1mon.nv:2.xh:2.yh:2",
        ),
        (
            "access-cm3/1981/ice/access-cm3.cice.1mon.mean.1981-01.nc",
            "AccessCm3Builder",
            "seaIce",
            None,
            "seaIce.1mon.nbnd:2.nc:2.ni:2.nj:2.nkaer:2.nkbio:2.nkice:2.nksnow:1",
        ),
        (
            "access-cm3/1984/access-cm3.mom6.2d.Rd_dx.1mon.mean.1984.nc",
            "AccessCm3Builder",
            "ocean",
            None,
            "ocean.1mon.nv:2.xh:1440.yh:1142",
        ),
        (
            "access-cm3/1981/atmosphere/atmosa.pa-198102-dai.nc",
            "AccessCm3Builder",
            "atmos",
            None,
            "atmos.1day.bnds:2.lat:2.lat_river:2.lat_v:2.lon:2.lon_river:2.lon_u:2.model_rho_level_number:2.model_theta_level_number:2.pressure:2",
        ),
        (
            "cmip6/agessc_Omon_ACCESS-ESM1-5_ssp585_r3i1p1f1_2081-2100_av.nc",
            "Cmip6Builder",
            "ocean",
            None,
            "ocean.20yr.bnds:2.i:2.j:2.lev:2.vertices:2",
        ),
        (
            "cmip6/uas_Amon_ACCESS-ESM1-5_historical_r9i1p1f1_1981-2000_r360x180.nc",
            "Cmip6Builder",
            "atmos",
            None,
            "atmos.1mon.bnds:2.lat:2.lon:2",
        ),
        (
            "cmip6/uas_Amon_ACCESS-ESM1-5_historical_r9i1p1f1_1981-2000_r360x180.nc",
            "Cmip6Builder",
            "atmos",
            "r9",
            "atmos.1mon.bnds:2.lat:2.lon:2",
        ),
    ],
)
@pytest.mark.filterwarnings("ignore:Time coordinate does not include bounds")
def test_builder_parser(test_data, filename, builder, realm, member, file_id):
    Builder = getattr(builders, builder)
    info = Builder.parser(str(test_data / filename))
    assert info["realm"] == realm
    if member:
        assert info["member"] == member
    assert info["file_id"] == file_id


@mock.patch("access_nri_intake.source.utils._NCFileInfo.to_dict")
@pytest.mark.parametrize(
    "filename",
    [
        "mom6/output000/19000101.ice_daily.nc",
        "mom6/output001/19010101.ice_month.nc",
        "mom6/output001/19010101.ocean_annual_rho2.nc",
        "mom6/output000/19000101.ocean_annual_z.nc",
        "mom6/output000/19000101.ocean_daily.nc",
        "mom6/output001/19010101.ocean_month_rho2.nc",
        "mom6/output000/19000101.ocean_month_z.nc",
        "mom6/output001/19010101.ocean_month.nc",
        "mom6/output000/19000101.ocean_scalar_annual.nc",
        "mom6/output001/19010101.ocean_scalar_month.nc",
        "mom6/output000/19000101.ocean_static.nc",
    ],
)
def test_Mom6Builder_parser_bad_realm(to_dict_mock, test_data, filename):
    to_dict_mock.return_value = {
        "filename": filename.replace("ice", "badrealm").replace("ocean", "badrealm")
    }
    info = builders.Mom6Builder._parser_catch_invalid(str(test_data / filename))
    assert INVALID_ASSET in info.keys()
    assert TRACEBACK in info.keys()
    assert "ParserError" in info[TRACEBACK]


@pytest.mark.parametrize(
    "filename",
    ["rubbish_name.nc"],
)
@pytest.mark.parametrize(
    "builder",
    [
        "AccessOm2Builder",
        "AccessOm3Builder",
        "Mom6Builder",
        "AccessEsm15Builder",
        "AccessEsm16Builder",
        "AccessCm2Builder",
        "ROMSBuilder",
        "WoaBuilder",
        "Cmip6Builder",
    ],
)
def test_builder_parser_exception(test_data, filename, builder):
    Builder = getattr(builders, builder)
    info = Builder._parser_catch_invalid(str(test_data / filename))
    assert INVALID_ASSET in info.keys()
    assert info[INVALID_ASSET] == str(test_data / filename)
    assert TRACEBACK in info.keys()


def test_cmip6_builder_parser_no_ensemble(test_data):
    builder = builders.Cmip6Builder([str(test_data / "cmip6")], ensemble=True)
    with pytest.raises(builders.ParserError, match="Cannot determine member"):
        builder.parser(
            str(
                test_data
                / "cmip6"
                / "vas_Amon_ACCESS-ESM1-5_historical_r9i1p1f1_1981-2000_r360x180_no_realisation_index.nc"
            )
        )


def test_builder_columns_with_iterables(test_data):
    builder = builders.AccessOm2Builder(str(test_data / "access-om2"))
    assert not builder.columns_with_iterables
    builder.build()
    assert sorted(list(builder.columns_with_iterables)) == sorted(
        [
            col
            for col, val in builder.df.map(type).isin([list, tuple, set]).any().items()
            if val
        ]
    )


@pytest.mark.parametrize(
    "builder, filename, time_dim, expected",
    [
        # Example ACCESS-CM2 filenames
        (
            builders.AccessCm2Builder,
            "bz687a.pm107912_mon.nc",
            "time",
            "bnds:2.depth:6.lat:144.lat_v:145.lon:192.lon_u:192.model_rho_level_number:85.model_theta_level_number:85.pressure:19.pseudo_level:5.pseudo_level_0:6.pseudo_level_1:17.pseudo_level_2:13",
        ),
        (
            builders.AccessCm2Builder,
            "iceh_m.0498-09.nc",
            "time",
            "d2:2.nc:5.ni:1440.nj:1080.nkbio:2",
        ),
        (
            builders.AccessCm2Builder,
            "ocean-2d-surface_salt-1-daily-mean-ym_0409_01.nc",
            "time",
            "nv:2.xt_ocean:1440.yt_ocean:1080",
        ),
        (
            builders.AccessCm2Builder,
            "iceh_d.1972-04.nc",
            "time",
            "d2:2.nc:5.ni:360.nj:300.nkbio:2",
        ),
        (
            builders.AccessCm2Builder,
            "ocean_daily.nc-19960630",
            "time",
            "nv:2.xt_ocean:360.yt_ocean:300",
        ),
        # Example ACCESS-ESM1.5 filenames
        (
            builders.AccessEsm15Builder,
            "PI-GWL-B2035.pe-109904_dai.nc",
            "time",
            "bnds:2.lat:145.lat_v:144.lon:192.lon_u:192.pseudo_level:17.soil_model_level_number:6",
        ),
        (
            builders.AccessEsm15Builder,
            "PI-GWL-B2035.pa-109904_mon.nc",
            "time",
            "bnds:2.lat:145.lat_v:144.lon:192.lon_u:192.model_rho_level_number:38.model_theta_level_number:38.pressure:19.pseudo_level:6.pseudo_level_0:17.pseudo_level_2:13.soil_model_level_number:6.time_0:1",
        ),
        (
            builders.AccessEsm15Builder,
            "PI-1pct-02.pe-011802_dai.nc_dai.nc",
            "time",
            "bnds:2.lat:145.lat_v:144.lon:192.lon_u:192.pressure:8.pseudo_level:17.soil_model_level_number:6",
        ),
        (
            builders.AccessEsm15Builder,
            "iceh.1917-05.nc",
            "time",
            "d2:2.nc:5.ni:360.nj:300.nkice:1.nksnow:1",
        ),
        # Example ACCESS-OM2 filenames
        (
            builders.AccessOm2Builder,
            "ocean_month.nc",
            "time",
            "grid_xt_ocean:360.grid_xu_ocean:360.grid_yt_ocean:300.grid_yu_ocean:300.neutral:80.neutralrho_edges:81.nv:2.potrho:80.potrho_edges:81.st_edges_ocean:51.st_ocean:50.sw_edges_ocean:51.sw_ocean:50.xt_ocean:360.xu_ocean:360.yt_ocean:300.yu_ocean:300",
        ),
        (
            builders.AccessOm2Builder,
            "ocean_daily_3d_pot_rho_1.nc",
            "time",
            "nv:2.st_edges_ocean:76.st_ocean:75.xt_ocean:3600.yt_ocean:2700",
        ),
        (
            builders.AccessOm2Builder,
            "ocean_daily_3d_vhrho_nt_07.nc",
            "time",
            "nv:2.st_ocean:75.xt_ocean:3600.yt_ocean:900",
        ),
        (
            builders.AccessOm2Builder,
            "ocean-3d-v-1-monthly-pow02-ym_1958_04.nc",
            "time",
            "nv:2.st_edges_ocean:76.st_ocean:75.xu_ocean:3600.yu_ocean:2700",
        ),
        (
            builders.AccessOm2Builder,
            "ocean-2d-sfc_salt_flux_restore-1-monthly-mean-ym_1958_04.nc",
            "time",
            "nv:2.xt_ocean:3600.yt_ocean:2700",
        ),
        (
            builders.AccessOm2Builder,
            "ocean-2d-sea_level-540-seconds-snap-ym_2022_04_01.nc",
            "time",
            "xt_ocean:3600.yt_ocean:2700",
        ),
        (
            builders.AccessOm2Builder,
            "ocean-3d-salt-1-daily-mean-ym_2018_10_jmax511_sigfig4.nc",
            "time",
            "nv:2.st_ocean:75.xt_ocean:3600.yt_ocean:511",
        ),
        (
            builders.AccessOm2Builder,
            "oceanbgc-3d-caco3-1-yearly-mean-y_2015.nc",
            "time",
            "nv:2.st_edges_ocean:51.st_ocean:50.xt_ocean:360.yt_ocean:300",
        ),
        (
            builders.AccessOm2Builder,
            "oceanbgc-2d-wdet100-1-daily-mean-y_2015.nc",
            "time",
            "nv:2.xt_ocean:360.yt_ocean:300",
        ),
        (
            builders.AccessOm2Builder,
            "oceanbgc-3d-phy-1-daily-mean-3-sigfig-5-daily-ymd_2020_12_01.nc",
            "time",
            "st_ocean:75.xt_ocean:3600.yt_ocean:2700",
        ),
        (
            builders.AccessOm2Builder,
            "rregionPrydz_temp_xflux_adv.nc",
            "time",
            "nv:2.st_edges_ocean:76.st_ocean:75.xu_ocean_sub02:351.yt_ocean_sub02:119",
        ),
        # Example ACCESS-OM3 filenames
        # No live OM3 data - test data cannot be opened by xarray
        # MOM6
        (
            builders.Mom6Builder,
            "19000101.ice_daily.nc",
            "time",
            "nv:2.xT:1440.xTe:1441.yT:1080.yTe:1081",
        ),
        (
            builders.Mom6Builder,
            "19000101.ice_month.nc",
            "time",
            "nv:2.xT:1440.xTe:1441.yT:1080.yTe:1081",
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_annual.nc",
            "time",
            "nv:2.xh:1440.xq:1441.yh:1080.yq:1081.zi:76.zl:75",
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_month_rho2.nc",
            "time",
            "nv:2.rho2_i:36.rho2_l:35.xh:1440.yh:1080.yq:1081",
        ),
        (
            builders.ROMSBuilder,
            "roms_his_0016.nc",
            "ocean_time",
            "boundary:4.eta_psi:309.eta_rho:310.eta_u:310.eta_v:309.s_rho:31.s_w:32.tracer:2.xi_psi:294.xi_rho:295.xi_u:294.xi_v:295",
        ),
        (
            builders.WoaBuilder,
            "woa13_ts_01_mom01.nc",
            "time",
            "GRID_X_T:2.GRID_Y_T:2.ZT:2",
        ),
    ],
)
def test_generate_file_shape_info(builder, filename, time_dim, expected):
    filepath = Path(__file__).parent / "data" / "size_info_test_data" / filename
    ds = xr.open_dataset(
        filepath, chunks={}, decode_cf=False, decode_times=False, decode_coords=False
    )
    assert builder._generate_file_shape_info(ds, time_dim=time_dim) == expected


@pytest.mark.parametrize(
    "builder, filename, expected",
    [
        # Example ACCESS-CM2 filenames
        (builders.AccessCm2Builder, "bz687a.pm107912_mon", (1, "mon")),
        (builders.AccessCm2Builder, "bz687a.p7107912_mon", (1, "mon")),
        (builders.AccessCm2Builder, "bz687a.p7107912_dai", (1, "day")),
        (builders.AccessCm2Builder, "iceh_m.2014-06", None),
        (builders.AccessCm2Builder, "iceh.1917-05-daily", (1, "day")),
        (builders.AccessCm2Builder, "iceh_03h.2016-01-3hourly", None),
        (builders.AccessCm2Builder, "ocean_bgc_ann", (1, "yr")),
        (builders.AccessCm2Builder, "ocean_daily", (1, "day")),
        # Example ACCESS-ESM1.5 filenames
        (builders.AccessEsm15Builder, "PI-GWL-B2035.pe-109904_dai", (1, "day")),
        (builders.AccessEsm15Builder, "PI-GWL-B2035.pa-109904_mon", (1, "mon")),
        (builders.AccessEsm15Builder, "PI-1pct-02.pe-011802_dai.nc_dai", (1, "day")),
        (builders.AccessEsm15Builder, "iceh.1917-05", None),
        # Example ACCESS-OM2 filenames
        (builders.AccessOm2Builder, "iceh.057-daily", (1, "day")),
        (builders.AccessOm2Builder, "iceh.1958-02-daily", (1, "day")),
        (builders.AccessOm2Builder, "iceh.1985-08-31", None),
        (builders.AccessOm2Builder, "ocean", None),
        (builders.AccessOm2Builder, "ocean_month", (1, "mon")),
        (builders.AccessOm2Builder, "ocean-2d-area_t", None),
        (builders.AccessOm2Builder, "ocean_daily_3d_pot_rho_1", (1, "day")),
        (builders.AccessOm2Builder, "ocean_daily_3d_vhrho_nt_07", (1, "day")),
        (
            builders.AccessOm2Builder,
            "ocean-3d-v-1-monthly-pow02-ym_1958_04",
            (1, "mon"),
        ),
        (
            builders.AccessOm2Builder,
            "ocean-2d-sfc_salt_flux_restore-1-monthly-mean-ym_1958_04",
            (1, "mon"),
        ),
        (
            builders.AccessOm2Builder,
            "ocean-2d-sea_level-540-seconds-snap-ym_2022_04_01",
            None,
        ),
        (
            builders.AccessOm2Builder,
            "ocean-3d-salt-1-daily-mean-ym_2018_10_jmax511_sigfig4",
            (1, "day"),
        ),
        (
            builders.AccessOm2Builder,
            "oceanbgc-3d-caco3-1-yearly-mean-y_2015",
            (1, "yr"),
        ),
        (
            builders.AccessOm2Builder,
            "oceanbgc-2d-wdet100-1-daily-mean-y_2015",
            (1, "day"),
        ),
        (
            builders.AccessOm2Builder,
            "oceanbgc-3d-phy-1-daily-mean-3-sigfig-5-daily-ymd_2020_12_01",
            (1, "day"),
        ),
        (builders.AccessOm2Builder, "rregionPrydz_temp_xflux_adv", None),
        # Example ACCESS-OM3 filenames
        (builders.AccessOm3Builder, "access-om3.ww3.hi.1958-01-02-00000", None),
        (builders.AccessOm3Builder, "access-om3.cice.h.1900-01-01", None),
        (builders.AccessOm3Builder, "access-om3.cice.h.1900-01", None),
        (builders.AccessOm3Builder, "access-om3.cice.h.1900-01-daily", (1, "day")),
        (builders.AccessOm3Builder, "access-om3.mom6.ocean_sfc_1900_01_01", None),
        (builders.AccessOm3Builder, "access-om3.mom6.sfc_1900_01", None),
        (builders.AccessOm3Builder, "access-om3.mom6.sfc_1900", None),
        (builders.AccessOm3Builder, "access-om3.mom6.static", None),
        (builders.AccessOm3Builder, "access-om3.mom6.static", None),
        (builders.AccessOm3Builder, "access-om3.mom6.3d.uh.1mon.mean.1900", (1, "mon")),
        (
            builders.AccessOm3Builder,
            "access-om3.mom6.3d.uh.1mon.mean.1900-01-01-00000",
            (1, "mon"),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3.mom6.3d.uh.1mon.mean.1900-01",
            (1, "mon"),
        ),
        (builders.AccessOm3Builder, "GMOM_JRA_WD.ww3.hi.1900-01-03-00000", None),
        (builders.AccessOm3Builder, "GMOM_JRA_WD.ww3.hi.1900", None),
        (builders.AccessOm3Builder, "GMOM_JRA_WD.ww3.hi.1900-01", None),
        (builders.AccessOm3Builder, "access-om3.mom6.3d.uh.1mon.mean.1900", (1, "mon")),
        (
            builders.AccessOm3Builder,
            "access-om3.mom6.3d.uh.1mon.mean.1900-01-01-00000",
            (1, "mon"),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3.mom6.3d.uh.1mon.mean.1900-01",
            (1, "mon"),
        ),
        (
            builders.AccessOm3Builder,
            "GMOM_JRA_WD.ww3.hi.1900-01-03-00000",
            None,
        ),
        (
            builders.AccessOm3Builder,
            "GMOM_JRA_WD.ww3.hi.1900",
            None,
        ),
        (
            builders.AccessOm3Builder,
            "GMOM_JRA_WD.ww3.hi.1900-01",
            None,
        ),
        # MOM6
        (
            builders.Mom6Builder,
            "19000101.ice_daily",
            (1, "day"),
        ),
        (
            builders.Mom6Builder,
            "19010101.ice_month",
            (1, "mon"),
        ),
        (
            builders.Mom6Builder,
            "19010101.ocean_annual_rho2",
            (1, "yr"),
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_annual_z",
            (1, "yr"),
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_annual",
            (1, "yr"),
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_daily",
            (1, "day"),
        ),
        (
            builders.Mom6Builder,
            "19010101.ocean_month_rho2",
            (1, "mon"),
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_month_z",
            (1, "mon"),
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_month",
            (1, "mon"),
        ),
        (
            builders.Mom6Builder,
            "19000101.ocean_scalar_annual",
            (1, "yr"),
        ),
        (
            builders.Mom6Builder,
            "19010101.ocean_scalar_month",
            (1, "mon"),
        ),
        (
            builders.Mom6Builder,
            "19010101.ocean_static",
            None,
        ),
        (
            builders.Mom6Builder,
            "20000201.ocean_daily_2000_032",
            (1, "day"),
        ),
        (
            builders.Mom6Builder,
            "20000201.ocean_daily_rho2_2000_056",
            (1, "day"),
        ),
        (
            builders.Mom6Builder,
            "20000201.ocean_daily_z_2000_119",
            (1, "day"),
        ),
        (
            builders.ROMSBuilder,
            "roms_his_0016",
            None,
        ),
        (
            builders.WoaBuilder,
            "woa13_ts_01_mom01",
            None,
        ),
    ],
)
def test_parse_filename(builder, filename, expected):
    assert builder.parse_filename_freq(filename) == expected


@pytest.mark.parametrize(
    "compare_files",
    [
        (True),
        (False),
    ],
)
@pytest.mark.parametrize(
    "builder, filename, expected",
    [
        (
            builders.AccessOm2Builder,
            "access-om2/output000/ocean/ocean_grid.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_grid.nc",
                file_id="xt_ocean:1.yt_ocean:1",
                frequency="fx",
                start_date="none",
                end_date="none",
                variable=["geolat_t", "geolon_t", "xt_ocean", "yt_ocean"],
                variable_long_name=[
                    "tracer latitude",
                    "tracer longitude",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=["", "", "", ""],
                variable_cell_methods=["time: point", "time: point", "", ""],
                variable_units=["degrees_N", "degrees_E", "degrees_E", "degrees_N"],
            ),
        ),
        (
            builders.AccessOm2Builder,
            "access-om2/output000/ocean/ocean.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean.nc",
                file_id="nv:2.st_ocean:1.xt_ocean:1.yt_ocean:1",
                frequency="1yr",
                start_date="1900-01-01, 00:00:00",
                end_date="1910-01-01, 00:00:00",
                variable=[
                    "nv",
                    "st_ocean",
                    "temp",
                    "time",
                    "time_bounds",
                    "xt_ocean",
                    "yt_ocean",
                ],
                variable_long_name=[
                    "vertex number",
                    "tcell zstar depth",
                    "Conservative temperature",
                    "time",
                    "time axis boundaries",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "sea_water_conservative_temperature",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=["", "", "time: mean", "", "", "", ""],
                variable_units=[
                    "none",
                    "meters",
                    "K",
                    "days since 1900-01-01 00:00:00",
                    "days",
                    "degrees_E",
                    "degrees_N",
                ],
            ),
        ),
        (
            builders.AccessOm2Builder,
            "access-om2/output000/ocean/ocean_month.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_month.nc",
                file_id="nv:2.xt_ocean:1.yt_ocean:1",
                frequency="1mon",
                start_date="1900-01-01, 00:00:00",
                end_date="1910-01-01, 00:00:00",
                variable=["mld", "nv", "time", "time_bounds", "xt_ocean", "yt_ocean"],
                variable_long_name=[
                    "mixed layer depth determined by density criteria",
                    "vertex number",
                    "time",
                    "time axis boundaries",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=[
                    "ocean_mixed_layer_thickness_defined_by_sigma_t",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=["time: mean", "", "", "", "", ""],
                variable_units=[
                    "m",
                    "none",
                    "days since 1900-01-01 00:00:00",
                    "days",
                    "degrees_E",
                    "degrees_N",
                ],
            ),
        ),
        (
            builders.AccessOm2Builder,
            "access-om2/output000/ocean/ocean_month_inst_nobounds.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_month_inst_nobounds.nc",
                file_id="xt_ocean:1.yt_ocean:1",
                frequency="1mon",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-02-01, 00:00:00",
                variable=["mld", "time", "xt_ocean", "yt_ocean"],
                variable_long_name=[
                    "mixed layer depth determined by density criteria",
                    "time",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=[
                    "ocean_mixed_layer_thickness_defined_by_sigma_t",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=["time: mean", "", "", ""],
                variable_units=[
                    "m",
                    "days since 1900-01-01 00:00:00",
                    "degrees_E",
                    "degrees_N",
                ],
            ),
        ),
        (
            builders.AccessOm2Builder,
            "access-om2/output000/ice/OUTPUT/iceh.1900-01.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="iceh.1900-01.nc",
                file_id="d2:2.ni:1.nj:1",
                frequency="1mon",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-02-01, 00:00:00",
                variable=["TLAT", "TLON", "aice_m", "tarea", "time", "time_bounds"],
                variable_long_name=[
                    "T grid center latitude",
                    "T grid center longitude",
                    "ice area  (aggregate)",
                    "area of T grid cells",
                    "model time",
                    "boundaries for time-averaging interval",
                ],
                variable_standard_name=["", "", "", "", "", ""],
                variable_cell_methods=["", "", "time: mean", "", "", ""],
                variable_units=[
                    "degrees_north",
                    "degrees_east",
                    "1",
                    "m^2",
                    "days since 1900-01-01 00:00:00",
                    "days since 1900-01-01 00:00:00",
                ],
            ),
        ),
        (
            builders.AccessCm2Builder,
            "access-cm2/by578/history/atm/netCDF/by578a.pd201501_dai.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="by578a.pd201501_dai.nc",
                file_id="bnds:2.lat:1.lon:1",
                frequency="1day",
                start_date="2015-01-01, 00:00:00",
                end_date="2015-02-01, 00:00:00",
                variable=["fld_s03i236"],
                variable_long_name=["TEMPERATURE AT 1.5M"],
                variable_standard_name=["air_temperature"],
                variable_cell_methods=["time: mean"],
                variable_units=["K"],
            ),
        ),
        (
            builders.AccessCm2Builder,
            "access-cm2/by578/history/ice/iceh_d.2015-01.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="iceh_d.2015-01.nc",
                file_id="d2:2.ni:1.nj:1",
                frequency="1day",
                start_date="2015-01-01, 00:00:00",
                end_date="2015-02-01, 00:00:00",
                variable=["TLAT", "TLON", "aice", "tarea", "time", "time_bounds"],
                variable_long_name=[
                    "T grid center latitude",
                    "T grid center longitude",
                    "ice area  (aggregate)",
                    "area of T grid cells",
                    "model time",
                    "boundaries for time-averaging interval",
                ],
                variable_standard_name=["", "", "", "", "", ""],
                variable_cell_methods=["", "", "time: mean", "", "", ""],
                variable_units=[
                    "degrees_north",
                    "degrees_east",
                    "1",
                    "m^2",
                    "days since 1850-01-01 00:00:00",
                    "days since 1850-01-01 00:00:00",
                ],
            ),
        ),
        (
            builders.AccessCm2Builder,
            "access-cm2/by578/history/ocn/ocean_daily.nc-20150630",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_daily.nc-20150630",
                file_id="nv:2.xt_ocean:1.yt_ocean:1",
                frequency="1day",
                start_date="2015-01-01, 00:00:00",
                end_date="2015-07-01, 00:00:00",
                variable=["nv", "sst", "time", "time_bounds", "xt_ocean", "yt_ocean"],
                variable_long_name=[
                    "vertex number",
                    "Potential temperature",
                    "time",
                    "time axis boundaries",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=["", "sea_surface_temperature", "", "", "", ""],
                variable_cell_methods=["", "time: mean", "", "", "", ""],
                variable_units=[
                    "none",
                    "K",
                    "days since 1850-01-01 00:00:00",
                    "days",
                    "degrees_E",
                    "degrees_N",
                ],
            ),
        ),
        (
            builders.AccessCm2Builder,
            "access-cm2/by578/history/ocn/ocean_scalar.nc-20150630",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_scalar.nc-20150630",
                file_id="nv:2.scalar_axis:1",
                frequency="1mon",
                start_date="2015-01-01, 00:00:00",
                end_date="2015-07-01, 00:00:00",
                variable=[
                    "nv",
                    "scalar_axis",
                    "temp_global_ave",
                    "time",
                    "time_bounds",
                ],
                variable_long_name=[
                    "vertex number",
                    "none",
                    "Global mean temp in liquid seawater",
                    "time",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "sea_water_potential_temperature",
                    "",
                    "",
                ],
                variable_cell_methods=["", "", "time: mean", "", ""],
                variable_units=[
                    "none",
                    "none",
                    "deg_C",
                    "days since 1850-01-01 00:00:00",
                    "days",
                ],
            ),
        ),
        (
            builders.AccessEsm15Builder,
            "access-esm1-5/history/atm/netCDF/HI-C-05-r1.pa-185001_mon.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="HI-C-05-r1.pa-185001_mon.nc",
                file_id="bnds:2.lat:1.lon:1",
                frequency="1mon",
                start_date="1850-01-01, 00:00:00",
                end_date="1850-02-01, 00:00:00",
                variable=["fld_s03i236"],
                variable_long_name=["TEMPERATURE AT 1.5M"],
                variable_standard_name=["air_temperature"],
                variable_cell_methods=["time: mean"],
                variable_units=["K"],
            ),
        ),
        (
            builders.AccessEsm15Builder,
            "access-esm1-5/history/ice/iceh.1850-01.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="iceh.1850-01.nc",
                file_id="d2:2.ni:1.nj:1",
                frequency="1mon",
                start_date="1850-01-01, 00:00:00",
                end_date="1850-02-01, 00:00:00",
                variable=["TLAT", "TLON", "aice", "tarea", "time", "time_bounds"],
                variable_long_name=[
                    "T grid center latitude",
                    "T grid center longitude",
                    "ice area  (aggregate)",
                    "area of T grid cells",
                    "model time",
                    "boundaries for time-averaging interval",
                ],
                variable_standard_name=["", "", "", "", "", ""],
                variable_cell_methods=["", "", "time: mean", "", "", ""],
                variable_units=[
                    "degrees_north",
                    "degrees_east",
                    "1",
                    "m^2",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                ],
            ),
        ),
        (
            builders.AccessEsm15Builder,
            "access-esm1-5/history/ocn/ocean_bgc_ann.nc-18501231",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_bgc_ann.nc-18501231",
                file_id="nv:2.xt_ocean:1.yt_ocean:1",
                frequency="1yr",
                start_date="1849-12-30, 00:00:00",
                end_date="1850-12-30, 00:00:00",
                variable=[
                    "fgco2_raw",
                    "nv",
                    "time",
                    "time_bounds",
                    "xt_ocean",
                    "yt_ocean",
                ],
                variable_long_name=[
                    "Flux into ocean - DIC, inc. anth.",
                    "vertex number",
                    "time",
                    "time axis boundaries",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=["", "", "", "", "", ""],
                variable_cell_methods=["time: mean", "", "", "", "", ""],
                variable_units=[
                    "mmol/m^2/s",
                    "none",
                    "days since 0001-01-01 00:00:00",
                    "days",
                    "degrees_E",
                    "degrees_N",
                ],
            ),
        ),
        (
            builders.AccessEsm15Builder,
            "access-esm1-5/history/ocn/ocean_bgc.nc-18501231",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="ocean_bgc.nc-18501231",
                file_id="nv:2.st_ocean:50.xt_ocean:1.yt_ocean:1",
                frequency="1mon",
                start_date="1849-12-30, 00:00:00",
                end_date="1850-12-30, 00:00:00",
                variable=[
                    "nv",
                    "o2",
                    "st_ocean",
                    "time",
                    "time_bounds",
                    "xt_ocean",
                    "yt_ocean",
                ],
                variable_long_name=[
                    "vertex number",
                    "o2",
                    "tcell zstar depth",
                    "time",
                    "time axis boundaries",
                    "tcell longitude",
                    "tcell latitude",
                ],
                variable_standard_name=["", "", "", "", "", "", ""],
                variable_cell_methods=["", "time: mean", "", "", "", "", ""],
                variable_units=[
                    "none",
                    "mmol/m^3",
                    "meters",
                    "days since 0001-01-01 00:00:00",
                    "days",
                    "degrees_E",
                    "degrees_N",
                ],
            ),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3/output000/GMOM_JRA_WD.mom6.h.native_1900_01.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="GMOM_JRA_WD.mom6.h.native_1900_01.nc",
                file_id="nv:2.xh:1.yh:1.zl:1",
                frequency="1mon",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-02-01, 00:00:00",
                variable=[
                    "average_DT",
                    "average_T1",
                    "average_T2",
                    "nv",
                    "thetao",
                    "time",
                    "time_bnds",
                    "xh",
                    "yh",
                    "zl",
                ],
                variable_long_name=[
                    "Length of average period",
                    "Start time for average period",
                    "End time for average period",
                    "vertex number",
                    "Sea Water Potential Temperature",
                    "time",
                    "time axis boundaries",
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Layer pseudo-depth, -z*",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "sea_water_potential_temperature",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "area:mean zl:mean yh:mean xh:mean time: mean",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "days",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                    "",
                    "degC",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                    "degrees_east",
                    "degrees_north",
                    "meter",
                ],
            ),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3/output000/GMOM_JRA_WD.mom6.h.sfc_1900_01_02.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="GMOM_JRA_WD.mom6.h.sfc_1900_01_02.nc",
                file_id="nv:2.xh:1.yh:1",
                frequency="1day",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-01-02, 00:00:00",
                variable=[
                    "average_DT",
                    "average_T1",
                    "average_T2",
                    "nv",
                    "time",
                    "time_bnds",
                    "tos",
                    "xh",
                    "yh",
                ],
                variable_long_name=[
                    "Length of average period",
                    "Start time for average period",
                    "End time for average period",
                    "vertex number",
                    "time",
                    "time axis boundaries",
                    "Sea Surface Temperature",
                    "h point nominal longitude",
                    "h point nominal latitude",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "sea_surface_temperature",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:mean yh:mean xh:mean time: mean",
                    "",
                    "",
                ],
                variable_units=[
                    "days",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                    "",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                    "degC",
                    "degrees_east",
                    "degrees_north",
                ],
            ),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3/output000/GMOM_JRA_WD.mom6.h.static.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="GMOM_JRA_WD.mom6.h.static.nc",
                file_id="xh:1.yh:1",
                frequency="fx",
                start_date="none",
                end_date="none",
                variable=["geolat", "geolon", "xh", "yh"],
                variable_long_name=[
                    "Latitude of tracer (T) points",
                    "Longitude of tracer (T) points",
                    "h point nominal longitude",
                    "h point nominal latitude",
                ],
                variable_standard_name=["", "", "", ""],
                variable_cell_methods=["time: point", "time: point", "", ""],
                variable_units=[
                    "degrees_north",
                    "degrees_east",
                    "degrees_east",
                    "degrees_north",
                ],
            ),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3/output000/GMOM_JRA_WD.mom6.h.z_1900_01.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="GMOM_JRA_WD.mom6.h.z_1900_01.nc",
                file_id="nv:2.xh:1.yh:1.z_l:1",
                frequency="1mon",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-02-01, 00:00:00",
                variable=[
                    "average_DT",
                    "average_T1",
                    "average_T2",
                    "nv",
                    "thetao",
                    "time",
                    "time_bnds",
                    "xh",
                    "yh",
                    "z_l",
                ],
                variable_long_name=[
                    "Length of average period",
                    "Start time for average period",
                    "End time for average period",
                    "vertex number",
                    "Sea Water Potential Temperature",
                    "time",
                    "time axis boundaries",
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Depth at cell center",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "sea_water_potential_temperature",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "area:mean z_l:mean yh:mean xh:mean time: mean",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "days",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                    "",
                    "degC",
                    "days since 0001-01-01 00:00:00",
                    "days since 0001-01-01 00:00:00",
                    "degrees_east",
                    "degrees_north",
                    "meters",
                ],
            ),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3/output000/GMOM_JRA_WD.cice.h.1900-01-01.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="GMOM_JRA_WD.cice.h.1900-01-01.nc",
                file_id="nbnd:2.ni:1.nj:1",
                frequency="1day",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-01-02, 00:00:00",
                variable=["TLAT", "TLON", "aice", "tarea", "time", "time_bounds"],
                variable_long_name=[
                    "T grid center latitude",
                    "T grid center longitude",
                    "ice area  (aggregate)",
                    "area of T grid cells",
                    "time",
                    "time interval endpoints",
                ],
                variable_standard_name=["", "", "", "", "", ""],
                variable_cell_methods=["", "", "time: mean", "", "", ""],
                variable_units=[
                    "degrees_north",
                    "degrees_east",
                    "1",
                    "m^2",
                    "days since 0000-01-01 00:00:00",
                    "days since 0000-01-01 00:00:00",
                ],
            ),
        ),
        (
            builders.AccessOm3Builder,
            "access-om3/output000/GMOM_JRA_WD.ww3.hi.1900-01-02-00000.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="GMOM_JRA_WD.ww3.hi.1900-01-02-00000.nc",
                file_id="freq:1.nx:1.ny:1",
                frequency="fx",  # WW3 provides no time bounds
                start_date="1900-01-02, 00:00:00",
                end_date="1900-01-02, 00:00:00",
                variable=["EF", "mapsta"],
                variable_long_name=["1D spectral density", "map status"],
                variable_standard_name=["", ""],
                variable_cell_methods=["", ""],
                variable_units=["m2 s", "unitless"],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output000/19000101.ice_daily.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="19000101.ice_daily.nc",
                file_id="nv:2.xT:1.xTe:1.yT:1.yTe:1",
                frequency="1day",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-01-02, 00:00:00",
                variable=[
                    "xT",
                    "xTe",
                    "yT",
                    "yTe",
                    "time",
                    "nv",
                    "siconc",
                    "sithick",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "T point nominal longitude",
                    "T-cell edge nominal longitude",
                    "T point nominal latitude",
                    "T-cell edge nominal latitude",
                    "time",
                    "vertex number",
                    "ice concentration",
                    "ice thickness",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "time: mean",
                    "time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "degrees_E",
                    "degrees_E",
                    "degrees_N",
                    "degrees_N",
                    "days since 1900-01-01",
                    "",
                    "0-1",
                    "m-ice",
                    "days since 1900-01-01",
                    "days since 1900-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output000/19000101.ocean_annual_z.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="19000101.ocean_annual_z.nc",
                file_id="nv:2.xh:1.xq:1.yh:1.yq:1.z_i:1.z_l:1",
                frequency="1yr",
                start_date="1900-01-01, 00:00:00",
                end_date="1901-01-01, 00:00:00",
                variable=[
                    "xh",
                    "yh",
                    "z_l",
                    "z_i",
                    "time",
                    "nv",
                    "xq",
                    "yq",
                    "volcello",
                    "thetao",
                    "thetao_xyave",
                    "so",
                    "so_xyave",
                    "agessc",
                    "uo",
                    "vo",
                    "umo",
                    "vmo",
                    "uh",
                    "vh",
                    "T_adx",
                    "T_ady",
                    "S_adx",
                    "S_ady",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Depth at cell center",
                    "Depth at interface",
                    "time",
                    "vertex number",
                    "q point nominal longitude",
                    "q point nominal latitude",
                    "Ocean grid-cell volume",
                    "Sea Water Potential Temperature",
                    "Sea Water Potential Temperature",
                    "Sea Water Salinity",
                    "Sea Water Salinity",
                    "Ideal Age Tracer",
                    "Sea Water X Velocity",
                    "Sea Water Y Velocity",
                    "Ocean Mass X Transport",
                    "Ocean Mass Y Transport",
                    "Zonal Thickness Flux",
                    "Meridional Thickness Flux",
                    "Advective (by residual mean) Zonal Flux of Heat",
                    "Advective (by residual mean) Meridional Flux of Heat",
                    "Advective (by residual mean) Zonal Flux of Salt",
                    "Advective (by residual mean) Meridional Flux of Salt",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_volume",
                    "sea_water_potential_temperature",
                    "sea_water_potential_temperature",
                    "sea_water_salinity",
                    "sea_water_salinity",
                    "ideal_age_tracer",
                    "sea_water_x_velocity",
                    "sea_water_y_velocity",
                    "ocean_mass_x_transport",
                    "ocean_mass_y_transport",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:sum z_l:sum yh:sum xh:sum time: mean",
                    "area:mean z_l:mean yh:mean xh:mean time: mean",
                    "z_l:mean time: mean",
                    "area:mean z_l:mean yh:mean xh:mean time: mean",
                    "z_l:mean time: mean",
                    "area:mean z_l:mean yh:mean xh:mean time: mean",
                    "z_l:mean yh:mean xq:point time: mean",
                    "z_l:mean yq:point xh:mean time: mean",
                    "z_l:sum yh:sum xq:point time: mean",
                    "z_l:sum yq:point xh:sum time: mean",
                    "z_l:sum yh:sum xq:point time: mean",
                    "z_l:sum yq:point xh:sum time: mean",
                    "z_l:sum yh:sum xq:point time: mean",
                    "z_l:sum yq:point xh:sum time: mean",
                    "z_l:sum yh:sum xq:point time: mean",
                    "z_l:sum yq:point xh:sum time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "degrees_east",
                    "degrees_north",
                    "meters",
                    "meters",
                    "days since 1900-01-01",
                    "",
                    "degrees_east",
                    "degrees_north",
                    "m3",
                    "degC",
                    "degC",
                    "psu",
                    "psu",
                    "yr",
                    "m s-1",
                    "m s-1",
                    "kg s-1",
                    "kg s-1",
                    "m3 s-1",
                    "m3 s-1",
                    "W",
                    "W",
                    "psu m3 s-1",
                    "psu m3 s-1",
                    "days since 1900-01-01",
                    "days since 1900-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output000/19000101.ocean_month_rho2.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="19000101.ocean_month_rho2.nc",
                file_id="nv:2.rho2_i:1.rho2_l:1.xh:1.yh:1.yq:1",
                frequency="1mon",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-02-01, 00:00:00",
                variable=[
                    "xh",
                    "yh",
                    "rho2_l",
                    "rho2_i",
                    "time",
                    "nv",
                    "yq",
                    "volcello",
                    "thkcello",
                    "vmo",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Target Potential Density at cell center",
                    "Target Potential Density at interface",
                    "time",
                    "vertex number",
                    "q point nominal latitude",
                    "Ocean grid-cell volume",
                    "Cell Thickness",
                    "Ocean Mass Y Transport",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_volume",
                    "cell_thickness",
                    "ocean_mass_y_transport",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:sum rho2_l:sum yh:sum xh:sum time: mean",
                    "area:mean rho2_l:sum yh:mean xh:mean time: mean",
                    "rho2_l:sum yq:point xh:sum time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "degrees_east",
                    "degrees_north",
                    "kg m-3",
                    "kg m-3",
                    "days since 1900-01-01",
                    "",
                    "degrees_north",
                    "m3",
                    "m",
                    "kg s-1",
                    "days since 1900-01-01",
                    "days since 1900-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output000/19000101.ocean_scalar_annual.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="19000101.ocean_scalar_annual.nc",
                file_id="nv:2.scalar_axis:1",
                frequency="1yr",
                start_date="1900-01-01, 00:00:00",
                end_date="1901-01-01, 00:00:00",
                variable=[
                    "scalar_axis",
                    "time",
                    "nv",
                    "masso",
                    "volo",
                    "thetaoga",
                    "soga",
                    "sosga",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "none",
                    "time",
                    "vertex number",
                    "Mass of liquid ocean",
                    "Total volume of liquid ocean",
                    "Global Mean Ocean Potential Temperature",
                    "Global Mean Ocean Salinity",
                    "Sea Surface Salinity",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "sea_water_mass",
                    "sea_water_volume",
                    "sea_water_potential_temperature",
                    "sea_water_salinity",
                    "sea_surface_salinity",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "time: mean",
                    "time: mean",
                    "time: mean",
                    "time: mean",
                    "time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "",
                    "days since 1900-01-01",
                    "",
                    "kg",
                    "m3",
                    "degC",
                    "psu",
                    "psu",
                    "days since 1900-01-01",
                    "days since 1900-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output000/19000101.ocean_static.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="19000101.ocean_static.nc",
                file_id="xh:1.xq:1.yh:1.yq:1",
                frequency="fx",
                start_date="1900-01-01, 00:00:00",
                end_date="1900-01-01, 00:00:00",
                variable=[
                    "xh",
                    "yh",
                    "time",
                    "xq",
                    "yq",
                    "areacello",
                    "deptho",
                    "hfgeou",
                    "sftof",
                    "Coriolis",
                    "geolon",
                    "geolat",
                    "geolon_c",
                    "geolat_c",
                    "geolon_u",
                    "geolat_u",
                    "geolon_v",
                    "geolat_v",
                    "wet",
                    "wet_c",
                    "wet_u",
                    "wet_v",
                    "dxt",
                    "dyt",
                    "dxCu",
                    "dyCu",
                    "dxCv",
                    "dyCv",
                    "areacello_cu",
                    "areacello_cv",
                    "areacello_bu",
                ],
                variable_long_name=[
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "time",
                    "q point nominal longitude",
                    "q point nominal latitude",
                    "Ocean Grid-Cell Area",
                    "Sea Floor Depth",
                    "Upward geothermal heat flux at sea floor",
                    "Sea Area Fraction",
                    "Coriolis parameter at corner (Bu) points",
                    "Longitude of tracer (T) points",
                    "Latitude of tracer (T) points",
                    "Longitude of corner (Bu) points",
                    "Latitude of corner (Bu) points",
                    "Longitude of zonal velocity (Cu) points",
                    "Latitude of zonal velocity (Cu) points",
                    "Longitude of meridional velocity (Cv) points",
                    "Latitude of meridional velocity (Cv) points",
                    "0 if land, 1 if ocean at tracer points",
                    "0 if land, 1 if ocean at corner (Bu) points",
                    "0 if land, 1 if ocean at zonal velocity (Cu) points",
                    "0 if land, 1 if ocean at meridional velocity (Cv) points",
                    "Delta(x) at thickness/tracer points (meter)",
                    "Delta(y) at thickness/tracer points (meter)",
                    "Delta(x) at u points (meter)",
                    "Delta(y) at u points (meter)",
                    "Delta(x) at v points (meter)",
                    "Delta(y) at v points (meter)",
                    "Ocean Grid-Cell Area",
                    "Ocean Grid-Cell Area",
                    "Ocean Grid-Cell Area",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "cell_area",
                    "sea_floor_depth_below_geoid",
                    "upward_geothermal_heat_flux_at_sea_floor",
                    "SeaAreaFraction",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "cell_area",
                    "cell_area",
                    "cell_area",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:sum yh:sum xh:sum time: point",
                    "area:mean yh:mean xh:mean time: point",
                    "area:mean yh:mean xh:mean time: point",
                    "area:mean yh:mean xh:mean time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "time: point",
                    "area:sum yh:sum xq:sum time: point",
                    "area:sum yq:sum xh:sum time: point",
                    "area:sum yq:sum xq:sum time: point",
                ],
                variable_units=[
                    "degrees_east",
                    "degrees_north",
                    "days since 1900-01-01",
                    "degrees_east",
                    "degrees_north",
                    "m2",
                    "m",
                    "W m-2",
                    "%",
                    "s-1",
                    "degrees_east",
                    "degrees_north",
                    "degrees_east",
                    "degrees_north",
                    "degrees_east",
                    "degrees_north",
                    "degrees_east",
                    "degrees_north",
                    "",
                    "",
                    "",
                    "",
                    "m",
                    "m",
                    "m",
                    "m",
                    "m",
                    "m",
                    "m2",
                    "m2",
                    "m2",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output053/20051101.ocean_daily_2005_360.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="20051101.ocean_daily_2005_360.nc",
                file_id="nv:2.xh:1.xq:1.yh:1.yq:1.zl:1",
                frequency="1day",
                start_date="2005-12-26, 00:00:00",
                end_date="2005-12-27, 00:00:00",
                variable=[
                    "xh",
                    "yh",
                    "zl",
                    "time",
                    "nv",
                    "xq",
                    "yq",
                    "volcello",
                    "zos",
                    "mlotst",
                    "tauuo",
                    "tauvo",
                    "hfds",
                    "wfo",
                    "fsitherm",
                    "salt_flux",
                    "T_adx_2d",
                    "T_ady_2d",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Layer pseudo-depth, -z*",
                    "time",
                    "vertex number",
                    "q point nominal longitude",
                    "q point nominal latitude",
                    "Ocean grid-cell volume",
                    "Sea surface height above geoid",
                    "Ocean Mixed Layer Thickness Defined by Sigma T",
                    "Surface Downward X Stress",
                    "Surface Downward Y Stress",
                    "Surface ocean heat flux from SW+LW+latent+sensible+masstransfer+frazil+seaice_melt_heat",
                    "Water Flux Into Sea Water",
                    "water flux to ocean from sea ice melt(> 0) or form(< 0)",
                    "Net salt flux into ocean at surface (restoring + sea-ice)",
                    "Advective (by residual mean) Zonal Flux of Heat",
                    "Advective (by residual mean) Meridional Flux of Heat",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_volume",
                    "sea_surface_height_above_geoid",
                    "ocean_mixed_layer_thickness_defined_by_sigma_t",
                    "surface_downward_x_stress",
                    "surface_downward_y_stress",
                    "surface_downward_heat_flux_in_sea_water",
                    "water_flux_into_sea_water",
                    "water_flux_into_sea_water_due_to_sea_ice_thermodynamics",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:sum zl:sum yh:sum xh:sum time: mean",
                    "area:mean yh:mean xh:mean time: mean",
                    "area:mean yh:mean xh:mean time: mean",
                    "yh:mean xq:point time: mean",
                    "yq:point xh:mean time: mean",
                    "area:mean yh:mean xh:mean time: mean",
                    "area:mean yh:mean xh:mean time: mean",
                    "area:mean yh:mean xh:mean time: mean",
                    "area:mean yh:mean xh:mean time: mean",
                    "zl:sum yh:sum xq:point time: mean",
                    "zl:sum yq:point xh:sum time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "degrees_east",
                    "degrees_north",
                    "meter",
                    "days since 1991-01-01",
                    "",
                    "degrees_east",
                    "degrees_north",
                    "m3",
                    "m",
                    "m",
                    "N m-2",
                    "N m-2",
                    "W m-2",
                    "kg m-2 s-1",
                    "kg m-2 s-1",
                    "kg m-2 s-1",
                    "W",
                    "W",
                    "days since 1991-01-01",
                    "days since 1991-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output053/20051101.ocean_daily_rho2_2005_360.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="20051101.ocean_daily_rho2_2005_360.nc",
                file_id="nv:2.rho2_i:1.rho2_l:1.xh:1.xq:1.yh:1.yq:1",
                frequency="1day",
                start_date="2005-12-26, 00:00:00",
                end_date="2005-12-27, 00:00:00",
                variable=[
                    "xh",
                    "yh",
                    "rho2_l",
                    "rho2_i",
                    "time",
                    "nv",
                    "xq",
                    "yq",
                    "volcello",
                    "umo",
                    "vmo",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Target Potential Density at cell center",
                    "Target Potential Density at interface",
                    "time",
                    "vertex number",
                    "q point nominal longitude",
                    "q point nominal latitude",
                    "Ocean grid-cell volume",
                    "Ocean Mass X Transport",
                    "Ocean Mass Y Transport",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_volume",
                    "ocean_mass_x_transport",
                    "ocean_mass_y_transport",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:sum rho2_l:sum yh:sum xh:sum time: mean",
                    "rho2_l:sum yh:sum xq:point time: mean",
                    "rho2_l:sum yq:point xh:sum time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "degrees_east",
                    "degrees_north",
                    "kg m-3",
                    "kg m-3",
                    "days since 1991-01-01",
                    "",
                    "degrees_east",
                    "degrees_north",
                    "m3",
                    "kg s-1",
                    "kg s-1",
                    "days since 1991-01-01",
                    "days since 1991-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.Mom6Builder,
            "mom6/output053/20051101.ocean_daily_z_2005_360.nc",
            _NCFileInfo(
                path=None,  # type: ignore
                filename="20051101.ocean_daily_z_2005_360.nc",
                file_id="nv:2.xh:1.xq:1.yh:1.yq:1.z_i:1.z_l:1.z_l_sub01:1",
                frequency="1day",
                start_date="2005-12-26, 00:00:00",
                end_date="2005-12-27, 00:00:00",
                variable=[
                    "xh",
                    "yh",
                    "z_l",
                    "z_i",
                    "time",
                    "nv",
                    "xq",
                    "z_l_sub01",
                    "yq",
                    "volcello",
                    "uo",
                    "vo",
                    "thetao",
                    "so",
                    "average_T1",
                    "average_T2",
                    "average_DT",
                    "time_bnds",
                ],
                variable_long_name=[
                    "h point nominal longitude",
                    "h point nominal latitude",
                    "Depth at cell center",
                    "Depth at interface",
                    "time",
                    "vertex number",
                    "q point nominal longitude",
                    "Depth at cell center",
                    "q point nominal latitude",
                    "Ocean grid-cell volume",
                    "Sea Water X Velocity",
                    "Sea Water Y Velocity",
                    "Sea Water Potential Temperature",
                    "Sea Water Salinity",
                    "Start time for average period",
                    "End time for average period",
                    "Length of average period",
                    "time axis boundaries",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_volume",
                    "sea_water_x_velocity",
                    "sea_water_y_velocity",
                    "sea_water_potential_temperature",
                    "sea_water_salinity",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "area:sum z_l:sum yh:sum xh:sum time: mean",
                    "z_l:mean yh:mean xq:point time: mean",
                    "z_l:mean yq:point xh:mean time: mean",
                    "area:mean z_l:mean yh:mean xh:mean time: mean",
                    "area:mean z_l:mean yh:mean xh:mean time: mean",
                    "",
                    "",
                    "",
                    "",
                ],
                variable_units=[
                    "degrees_east",
                    "degrees_north",
                    "meters",
                    "meters",
                    "days since 1991-01-01",
                    "",
                    "degrees_east",
                    "meters",
                    "degrees_north",
                    "m3",
                    "m s-1",
                    "m s-1",
                    "degC",
                    "psu",
                    "days since 1991-01-01",
                    "days since 1991-01-01",
                    "days",
                    "days",
                ],
            ),
        ),
        (
            builders.ROMSBuilder,
            "roms/roms_his_0016.nc",
            _NCFileInfo(
                filename="roms_his_0016.nc",
                file_id="boundary:1.eta_psi:1.eta_rho:1.eta_u:1.eta_v:1.ocean_time:73.s_rho:1.s_w:1.tracer:1.xi_psi:1.xi_rho:1.xi_u:1.xi_v:1",
                path=None,  # type: ignore
                frequency="fx",
                start_date="none",
                end_date="none",
                variable=[
                    "ntimes",
                    "ndtfast",
                    "dt",
                    "dtfast",
                    "dstart",
                    "nHIS",
                    "ndefHIS",
                    "nRST",
                    "ntsAVG",
                    "nAVG",
                    "ndefAVG",
                    "Falpha",
                    "Fbeta",
                    "Fgamma",
                    "nl_tnu2",
                    "nl_visc2",
                    "LuvSponge",
                    "LtracerSponge",
                    "Akt_bak",
                    "Akv_bak",
                    "rdrg",
                    "rdrg2",
                    "Zob",
                    "Zos",
                    "Znudg",
                    "M2nudg",
                    "M3nudg",
                    "Tnudg",
                    "FSobc_in",
                    "FSobc_out",
                    "M2obc_in",
                    "M2obc_out",
                    "Tobc_in",
                    "Tobc_out",
                    "M3obc_in",
                    "M3obc_out",
                    "rho0",
                    "gamma2",
                    "LuvSrc",
                    "LwSrc",
                    "LtracerSrc",
                    "LsshCLM",
                    "Lm2CLM",
                    "Lm3CLM",
                    "LtracerCLM",
                    "LnudgeM2CLM",
                    "LnudgeM3CLM",
                    "LnudgeTCLM",
                    "spherical",
                    "xl",
                    "el",
                    "Vtransform",
                    "Vstretching",
                    "theta_s",
                    "theta_b",
                    "Tcline",
                    "hc",
                    "s_rho",
                    "s_w",
                    "Cs_r",
                    "Cs_w",
                    "h",
                    "zice",
                    "f",
                    "pm",
                    "pn",
                    "x_rho",
                    "y_rho",
                    "x_u",
                    "y_u",
                    "x_v",
                    "y_v",
                    "x_psi",
                    "y_psi",
                    "angle",
                    "mask_rho",
                    "mask_u",
                    "mask_v",
                    "mask_psi",
                    "ocean_time",
                    "m",
                    "Tb",
                    "Sb",
                    "zeta",
                    "ubar",
                    "vbar",
                    "u",
                    "v",
                    "w",
                    "temp",
                    "salt",
                    "rho",
                    "Hsbl",
                    "AKv",
                    "AKt",
                    "shflux",
                    "ssflux",
                    "sustr",
                    "svstr",
                ],
                variable_long_name=[
                    "number of long time-steps",
                    "number of short time-steps",
                    "size of long time-steps",
                    "size of short time-steps",
                    "time stamp assigned to model initilization",
                    "number of time-steps between history records",
                    "number of time-steps between the creation of history files",
                    "number of time-steps between restart records",
                    "starting time-step for accumulation of time-averaged fields",
                    "number of time-steps between time-averaged records",
                    "number of time-steps between the creation of average files",
                    "Power-law shape barotropic filter parameter",
                    "Power-law shape barotropic filter parameter",
                    "Power-law shape barotropic filter parameter",
                    "nonlinear model Laplacian mixing coefficient for tracers",
                    "nonlinear model Laplacian mixing coefficient for momentum",
                    "horizontal viscosity sponge activation switch",
                    "horizontal diffusivity sponge activation switch",
                    "background vertical mixing coefficient for tracers",
                    "background vertical mixing coefficient for momentum",
                    "linear drag coefficient",
                    "quadratic drag coefficient",
                    "bottom roughness",
                    "surface roughness",
                    "free-surface nudging/relaxation inverse time scale",
                    "2D momentum nudging/relaxation inverse time scale",
                    "3D momentum nudging/relaxation inverse time scale",
                    "Tracers nudging/relaxation inverse time scale",
                    "free-surface inflow, nudging inverse time scale",
                    "free-surface outflow, nudging inverse time scale",
                    "2D momentum inflow, nudging inverse time scale",
                    "2D momentum outflow, nudging inverse time scale",
                    "tracers inflow, nudging inverse time scale",
                    "tracers outflow, nudging inverse time scale",
                    "3D momentum inflow, nudging inverse time scale",
                    "3D momentum outflow, nudging inverse time scale",
                    "mean density used in Boussinesq approximation",
                    "slipperiness parameter",
                    "momentum point sources and sink activation switch",
                    "mass point sources and sink activation switch",
                    "tracer point sources and sink activation switch",
                    "sea surface height climatology processing switch",
                    "2D momentum climatology processing switch",
                    "3D momentum climatology processing switch",
                    "tracer climatology processing switch",
                    "2D momentum climatology nudging activation switch",
                    "3D momentum climatology nudging activation switch",
                    "tracer climatology nudging activation switch",
                    "grid type logical switch",
                    "domain length in the XI-direction",
                    "domain length in the ETA-direction",
                    "vertical terrain-following transformation equation",
                    "vertical terrain-following stretching function",
                    "S-coordinate surface control parameter",
                    "S-coordinate bottom control parameter",
                    "S-coordinate surface/bottom layer width",
                    "S-coordinate parameter, critical depth",
                    "S-coordinate at RHO-points",
                    "S-coordinate at W-points",
                    "S-coordinate stretching curves at RHO-points",
                    "S-coordinate stretching curves at W-points",
                    "bathymetry at RHO-points",
                    "ice draft at RHO-points",
                    "Coriolis parameter at RHO-points",
                    "curvilinear coordinate metric in XI",
                    "curvilinear coordinate metric in ETA",
                    "x-locations of RHO-points",
                    "y-locations of RHO-points",
                    "x-locations of U-points",
                    "y-locations of U-points",
                    "x-locations of V-points",
                    "y-locations of V-points",
                    "x-locations of PSI-points",
                    "y-locations of PSI-points",
                    "angle between XI-axis and EAST",
                    "mask on RHO-points",
                    "mask on U-points",
                    "mask on V-points",
                    "mask on psi-points",
                    "time since initialization",
                    "melt rate",
                    "ice base temperature",
                    "ice base salinity",
                    "free-surface",
                    "vertically integrated u-momentum component",
                    "vertically integrated v-momentum component",
                    "u-momentum component",
                    "v-momentum component",
                    "vertical momentum component",
                    "potential temperature",
                    "salinity",
                    "density anomaly",
                    "depth of oceanic surface boundary layer",
                    "vertical viscosity coefficient",
                    "temperature vertical diffusion coefficient",
                    "surface net heat flux",
                    "kinematic surface net salt flux, SALT (E-P)/rhow",
                    "surface u-momentum stress",
                    "surface v-momentum stress",
                ],
                variable_standard_name=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_s_coordinate_g2",
                    "ocean_s_coordinate_g2",
                    "",
                    "",
                    "sea_floor_depth",
                    "",
                    "coriolis_parameter",
                    "inverse_grid_x_spacing",
                    "inverse_grid_y_spacing",
                    "grid_x_location_at_cell_center",
                    "grid_y_location_at_cell_center",
                    "grid_x_location_at_cell_y_edges",
                    "grid_y_location_at_cell_y_edges",
                    "grid_x_location_at_cell_x_edges",
                    "grid_y_location_at_cell_x_edges",
                    "grid_x_location_at_cell_corners",
                    "grid_y_location_at_cell_corners",
                    "grid_angle_of_rotation_from_east_to_y",
                    "land_sea_mask_at_cell_center",
                    "land_sea_mask_at_cell_y_edges",
                    "land_sea_mask_at_cell_x_edges",
                    "land_sea_mask_at_cell_corners",
                    "time",
                    "ice_shelf_melt_rate",
                    "temperature_at_ice_base",
                    "salinity_at_ice_base",
                    "sea_surface_elevation_anomaly",
                    "barotropic_sea_water_x_velocity",
                    "barotropic_sea_water_y_velocity",
                    "sea_water_x_velocity",
                    "sea_water_y_velocity",
                    "upward_sea_water_velocity",
                    "sea_water_potential_temperature",
                    "sea_water_practical_salinity",
                    "sea_water_density_anomaly",
                    "ocean_surface_boundary_layer_thickness",
                    "vertical_viscosity_coefficient_of_sea_water",
                    "vertical_diffusion_coefficient_of_temperature_in_sea_water",
                    "surface_downward_heat_flux_in_sea_water",
                    "salt_flux_into_sea_water",
                    "",
                    "surface_downward_x_stress",
                ],
                variable_cell_methods=[
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                    "ocean_time: point",
                ],
                variable_units=[
                    "",
                    "",
                    "second",
                    "second",
                    "days since 2007-01-01 00:00:00",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "meter2 second-1",
                    "meter2 second-1",
                    "",
                    "",
                    "meter2 second-1",
                    "meter2 second-1",
                    "meter second-1",
                    "",
                    "meter",
                    "meter",
                    "day-1",
                    "day-1",
                    "day-1",
                    "day-1",
                    "second-1",
                    "second-1",
                    "second-1",
                    "second-1",
                    "second-1",
                    "second-1",
                    "second-1",
                    "second-1",
                    "kilogram meter-3",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    "meter",
                    "meter",
                    "",
                    "",
                    "",
                    "",
                    "meter",
                    "meter",
                    "",
                    "",
                    "",
                    "",
                    "meter",
                    "meter",
                    "second-1",
                    "meter-1",
                    "meter-1",
                    "meter",
                    "meter",
                    "meter",
                    "meter",
                    "meter",
                    "meter",
                    "meter",
                    "meter",
                    "radians",
                    "",
                    "",
                    "",
                    "",
                    "seconds since 2007-01-01 00:00:00",
                    "meter second-1",
                    "Celsius",
                    "",
                    "meter",
                    "meter second-1",
                    "meter second-1",
                    "meter second-1",
                    "meter second-1",
                    "meter second-1",
                    "Celsius",
                    "",
                    "kilogram meter-3",
                    "meter",
                    "meter2 second-1",
                    "meter2 second-1",
                    "watt meter-2",
                    "meter second-1",
                    "newton meter-2",
                    "newton meter-2",
                ],
            ),
        ),
    ],
)
@pytest.mark.filterwarnings("ignore:Time coordinate does not include bounds")
@pytest.mark.filterwarnings(
    "ignore:The frequency '\\(1, 'mon'\\)' determined from filename"
)
def test_parse_access_ncfile(test_data, builder, filename, expected, compare_files):
    """
    Tests for correctness of parser. Note that if we are using intake-esm without
    coordinate discovery enabled, the `conftest.py` fixture will dynamically set
    the `xfails=True` on parametrisations of these tests where we compare files.
    This is currently default behaviour as the latest release of intake-esm does
    not support coordinate discovery. However, it is on main.
    """
    file = str(test_data / Path(filename))

    # Set the path to the test data directory
    expected.path = file

    parsed_file = builder.parse_ncfile(file)

    exptd_dict = expected.to_dict()
    parsed_dict = parsed_file.to_dict()
    assert exptd_dict == parsed_dict

    if not compare_files:
        return None

    """
    In the rest of this test, we refer to the dataset loaded using intake-esm
    as ie_ds and the dataset loaded directly with xarray as xr_ds.

    We also need to perform some additional logic that intake-esm does to avoid
    xr.testing.assert_equal from failing due to preprocessing differences.
    """
    xarray_open_kwargs = _get_xarray_open_kwargs("netcdf")

    ie_ds = _open_dataset(
        urlpath=expected.path,
        varname=expected.variable,
        xarray_open_kwargs=xarray_open_kwargs,
        requested_variables=expected.variable,
    ).compute()
    ie_ds.set_coords(set(ie_ds.variables) - set(ie_ds.attrs[OPTIONS["vars_key"]]))

    xr_ds = xr.open_dataset(file, **xarray_open_kwargs)

    scalar_variables = [v for v in xr_ds.data_vars if len(xr_ds[v].dims) == 0]
    xr_ds = xr_ds.set_coords(scalar_variables)

    xr_ds = xr_ds[expected.variable]

    xr.testing.assert_equal(ie_ds, xr_ds)


@pytest.mark.parametrize(
    "builder,default_include,default_exclude",
    [
        (builders.BaseBuilder, [], []),
        (builders.AccessOm2Builder, ["*.nc"], ["*restart*", "*o2i.nc"]),
        (
            builders.AccessOm3Builder,
            ["*.nc"],
            [
                "*restart*",
                "*MOM_IC.nc",
                "*ocean_geometry.nc",
                "*Vertical_coordinate.nc",
            ],
        ),
        (
            builders.Mom6Builder,
            ["*.nc"],
            [
                "*restart*",
                "*MOM_IC.nc",
                "*sea_ice_geometry.nc",
                "*ocean_geometry.nc",
                "*ocean.stats.nc",
                "*Vertical_coordinate.nc",
            ],
        ),
        (builders.AccessEsm15Builder, ["*.nc*"], ["*restart*"]),
        (builders.AccessCm2Builder, ["*.nc*"], ["*restart*"]),
        (builders.ROMSBuilder, ["*.nc"], ["*avg*", "*rst*"]),
    ],
)
@pytest.mark.parametrize("include_patts", [None, ["include", "patterns"]])
@pytest.mark.parametrize("exclude_patts", [None, ["exclude", "patterns"]])
def test_builder_include_exclude_patterns(
    builder, default_include, default_exclude, include_patts, exclude_patts, tmpdir
):
    if builder == builders.AccessEsm15Builder or builder == builders.AccessCm2Builder:
        additional = [
            False,  # ensemble needed for ESM builder
        ]
    else:
        additional = []

    kwargs = {}
    if include_patts is not None:
        kwargs["include_patterns"] = include_patts
    if exclude_patts is not None:
        kwargs["exclude_patterns"] = exclude_patts

    bld = builder(str(tmpdir), *additional, **kwargs)
    assert bld.include_patterns == (
        include_patts if include_patts is not None else default_include
    )
    assert bld.exclude_patterns == (
        exclude_patts if exclude_patts is not None else default_exclude
    )


@pytest.mark.parametrize(
    "test_dir,valid,realm,n_assets",
    [
        ("om3_realm/single-realm", True, ["ocean"], 1),
        ("om3_realm/no-realm", False, None, 1),
        ("om3_realm", True, ["ocean"], 2),  # All files
    ],
)
@pytest.mark.filterwarnings("ignore:Unable to parse 1 assets")
def test_builder_om3_realm(test_data, test_dir, valid, realm, n_assets):
    """
    Tests the OM3 builder with the .nc files in om3_realm.

    The single-realm file has the 'realm' global attribute set and the
    filenames does not match any of the normal patterns for extracting 'realm'.
    The no-realm file does not have the 'realm' attribute set.
    """
    data_path = str(test_data / test_dir)
    builder = builders.AccessOm3Builder(path=data_path)

    if valid:
        builder = builder.build()

        assert all(builder.df["realm"].isin(realm))
    else:
        with pytest.raises(builders.ParserError):
            builder = builder.build()
            import pdb

            pdb.set_trace()

    assert len(builder.assets) == n_assets


@pytest.mark.parametrize(
    "test_file,builder,is_monthly,expected_start_date",
    [
        ("woa/woa23_A5B4_s00_04.nc", "WoaBuilder", True, "2041-07-01, 00:00:00"),
        ("woa/woa23_A5B4_s01_04.nc", "WoaBuilder", True, "2009-01-16, 12:00:00"),
        ("woa/woa23_A5B4_s02_04.nc", "WoaBuilder", True, "2009-02-10, 19:11:56"),
        ("woa/woa23_A5B4_s00_04.nc.not-monthly", "WoaBuilder", False, None),
    ],
)
def test_builder_no_calendar(
    test_data, test_file, builder, is_monthly, expected_start_date
):
    """
    Test the cases where the .nc file's time variable is missing the calendar

    *s00* is an annual average file with time value 438
    *s01* is a January file with with time value 48.5
    *s02* is a February file with with time value changed to 49.35

    The woa23_A5B4_s00_04.nc.not-monthly has had it's time units attribute
    altered to be 'foobars since 2005-01-01 00:00:00'.
    """
    file_path = str(test_data / test_file)

    ncinfo_dict = getattr(builders, builder)._parser_catch_invalid(file_path)

    # File parse should succeed if monthly, fail otherwise
    assert ("INVALID_ASSET" not in ncinfo_dict) == is_monthly

    if is_monthly:
        assert ncinfo_dict["start_date"] == expected_start_date


@pytest.mark.parametrize(
    "test_file,builder,expected_startdate_str",
    [
        # get_timeinfo uses cftime for woa13_ts_01_mom01.nc
        ("woa/woa13_ts_01_mom01.nc", "WoaBuilder", "0001-01-02, 00:00:00"),
        # get_timeinfo uses datetime for woa13_decav_ts_01_04v2.nc
        ("woa/woa13_decav_ts_01_04v2.nc", "WoaBuilder", "0001-02-01, 00:00:00"),
    ],
)
def test_builder_year_before_1000(
    test_data, test_file, builder, expected_startdate_str
):
    """
    Test that time values with year<1000 are formatted correctly. In some instances
    if year<1000 then leading zeroes used to be missing. i.e. '1-02-01, 00:00:00' instead
    of '0001-01-01, 00:00:00'.

    If get_timeinfo ends up using a cftime object instead of a python datetime this
    issue does not manifest.
    """
    path = str(test_data / test_file)
    builder = getattr(builders, builder)
    asset = builder.parser(path)

    assert asset["start_date"] == expected_startdate_str
