"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineIamRole = exports.InfraIamRole = exports.DataBrewCodePipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const cdk_databrew_lambda_1 = require("./cdk-databrew-lambda");
/**
 * @stability stable
 */
class DataBrewCodePipeline extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, name, props) {
        var _d, _e, _f, _g, _h;
        super(scope, name);
        /**
             * Creates a source action.
             *
             * @param codeCommitRepo the CodeCommit repository used in the DataBrew CICD pipeline.
             * @param role the IAM role used by the CodePipeline pipeline.
             * @returns the CodeCOmmit source action.
             */
        this.createSourceAction = (codeCommitRepo, role) => {
            const sourceOutput = new codepipeline.Artifact(this.firstStageArtifactName);
            const sourceAction = new codepipeline_actions.CodeCommitSourceAction({
                actionName: 'Source',
                output: sourceOutput,
                branch: this.branchName,
                trigger: codepipeline_actions.CodeCommitTrigger.EVENTS,
                repository: codeCommitRepo,
                runOrder: 1,
                role: role,
            });
            return sourceAction;
        };
        this.firstStageArtifactName = (_d = props.firstStageArtifactName) !== null && _d !== void 0 ? _d : 'SourceOutput';
        this.repoName = (_e = props.repoName) !== null && _e !== void 0 ? _e : 'DataBrew-Recipes-Repo';
        this.branchName = (_f = props.branchName) !== null && _f !== void 0 ? _f : 'main';
        // create a bucket
        const pipelineBucket = new s3.Bucket(this, 'CodePipelineBucket', {
            bucketName: (_g = props.bucketName) !== null && _g !== void 0 ? _g : 'databrew-cicd-codepipelineartifactstorebucket',
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        pipelineBucket.addToResourcePolicy(new iam.PolicyStatement({
            sid: 'DenyUnEncryptedObjectUploads',
            effect: iam.Effect.DENY,
            principals: [new iam.AnyPrincipal()],
            actions: ['s3:PutObject'],
            resources: [`${pipelineBucket.bucketArn}/*`],
            conditions: {
                StringNotEquals: {
                    's3:x-amz-server-side-encryption': 'aws:kms',
                },
            },
        }));
        this.bucketArn = pipelineBucket.bucketArn;
        const preproductionLambda = new cdk_databrew_lambda_1.PreProductionLambda(this, 'PreProductionLambda', {
            bucketArn: pipelineBucket.bucketArn,
            preproductionIamRoleArn: props.preproductionIamRoleArn,
        });
        const productionLambda = new cdk_databrew_lambda_1.ProductionLambda(this, 'ProductionLambda', {
            bucketArn: pipelineBucket.bucketArn,
            productionIamRoleArn: props.productionIamRoleArn,
        });
        this.preproductionFunctionArn = preproductionLambda.function.functionArn;
        this.productionFunctionArn = productionLambda.function.functionArn;
        // create a CodeCommit repo
        const codeCommitRepo = new codecommit.Repository(this, 'DataBrewRepository', {
            repositoryName: 'DataBrew-Recipes-Repo',
        });
        this.codeCommitRepoArn = codeCommitRepo.repositoryArn;
        const firstCommitHelper = new cdk_databrew_lambda_1.FirstCommitHandler(this, 'FirstCommitLambda', {
            codeCommitRepoArn: codeCommitRepo.repositoryArn,
            repoName: codeCommitRepo.repositoryName,
            branchName: this.branchName,
        });
        const onEvent = firstCommitHelper.function;
        const lambdaInvoker = new cr.Provider(this, 'LambdaInvoker', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.FIVE_DAYS,
        });
        new cdk.CustomResource(this, 'CodeCommitCustomResource', {
            serviceToken: lambdaInvoker.serviceToken,
            resourceType: 'Custom::LambdaInvoker',
        });
        // create a CodePipeline pipeline
        const pipelineRole = new CodePipelineIamRole(this, 'DataBrewCodePipelineRole', {
            bucketArn: pipelineBucket.bucketArn,
            preproductionLambdaArn: this.preproductionFunctionArn,
            productionLambdaArn: this.productionFunctionArn,
        });
        pipelineRole.node.addDependency(preproductionLambda);
        pipelineRole.node.addDependency(productionLambda);
        const databrewCicdPipeline = new codepipeline.Pipeline(this, 'DataBrewCicdPipeline', {
            artifactBucket: pipelineBucket,
            role: pipelineRole.role.withoutPolicyUpdates(),
            pipelineName: (_h = props.pipelineName) !== null && _h !== void 0 ? _h : 'DataBrew-Recipe-Application',
        });
        databrewCicdPipeline.node.addDependency(pipelineRole.role);
        const sourceAction = this.createSourceAction(codeCommitRepo, pipelineRole.role);
        const preproductionLambdaAction = new codepipeline_actions.LambdaInvokeAction({
            inputs: [new codepipeline.Artifact(this.firstStageArtifactName)],
            actionName: 'PreProd-DeployRecipe',
            lambda: preproductionLambda.function,
            role: pipelineRole.role,
        });
        const productionLambdaAction = new codepipeline_actions.LambdaInvokeAction({
            inputs: [new codepipeline.Artifact(this.firstStageArtifactName)],
            actionName: 'Prod-DeployRecipe',
            lambda: productionLambda.function,
            role: pipelineRole.role,
        });
        databrewCicdPipeline.addStage({
            stageName: 'Source',
            actions: [sourceAction],
        });
        databrewCicdPipeline.addStage({
            stageName: 'PreProd-DeployRecipe',
            actions: [preproductionLambdaAction],
        });
        databrewCicdPipeline.addStage({
            stageName: 'Prod-DeployRecipe',
            actions: [productionLambdaAction],
        });
        this.codePipelineArn = databrewCicdPipeline.pipelineArn;
    }
}
exports.DataBrewCodePipeline = DataBrewCodePipeline;
_a = JSII_RTTI_SYMBOL_1;
DataBrewCodePipeline[_a] = { fqn: "cdk-databrew-cicd.DataBrewCodePipeline", version: "0.1.30" };
/**
 * @stability stable
 */
class InfraIamRole extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, name, props) {
        var _d;
        super(scope, name);
        const role = new iam.Role(this, 'InfraIamRole', {
            assumedBy: new iam.AccountRootPrincipal(),
            roleName: (_d = props.roleName) !== null && _d !== void 0 ? _d : 'CrossAccountRepositoryContributorRole',
            description: 'The IAM role for DataBrew CICD in the infrastructure account.',
        });
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'CodeCommitPermissions',
            effect: iam.Effect.ALLOW,
            actions: [
                'codecommit:BatchGet*',
                'codecommit:Create*',
                'codecommit:DeleteBranch',
                'codecommit:Get*',
                'codecommit:List*',
                'codecommit:Describe*',
                'codecommit:Put*',
                'codecommit:Post*',
                'codecommit:Merge*',
                'codecommit:Test*',
                'codecommit:Update*',
                'codecommit:GitPull',
                'codecommit:GitPush',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:codecommit:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:DataBrew-Recipes-Repo`,
            ],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'CodeCommitListReposPermission',
            effect: iam.Effect.ALLOW,
            actions: [
                'codecommit:ListRepositories',
            ],
            resources: ['*'],
        }));
        this.roleArn = role.roleArn;
    }
}
exports.InfraIamRole = InfraIamRole;
_b = JSII_RTTI_SYMBOL_1;
InfraIamRole[_b] = { fqn: "cdk-databrew-cicd.InfraIamRole", version: "0.1.30" };
/**
 * @stability stable
 */
class CodePipelineIamRole extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, name, props) {
        var _d;
        super(scope, name);
        const role = new iam.Role(this, 'CodePipelineIamRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codepipeline.amazonaws.com'), new iam.ServicePrincipal('lambda.amazonaws.com')),
            roleName: (_d = props.roleName) !== null && _d !== void 0 ? _d : 'DataBrew-Recipe-Pipeline-Role',
            description: 'The IAM role for the CodePipeline CICD pipeline in the infrastructure account.',
        });
        // CodeDeploy permissions
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'CodeDeployPermissions',
            effect: iam.Effect.ALLOW,
            actions: [
                'codedeploy:CreateDeployment',
                'codedeploy:GetApplicationRevision',
                'codedeploy:GetDeployment',
                'codedeploy:GetDeploymentConfig',
                'codedeploy:RegisterApplicationRevision',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:codedeploy:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:deploymentgroup:DataBrew-Recipe-Application*`,
                `arn:${cdk.Aws.PARTITION}:codedeploy:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:application:DataBrew-Recipe-Application`,
            ],
        }));
        // CodeCommit permissions
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'CodeCommitPermissions',
            effect: iam.Effect.ALLOW,
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:UploadArchive',
            ],
            resources: [
                `arn:${cdk.Aws.PARTITION}:codecommit:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:DataBrew-Recipes-Repo`,
            ],
        }));
        // S3 permissions
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'S3Permissions',
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:PutObject',
            ],
            resources: [props.bucketArn],
        }));
        // Lambda permissions
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'LambdaPermissions',
            effect: iam.Effect.ALLOW,
            actions: [
                'lambda:InvokeFunction',
            ],
            resources: [props.preproductionLambdaArn, props.productionLambdaArn],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            sid: 'AssumRolePermissions',
            effect: iam.Effect.ALLOW,
            actions: [
                'sts:AssumeRole',
            ],
            resources: [role.roleArn],
        }));
        this.role = role;
        this.roleArn = role.roleArn;
    }
}
exports.CodePipelineIamRole = CodePipelineIamRole;
_c = JSII_RTTI_SYMBOL_1;
CodePipelineIamRole[_c] = { fqn: "cdk-databrew-cicd.CodePipelineIamRole", version: "0.1.30" };
//# sourceMappingURL=data:application/json;base64,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