"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeDeployEcsDeployAction = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * @stability stable
 */
class CodeDeployEcsDeployAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_actions_CodeDeployEcsDeployActionProps(props);
        const inputs = [];
        inputs.push(determineTaskDefinitionArtifact(props));
        inputs.push(determineAppSpecArtifact(props));
        if (props.containerImageInputs) {
            if (props.containerImageInputs.length > 4) {
                throw new Error(`Action cannot have more than 4 container image inputs, got: ${props.containerImageInputs.length}`);
            }
            for (const imageInput of props.containerImageInputs) {
                inputs.push(imageInput.input);
            }
        }
        super({
            ...props,
            resource: props.deploymentGroup,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'CodeDeployToECS',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 0 },
            inputs,
        });
        this.actionProps = props;
    }
    /**
     * This is a renamed version of the {@link IAction.bind} method.
     *
     * @stability stable
     */
    bound(_scope, _stage, options) {
        // permissions, based on:
        // https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.application.applicationArn],
            actions: ['codedeploy:GetApplication', 'codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentGroupArn],
            actions: ['codedeploy:CreateDeployment', 'codedeploy:GetDeployment'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentConfig.deploymentConfigArn],
            actions: ['codedeploy:GetDeploymentConfig'],
        }));
        // Allow action to register the task definition template file with ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['ecs:RegisterTaskDefinition'],
        }));
        // Allow passing any roles specified in the task definition template file to ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        // the Action's Role needs to read from the Bucket to get artifacts
        options.bucket.grantRead(options.role);
        const taskDefinitionTemplateArtifact = determineTaskDefinitionArtifact(this.actionProps);
        const appSpecTemplateArtifact = determineAppSpecArtifact(this.actionProps);
        const actionConfig = {
            configuration: {
                ApplicationName: this.actionProps.deploymentGroup.application.applicationName,
                DeploymentGroupName: this.actionProps.deploymentGroup.deploymentGroupName,
                TaskDefinitionTemplateArtifact: core_1.Lazy.string({ produce: () => taskDefinitionTemplateArtifact.artifactName }),
                TaskDefinitionTemplatePath: this.actionProps.taskDefinitionTemplateFile
                    ? this.actionProps.taskDefinitionTemplateFile.fileName
                    : 'taskdef.json',
                AppSpecTemplateArtifact: core_1.Lazy.string({ produce: () => appSpecTemplateArtifact.artifactName }),
                AppSpecTemplatePath: this.actionProps.appSpecTemplateFile
                    ? this.actionProps.appSpecTemplateFile.fileName
                    : 'appspec.yaml',
            },
        };
        if (this.actionProps.containerImageInputs) {
            for (let i = 1; i <= this.actionProps.containerImageInputs.length; i++) {
                const imageInput = this.actionProps.containerImageInputs[i - 1];
                actionConfig.configuration[`Image${i}ArtifactName`] = core_1.Lazy.string({ produce: () => imageInput.input.artifactName });
                actionConfig.configuration[`Image${i}ContainerName`] = imageInput.taskDefinitionPlaceholder
                    ? imageInput.taskDefinitionPlaceholder
                    : 'IMAGE';
            }
        }
        return actionConfig;
    }
}
exports.CodeDeployEcsDeployAction = CodeDeployEcsDeployAction;
_a = JSII_RTTI_SYMBOL_1;
CodeDeployEcsDeployAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CodeDeployEcsDeployAction", version: "1.138.1" };
function determineTaskDefinitionArtifact(props) {
    if (props.taskDefinitionTemplateFile && props.taskDefinitionTemplateInput) {
        throw new Error("Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.taskDefinitionTemplateFile) {
        return props.taskDefinitionTemplateFile.artifact;
    }
    if (props.taskDefinitionTemplateInput) {
        return props.taskDefinitionTemplateInput;
    }
    throw new Error("Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action");
}
function determineAppSpecArtifact(props) {
    if (props.appSpecTemplateFile && props.appSpecTemplateInput) {
        throw new Error("Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.appSpecTemplateFile) {
        return props.appSpecTemplateFile.artifact;
    }
    if (props.appSpecTemplateInput) {
        return props.appSpecTemplateInput;
    }
    throw new Error("Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action");
}
//# sourceMappingURL=data:application/json;base64,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