import requests
import pandas as pd
from bs4 import BeautifulSoup
import sys

"""
.. module:: util
   :synopsis: General function for the package.

.. moduleauthor:: Tianning Li <ltianningli@gmail.com>
"""

headers = {'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_4) \
            AppleWebKit/537.36 (KHTML, like Gecko) Chrome/81.0.4044.138 Safari/537.36'}

NUMBER_COL = ['Market Cap', 'P/E', 'Fwd P/E', 'PEG', 'P/S', 'P/B', 'P/C',
              'P/FCF', 'Dividend', 'Payout Ratio', 'EPS', 'EPS this Y', 'EPS next Y',
              'EPS past 5Y', 'EPS next 5Y', 'Sales past 5Y', 'EPS Q/Q', 'Sales Q/Q',
              'Outstanding', 'Float', 'Insider Own', 'Insider Trans', 'Inst Own', 'Inst Trans',
              'Float Short', 'Short Ratio', 'ROA', 'ROE', 'ROI', 'Curr R', 'Quick R', 'LTDebt/Eq',
              'Debt/Eq', 'Gross M', 'Oper M', 'Profit M', 'Perf Week', 'Perf Month', 'Perf Quart',
              'Perf Half', 'Perf Year', 'Perf YTD', 'Beta', 'ATR', 'Volatility W', 'Volatility M',
              'SMA20', 'SMA50', 'SMA200', '50D High', '50D Low', '52W High', '52W Low', 'RSI',
              'from Open', 'Gap', 'Recom', 'Avg Volume', 'Rel Volume', 'Price', 'Change', 'Volume',
              'Target Price']

session = requests.Session()


def webScrap(url):
    """Scrap website.

    Args:
        url(str): website
    Returns:
        soup(beautiful soup): website html
    """
    try:
        website = session.get(url, headers=headers, timeout=10)
        website.raise_for_status()
        soup = BeautifulSoup(website.text, 'lxml')
    except requests.exceptions.HTTPError as err:
        raise Exception(err)
    except requests.exceptions.Timeout as err:
        raise Exception(err)
    return soup


def imageScrap(url, ticker, out_dir):
    """scrap website and download image

    Args:
        url(str): website (image)
        ticker(str): output image name
        out_dir(str): output directory
    """
    try:
        r = session.get(url, stream=True, headers=headers, timeout=10)
        r.raise_for_status()
        r.raw.decode_content = True
        if len(out_dir) != 0:
            out_dir += '/'
        f = open('{}{}.jpg'.format(out_dir, ticker), "wb")
        f.write(r.content)
        f.close()
    except requests.exceptions.HTTPError as err:
        raise Exception(err)
    except requests.exceptions.Timeout as err:
        raise Exception(err)


def scrapFunction(url):
    """Scrap forex, crypto information.

    Args:
        url(str): website
    Returns:
        df(pandas.DataFrame): performance table
    """
    soup = webScrap(url)
    table = soup.findAll('table')[3]
    rows = table.findAll('tr')
    table_header = [i.text.strip() for i in rows[0].findAll('td')][1:]
    df = pd.DataFrame([], columns=table_header)
    rows = rows[1:]
    num_col_index = [i for i in range(2, len(table_header))]
    for row in rows:
        cols = row.findAll('td')[1:]
        info_dict = {}
        for i, col in enumerate(cols):
            if i not in num_col_index:
                info_dict[table_header[i]] = col.text
            else:
                info_dict[table_header[i]] = numberCovert(col.text)
        df = df.append(info_dict, ignore_index=True)
    return df


def imageScrapFunction(url, chart, timeframe, urlonly):
    """Scrap forex, crypto information.

    Args:
        url(str): website
        chart(str): choice of chart
        timeframe (str): choice of timeframe(5M, H, D, W, M)
        urlonly (boolean):  choice of downloading chart
    """
    if timeframe == '5M':
        url += 'm5'
    elif timeframe == 'H':
        url += 'h1'
    elif timeframe == 'D':
        url += 'd1'
    elif timeframe == 'W':
        url += 'w1'
    elif timeframe == 'M':
        url += 'mo'
    else:
        raise ValueError('Invalid timeframe.')

    soup = webScrap(url)
    content = soup.find('div', class_='container')
    imgs = content.findAll('img')
    for img in imgs:
        website = img['src']
        name = website.split('?')[1].split('&')[0].split('.')[0]
        chart_name = name.split('_')[0]
        if chart.lower() == chart_name:
            charturl = 'https://finviz.com/' + website
            if not urlonly:
                imageScrap(charturl, name, '')
            return charturl
        else:
            continue


def numberCovert(num):
    """covert number(str) to number(float)

    Args:
        num(str): number of string
    Return:
        num(float): number of string
    """
    if num == '-':
        return None
    elif num[-1] == '%':
        return float(num[:-1]) / 100
    elif num[-1] == 'B':
        return float(num[:-1]) * 1000000000
    elif num[-1] == 'M':
        return float(num[:-1]) * 1000000
    elif num[-1] == 'K':
        return float(num[:-1]) * 1000
    else:
        return float(''.join(num.split(',')))


def progressBar(page, total):
    bar_len = 30
    filled_len = int(round(bar_len * page / float(total)))
    bar = '#' * filled_len + '-' * (bar_len - filled_len)
    sys.stdout.write('[Info] loading page [{}] {}/{} \r'.format(bar, page, total))
    sys.stdout.flush()


util_dict = {"signal": {"Top Gainers": "ta_topgainers", "Top Losers": "ta_toplosers", "New High": "ta_newhigh", "New Low": "ta_newlow", "Most Volatile": "ta_mostvolatile", "Most Active": "ta_mostactive", "Unusual Volume": "ta_unusualvolume", "Overbought": "ta_overbought", "Oversold": "ta_oversold", "Downgrades": "n_downgrades", "Upgrades": "n_upgrades", "Earnings Before": "n_earningsbefore", "Earnings After": "n_earningsafter", "Recent Insider Buying": "it_latestbuys", "Recent Insider Selling": "it_latestsales", "Major News": "n_majornews", "Horizontal S/R": "ta_p_horizontal", "TL Resistance": "ta_p_tlresistance", "TL Support": "ta_p_tlsupport", "Wedge Up": "ta_p_wedgeup", "Wedge Down": "ta_p_wedgedown", "Triangle Ascending": "ta_p_wedgeresistance", "Triangle Descending": "ta_p_wedgesupport", "Wedge": "ta_p_wedge", "Channel Up": "ta_p_channelup", "Channel Down": "ta_p_channeldown", "Channel": "ta_p_channel", "Double Top": "ta_p_doubletop", "Double Bottom": "ta_p_doublebottom", "Multiple Top": "ta_p_multipletop", "Multiple Bottom": "ta_p_multiplebottom", "Head & Shoulders": "ta_p_headandshoulders", "Head & Shoulders Inverse": "ta_p_headandshouldersinv"}, "filter": {"Exchange": {"prefix": "exch", "option": {"Any": "", "AMEX": "amex", "NASDAQ": "nasd", "NYSE": "nyse"}}, "Index": {"prefix": "idx", "option": {"Any": "", "S&P 500": "sp500", "DJIA": "dji"}}, "Sector": {"prefix": "sec", "option": {"Any": "", "Basic Materials": "basicmaterials", "Communication Services": "communicationservices", "Consumer Cyclical": "consumercyclical", "Consumer Defensive": "consumerdefensive", "Energy": "energy", "Financial": "financial", "Healthcare": "healthcare", "Industrials": "industrials", "Real Estate": "realestate", "Technology": "technology", "Utilities": "utilities"}}, "Industry": {"prefix": "ind", "option": {"Any": "", "Stocks only (ex-Funds)": "stocksonly", "Exchange Traded Fund": "exchangetradedfund", "Advertising Agencies": "advertisingagencies", "Aerospace & Defense": "aerospacedefense", "Agricultural Inputs": "agriculturalinputs", "Airlines": "airlines", "Airports & Air Services": "airportsairservices", "Aluminum": "aluminum", "Apparel Manufacturing": "apparelmanufacturing", "Apparel Retail": "apparelretail", "Asset Management": "assetmanagement", "Auto Manufacturers": "automanufacturers", "Auto Parts": "autoparts", "Auto & Truck Dealerships": "autotruckdealerships", "Banks - Diversified": "banksdiversified", "Banks - Regional": "banksregional", "Beverages - Brewers": "beveragesbrewers", "Beverages - Non-Alcoholic": "beveragesnonalcoholic", "Beverages - Wineries & Distilleries": "beverageswineriesdistilleries", "Biotechnology": "biotechnology", "Broadcasting": "broadcasting", "Building Materials": "buildingmaterials", "Building Products & Equipment": "buildingproductsequipment", "Business Equipment & Supplies": "businessequipmentsupplies", "Capital Markets": "capitalmarkets", "Chemicals": "chemicals", "Closed-End Fund - Debt": "closedendfunddebt", "Closed-End Fund - Equity": "closedendfundequity", "Closed-End Fund - Foreign": "closedendfundforeign", "Coking Coal": "cokingcoal", "Communication Equipment": "communicationequipment", "Computer Hardware": "computerhardware", "Confectioners": "confectioners", "Conglomerates": "conglomerates", "Consulting Services": "consultingservices", "Consumer Electronics": "consumerelectronics", "Copper": "copper", "Credit Services": "creditservices", "Department Stores": "departmentstores", "Diagnostics & Research": "diagnosticsresearch", "Discount Stores": "discountstores", "Drug Manufacturers - General": "drugmanufacturersgeneral", "Drug Manufacturers - Specialty & Generic": "drugmanufacturersspecialtygeneric", "Education & Training Services": "educationtrainingservices", "Electrical Equipment & Parts": "electricalequipmentparts", "Electronic Components": "electroniccomponents", "Electronic Gaming & Multimedia": "electronicgamingmultimedia", "Electronics & Computer Distribution": "electronicscomputerdistribution", "Engineering & Construction": "engineeringconstruction", "Entertainment": "entertainment", "Farm & Heavy Construction Machinery": "farmheavyconstructionmachinery", "Farm Products": "farmproducts", "Financial Conglomerates": "financialconglomerates", "Financial Data & Stock Exchanges": "financialdatastockexchanges", "Food Distribution": "fooddistribution", "Footwear & Accessories": "footwearaccessories", "Furnishings, Fixtures & Appliances": "furnishingsfixturesappliances", "Gambling": "gambling", "Gold": "gold", "Grocery Stores": "grocerystores", "Healthcare Plans": "healthcareplans", "Health Information Services": "healthinformationservices", "Home Improvement Retail": "homeimprovementretail", "Household & Personal Products": "householdpersonalproducts", "Industrial Distribution": "industrialdistribution", "Information Technology Services": "informationtechnologyservices", "Infrastructure Operations": "infrastructureoperations", "Insurance Brokers": "insurancebrokers", "Insurance - Diversified": "insurancediversified", "Insurance - Life": "insurancelife", "Insurance - Property & Casualty": "insurancepropertycasualty", "Insurance - Reinsurance": "insurancereinsurance", "Insurance - Specialty": "insurancespecialty", "Integrated Freight & Logistics": "integratedfreightlogistics", "Internet Content & Information": "internetcontentinformation", "Internet Retail": "internetretail", "Leisure": "leisure", "Lodging": "lodging", "Lumber & Wood Production": "lumberwoodproduction", "Luxury Goods": "luxurygoods", "Marine Shipping": "marineshipping", "Medical Care Facilities": "medicalcarefacilities", "Medical Devices": "medicaldevices", "Medical Distribution": "medicaldistribution", "Medical Instruments & Supplies": "medicalinstrumentssupplies", "Metal Fabrication": "metalfabrication", "Mortgage Finance": "mortgagefinance", "Oil & Gas Drilling": "oilgasdrilling", "Oil & Gas E&P": "oilgasep", "Oil & Gas Equipment & Services": "oilgasequipmentservices", "Oil & Gas Integrated": "oilgasintegrated", "Oil & Gas Midstream": "oilgasmidstream", "Oil & Gas Refining & Marketing": "oilgasrefiningmarketing", "Other Industrial Metals & Mining": "otherindustrialmetalsmining", "Other Precious Metals & Mining": "otherpreciousmetalsmining", "Packaged Foods": "packagedfoods", "Packaging & Containers": "packagingcontainers", "Paper & Paper Products": "paperpaperproducts", "Personal Services": "personalservices", "Pharmaceutical Retailers": "pharmaceuticalretailers", "Pollution & Treatment Controls": "pollutiontreatmentcontrols", "Publishing": "publishing", "Railroads": "railroads", "Real Estate - Development": "realestatedevelopment", "Real Estate - Diversified": "realestatediversified", "Real Estate Services": "realestateservices", "Recreational Vehicles": "recreationalvehicles", "REIT - Diversified": "reitdiversified", "REIT - Healthcare Facilities": "reithealthcarefacilities", "REIT - Hotel & Motel": "reithotelmotel", "REIT - Industrial": "reitindustrial", "REIT - Mortgage": "reitmortgage", "REIT - Office": "reitoffice", "REIT - Residential": "reitresidential", "REIT - Retail": "reitretail", "REIT - Specialty": "reitspecialty", "Rental & Leasing Services": "rentalleasingservices", "Residential Construction": "residentialconstruction", "Resorts & Casinos": "resortscasinos", "Restaurants": "restaurants", "Scientific & Technical Instruments": "scientifictechnicalinstruments", "Security & Protection Services": "securityprotectionservices", "Semiconductor Equipment & Materials": "semiconductorequipmentmaterials", "Semiconductors": "semiconductors", "Shell Companies": "shellcompanies", "Silver": "silver", "Software - Application": "softwareapplication", "Software - Infrastructure": "softwareinfrastructure", "Solar": "solar", "Specialty Business Services": "specialtybusinessservices", "Specialty Chemicals": "specialtychemicals", "Specialty Industrial Machinery": "specialtyindustrialmachinery", "Specialty Retail": "specialtyretail", "Staffing & Employment Services": "staffingemploymentservices", "Steel": "steel", "Telecom Services": "telecomservices", "Textile Manufacturing": "textilemanufacturing", "Thermal Coal": "thermalcoal", "Tobacco": "tobacco", "Tools & Accessories": "toolsaccessories", "Travel Services": "travelservices", "Trucking": "trucking", "Uranium": "uranium", "Utilities - Diversified": "utilitiesdiversified", "Utilities - Independent Power Producers": "utilitiesindependentpowerproducers", "Utilities - Regulated Electric": "utilitiesregulatedelectric", "Utilities - Regulated Gas": "utilitiesregulatedgas", "Utilities - Regulated Water": "utilitiesregulatedwater", "Utilities - Renewable": "utilitiesrenewable", "Waste Management": "wastemanagement"}}, "Country": {"prefix": "geo", "option": {"Any": "", "USA": "usa", "Foreign (ex-USA)": "notusa", "Asia": "asia", "Europe": "europe", "Latin America": "latinamerica", "BRIC": "bric", "Argentina": "argentina", "Australia": "australia", "Bahamas": "bahamas", "Belgium": "belgium", "BeNeLux": "benelux", "Bermuda": "bermuda", "Brazil": "brazil", "Canada": "canada", "Cayman Islands": "caymanislands", "Chile": "chile", "China": "china", "China & Hong Kong": "chinahongkong", "Colombia": "colombia", "Cyprus": "cyprus", "Denmark": "denmark", "Finland": "finland", "France": "france", "Germany": "germany", "Greece": "greece", "Hong Kong": "hongkong", "Hungary": "hungary", "Iceland": "iceland", "India": "india", "Indonesia": "indonesia", "Ireland": "ireland", "Israel": "israel", "Italy": "italy", "Japan": "japan", "Kazakhstan": "kazakhstan", "Luxembourg": "luxembourg", "Malaysia": "malaysia", "Malta": "malta", "Mexico": "mexico", "Monaco": "monaco", "Netherlands": "netherlands", "New Zealand": "newzealand", "Norway": "norway", "Panama": "panama", "Peru": "peru", "Philippines": "philippines", "Portugal": "portugal", "Russia": "russia", "Singapore": "singapore", "South Africa": "southafrica", "South Korea": "southkorea", "Spain": "spain", "Sweden": "sweden", "Switzerland": "switzerland", "Taiwan": "taiwan", "Turkey": "turkey", "United Arab Emirates": "unitedarabemirates", "United Kingdom": "unitedkingdom", "Uruguay": "uruguay"}}, "Market Cap.": {"prefix": "cap", "option": {"Any": "", "Mega ($200bln and more)": "mega", "Large ($10bln to $200bln)": "large", "Mid ($2bln to $10bln)": "mid", "Small ($300mln to $2bln)": "small", "Micro ($50mln to $300mln)": "micro", "Nano (under $50mln)": "nano", "+Large (over $10bln)": "largeover", "+Mid (over $2bln)": "midover", "+Small (over $300mln)": "smallover", "+Micro (over $50mln)": "microover", "-Large (under $200bln)": "largeunder", "-Mid (under $10bln)": "midunder", "-Small (under $2bln)": "smallunder", "-Micro (under $300mln)": "microunder"}}, "P/E": {"prefix": "fa_pe", "option": {"Any": "", "Low (<15)": "low", "Profitable (>0)": "profitable", "High (>50)": "high", "Under 5": "u5", "Under 10": "u10", "Under 15": "u15", "Under 20": "u20", "Under 25": "u25", "Under 30": "u30", "Under 35": "u35", "Under 40": "u40", "Under 45": "u45", "Under 50": "u50", "Over 5": "o5", "Over 10": "o10", "Over 15": "o15", "Over 20": "o20", "Over 25": "o25", "Over 30": "o30", "Over 35": "o35", "Over 40": "o40", "Over 45": "o45", "Over 50": "o50"}}, "Forward P/E": {"prefix": "fa_fpe", "option": {"Any": "", "Low (<15)": "low", "Profitable (>0)": "profitable", "High (>50)": "high", "Under 5": "u5", "Under 10": "u10", "Under 15": "u15", "Under 20": "u20", "Under 25": "u25", "Under 30": "u30", "Under 35": "u35", "Under 40": "u40", "Under 45": "u45", "Under 50": "u50", "Over 5": "o5", "Over 10": "o10", "Over 15": "o15", "Over 20": "o20", "Over 25": "o25", "Over 30": "o30", "Over 35": "o35", "Over 40": "o40", "Over 45": "o45", "Over 50": "o50"}}, "PEG": {"prefix": "fa_peg", "option": {"Any": "", "Low (<1)": "low", "High (>2)": "high", "Under 1": "u1", "Under 2": "u2", "Under 3": "u3", "Over 1": "o1", "Over 2": "o2", "Over 3": "o3"}}, "P/S": {"prefix": "fa_ps", "option": {"Any": "", "Low (<1)": "low", "High (>10)": "high", "Under 1": "u1", "Under 2": "u2", "Under 3": "u3", "Under 4": "u4", "Under 5": "u5", "Under 6": "u6", "Under 7": "u7", "Under 8": "u8", "Under 9": "u9", "Under 10": "u10", "Over 1": "o1", "Over 2": "o2", "Over 3": "o3", "Over 4": "o4", "Over 5": "o5", "Over 6": "o6", "Over 7": "o7", "Over 8": "o8", "Over 9": "o9", "Over 10": "o10"}}, "P/B": {"prefix": "fa_pb", "option": {"Any": "", "Low (<1)": "low", "High (>5)": "high", "Under 1": "u1", "Under 2": "u2", "Under 3": "u3", "Under 4": "u4", "Under 5": "u5", "Under 6": "u6", "Under 7": "u7", "Under 8": "u8", "Under 9": "u9", "Under 10": "u10", "Over 1": "o1", "Over 2": "o2", "Over 3": "o3", "Over 4": "o4", "Over 5": "o5", "Over 6": "o6", "Over 7": "o7", "Over 8": "o8", "Over 9": "o9", "Over 10": "o10"}}, "Price/Cash": {"prefix": "fa_pc", "option": {"Any": "", "Low (<3)": "low", "High (>50)": "high", "Under 1": "u1", "Under 2": "u2", "Under 3": "u3", "Under 4": "u4", "Under 5": "u5", "Under 6": "u6", "Under 7": "u7", "Under 8": "u8", "Under 9": "u9", "Under 10": "u10", "Over 1": "o1", "Over 2": "o2", "Over 3": "o3", "Over 4": "o4", "Over 5": "o5", "Over 6": "o6", "Over 7": "o7", "Over 8": "o8", "Over 9": "o9", "Over 10": "o10", "Over 20": "o20", "Over 30": "o30", "Over 40": "o40", "Over 50": "o50"}}, "Price/Free Cash Flow": {"prefix": "fa_pfcf", "option": {"Any": "", "Low (<15)": "low", "High (>50)": "high", "Under 5": "u5", "Under 10": "u10", "Under 15": "u15", "Under 20": "u20", "Under 25": "u25", "Under 30": "u30", "Under 35": "u35", "Under 40": "u40", "Under 45": "u45", "Under 50": "u50", "Under 60": "u60", "Under 70": "u70", "Under 80": "u80", "Under 90": "u90", "Under 100": "u100", "Over 5": "o5", "Over 10": "o10", "Over 15": "o15", "Over 20": "o20", "Over 25": "o25", "Over 30": "o30", "Over 35": "o35", "Over 40": "o40", "Over 45": "o45", "Over 50": "o50", "Over 60": "o60", "Over 70": "o70", "Over 80": "o80", "Over 90": "o90", "Over 100": "o100"}}, "EPS growththis year": {"prefix": "fa_epsyoy", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (0-10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "EPS growthnext year": {"prefix": "fa_epsyoy1", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (0-10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "EPS growthpast 5 years": {"prefix": "fa_eps5years", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (0-10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "EPS growthnext 5 years": {"prefix": "fa_estltgrowth", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (<10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "Sales growthpast 5 years": {"prefix": "fa_sales5years", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (0-10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "EPS growthqtr over qtr": {"prefix": "fa_epsqoq", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (0-10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "Sales growthqtr over qtr": {"prefix": "fa_salesqoq", "option": {"Any": "", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Positive Low (0-10%)": "poslow", "High (>25%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "Dividend Yield": {"prefix": "fa_div", "option": {"Any": "", "None (0%)": "none", "Positive (>0%)": "pos", "High (>5%)": "high", "Very High (>10%)": "veryhigh", "Over 1%": "o1", "Over 2%": "o2", "Over 3%": "o3", "Over 4%": "o4", "Over 5%": "o5", "Over 6%": "o6", "Over 7%": "o7", "Over 8%": "o8", "Over 9%": "o9", "Over 10%": "o10"}}, "Return on Assets": {"prefix": "fa_roa", "option": {"Any": "", "Positive (>0%)": "pos", "Negative (<0%)": "neg", "Very Positive (>15%)": "verypos", "Very Negative (": "veryneg", "Under -50%": "u-50", "Under -45%": "u-45", "Under -40%": "u-40", "Under -35%": "u-35", "Under -30%": "u-30", "Under -25%": "u-25", "Under -20%": "u-20", "Under -15%": "u-15", "Under -10%": "u-10", "Under -5%": "u-5", "Over +5%": "o5", "Over +10%": "o10", "Over +15%": "o15", "Over +20%": "o20", "Over +25%": "o25", "Over +30%": "o30", "Over +35%": "o35", "Over +40%": "o40", "Over +45%": "o45", "Over +50%": "o50"}}, "Return on Equity": {"prefix": "fa_roe", "option": {"Any": "", "Positive (>0%)": "pos", "Negative (<0%)": "neg", "Very Positive (>30%)": "verypos", "Very Negative (": "veryneg", "Under -50%": "u-50", "Under -45%": "u-45", "Under -40%": "u-40", "Under -35%": "u-35", "Under -30%": "u-30", "Under -25%": "u-25", "Under -20%": "u-20", "Under -15%": "u-15", "Under -10%": "u-10", "Under -5%": "u-5", "Over +5%": "o5", "Over +10%": "o10", "Over +15%": "o15", "Over +20%": "o20", "Over +25%": "o25", "Over +30%": "o30", "Over +35%": "o35", "Over +40%": "o40", "Over +45%": "o45", "Over +50%": "o50"}}, "Return on Investment": {"prefix": "fa_roi", "option": {"Any": "", "Positive (>0%)": "pos", "Negative (<0%)": "neg", "Very Positive (>25%)": "verypos", "Very Negative (": "veryneg", "Under -50%": "u-50", "Under -45%": "u-45", "Under -40%": "u-40", "Under -35%": "u-35", "Under -30%": "u-30", "Under -25%": "u-25", "Under -20%": "u-20", "Under -15%": "u-15", "Under -10%": "u-10", "Under -5%": "u-5", "Over +5%": "o5", "Over +10%": "o10", "Over +15%": "o15", "Over +20%": "o20", "Over +25%": "o25", "Over +30%": "o30", "Over +35%": "o35", "Over +40%": "o40", "Over +45%": "o45", "Over +50%": "o50"}}, "Current Ratio": {"prefix": "fa_curratio", "option": {"Any": "", "High (>3)": "high", "Low (<1)": "low", "Under 1": "u1", "Under 0.5": "u0.5", "Over 0.5": "o0.5", "Over 1": "o1", "Over 1.5": "o1.5", "Over 2": "o2", "Over 3": "o3", "Over 4": "o4", "Over 5": "o5", "Over 10": "o10"}}, "Quick Ratio": {"prefix": "fa_quickratio", "option": {"Any": "", "High (>3)": "high", "Low (<0.5)": "low", "Under 1": "u1", "Under 0.5": "u0.5", "Over 0.5": "o0.5", "Over 1": "o1", "Over 1.5": "o1.5", "Over 2": "o2", "Over 3": "o3", "Over 4": "o4", "Over 5": "o5", "Over 10": "o10"}}, "LT Debt/Equity": {"prefix": "fa_ltdebteq", "option": {"Any": "", "High (>0.5)": "high", "Low (<0.1)": "low", "Under 1": "u1", "Under 0.9": "u0.9", "Under 0.8": "u0.8", "Under 0.7": "u0.7", "Under 0.6": "u0.6", "Under 0.5": "u0.5", "Under 0.4": "u0.4", "Under 0.3": "u0.3", "Under 0.2": "u0.2", "Under 0.1": "u0.1", "Over 0.1": "o0.1", "Over 0.2": "o0.2", "Over 0.3": "o0.3", "Over 0.4": "o0.4", "Over 0.5": "o0.5", "Over 0.6": "o0.6", "Over 0.7": "o0.7", "Over 0.8": "o0.8", "Over 0.9": "o0.9", "Over 1": "o1"}}, "Debt/Equity": {"prefix": "fa_debteq", "option": {"Any": "", "High (>0.5)": "high", "Low (<0.1)": "low", "Under 1": "u1", "Under 0.9": "u0.9", "Under 0.8": "u0.8", "Under 0.7": "u0.7", "Under 0.6": "u0.6", "Under 0.5": "u0.5", "Under 0.4": "u0.4", "Under 0.3": "u0.3", "Under 0.2": "u0.2", "Under 0.1": "u0.1", "Over 0.1": "o0.1", "Over 0.2": "o0.2", "Over 0.3": "o0.3", "Over 0.4": "o0.4", "Over 0.5": "o0.5", "Over 0.6": "o0.6", "Over 0.7": "o0.7", "Over 0.8": "o0.8", "Over 0.9": "o0.9", "Over 1": "o1"}}, "Gross Margin": {"prefix": "fa_grossmargin", "option": {"Any": "", "Positive (>0%)": "pos", "Negative (<0%)": "neg", "High (>50%)": "high", "Under 90%": "u90", "Under 80%": "u80", "Under 70%": "u70", "Under 60%": "u60", "Under 50%": "u50", "Under 45%": "u45", "Under 40%": "u40", "Under 35%": "u35", "Under 30%": "u30", "Under 25%": "u25", "Under 20%": "u20", "Under 15%": "u15", "Under 10%": "u10", "Under 5%": "u5", "Under 0%": "u0", "Under -10%": "u-10", "Under -20%": "u-20", "Under -30%": "u-30", "Under -50%": "u-50", "Under -70%": "u-70", "Under -100%": "u-100", "Over 0%": "o0", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30", "Over 35%": "o35", "Over 40%": "o40", "Over 45%": "o45", "Over 50%": "o50", "Over 60%": "o60", "Over 70%": "o70", "Over 80%": "o80", "Over 90%": "o90"}}, "Operating Margin": {"prefix": "fa_opermargin", "option": {"Any": "", "Positive (>0%)": "pos", "Negative (<0%)": "neg", "Very Negative (": "veryneg", "High (>25%)": "high", "Under 90%": "u90", "Under 80%": "u80", "Under 70%": "u70", "Under 60%": "u60", "Under 50%": "u50", "Under 45%": "u45", "Under 40%": "u40", "Under 35%": "u35", "Under 30%": "u30", "Under 25%": "u25", "Under 20%": "u20", "Under 15%": "u15", "Under 10%": "u10", "Under 5%": "u5", "Under 0%": "u0", "Under -10%": "u-10", "Under -20%": "u-20", "Under -30%": "u-30", "Under -50%": "u-50", "Under -70%": "u-70", "Under -100%": "u-100", "Over 0%": "o0", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30", "Over 35%": "o35", "Over 40%": "o40", "Over 45%": "o45", "Over 50%": "o50", "Over 60%": "o60", "Over 70%": "o70", "Over 80%": "o80", "Over 90%": "o90"}}, "Net Profit Margin": {"prefix": "fa_netmargin", "option": {"Any": "", "Positive (>0%)": "pos", "Negative (<0%)": "neg", "Very Negative (": "veryneg", "High (>20%)": "high", "Under 90%": "u90", "Under 80%": "u80", "Under 70%": "u70", "Under 60%": "u60", "Under 50%": "u50", "Under 45%": "u45", "Under 40%": "u40", "Under 35%": "u35", "Under 30%": "u30", "Under 25%": "u25", "Under 20%": "u20", "Under 15%": "u15", "Under 10%": "u10", "Under 5%": "u5", "Under 0%": "u0", "Under -10%": "u-10", "Under -20%": "u-20", "Under -30%": "u-30", "Under -50%": "u-50", "Under -70%": "u-70", "Under -100%": "u-100", "Over 0%": "o0", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30", "Over 35%": "o35", "Over 40%": "o40", "Over 45%": "o45", "Over 50%": "o50", "Over 60%": "o60", "Over 70%": "o70", "Over 80%": "o80", "Over 90%": "o90"}}, "Payout Ratio": {"prefix": "fa_payoutratio", "option": {"Any": "", "None (0%)": "none", "Positive (>0%)": "pos", "Low (<20%)": "low", "High (>50%)": "high", "Over 0%": "o0", "Over 10%": "o10", "Over 20%": "o20", "Over 30%": "o30", "Over 40%": "o40", "Over 50%": "o50", "Over 60%": "o60", "Over 70%": "o70", "Over 80%": "o80", "Over 90%": "o90", "Over 100%": "o100", "Under 10%": "u10", "Under 20%": "u20", "Under 30%": "u30", "Under 40%": "u40", "Under 50%": "u50", "Under 60%": "u60", "Under 70%": "u70", "Under 80%": "u80", "Under 90%": "u90", "Under 100%": "u100"}}, "InsiderOwnership": {"prefix": "sh_insiderown", "option": {"Any": "", "Low (<5%)": "low", "High (>30%)": "high", "Very High (>50%)": "veryhigh", "Over 10%": "o10", "Over 20%": "o20", "Over 30%": "o30", "Over 40%": "o40", "Over 50%": "o50", "Over 60%": "o60", "Over 70%": "o70", "Over 80%": "o80", "Over 90%": "o90"}}, "InsiderTransactions": {"prefix": "sh_insidertrans", "option": {"Any": "", "Very Negative (<20%)": "veryneg", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Very Positive (>20%)": "verypos", "Under -90%": "u-90", "Under -80%": "u-80", "Under -70%": "u-70", "Under -60%": "u-60", "Under -50%": "u-50", "Under -45%": "u-45", "Under -40%": "u-40", "Under -35%": "u-35", "Under -30%": "u-30", "Under -25%": "u-25", "Under -20%": "u-20", "Under -15%": "u-15", "Under -10%": "u-10", "Under -5%": "u-5", "Over +5%": "o5", "Over +10%": "o10", "Over +15%": "o15", "Over +20%": "o20", "Over +25%": "o25", "Over +30%": "o30", "Over +35%": "o35", "Over +40%": "o40", "Over +45%": "o45", "Over +50%": "o50", "Over +60%": "o60", "Over +70%": "o70", "Over +80%": "o80", "Over +90%": "o90"}}, "InstitutionalOwnership": {"prefix": "sh_instown", "option": {"Any": "", "Low (<5%)": "low", "High (>90%)": "high", "Under 90%": "u90", "Under 80%": "u80", "Under 70%": "u70", "Under 60%": "u60", "Under 50%": "u50", "Under 40%": "u40", "Under 30%": "u30", "Under 20%": "u20", "Under 10%": "u10", "Over 10%": "o10", "Over 20%": "o20", "Over 30%": "o30", "Over 40%": "o40", "Over 50%": "o50", "Over 60%": "o60", "Over 70%": "o70", "Over 80%": "o80", "Over 90%": "o90"}}, "InstitutionalTransactions": {"prefix": "sh_insttrans", "option": {"Any": "", "Very Negative (<20%)": "veryneg", "Negative (<0%)": "neg", "Positive (>0%)": "pos", "Very Positive (>20%)": "verypos", "Under -50%": "u-50", "Under -45%": "u-45", "Under -40%": "u-40", "Under -35%": "u-35", "Under -30%": "u-30", "Under -25%": "u-25", "Under -20%": "u-20", "Under -15%": "u-15", "Under -10%": "u-10", "Under -5%": "u-5", "Over +5%": "o5", "Over +10%": "o10", "Over +15%": "o15", "Over +20%": "o20", "Over +25%": "o25", "Over +30%": "o30", "Over +35%": "o35", "Over +40%": "o40", "Over +45%": "o45", "Over +50%": "o50"}}, "Float Short": {"prefix": "sh_short", "option": {"Any": "", "Low (<5%)": "low", "High (>20%)": "high", "Under 5%": "u5", "Under 10%": "u10", "Under 15%": "u15", "Under 20%": "u20", "Under 25%": "u25", "Under 30%": "u30", "Over 5%": "o5", "Over 10%": "o10", "Over 15%": "o15", "Over 20%": "o20", "Over 25%": "o25", "Over 30%": "o30"}}, "Analyst Recom.": {"prefix": "an_recom", "option": {"Any": "", "Strong Buy (1)": "strongbuy", "Buy or better": "buybetter", "Buy": "buy", "Hold or better": "holdbetter", "Hold": "hold", "Hold or worse": "holdworse", "Sell": "sell", "Sell or worse": "sellworse", "Strong Sell (5)": "strongsell"}}, "Option/Short": {"prefix": "sh_opt", "option": {"Any": "", "Optionable": "option", "Shortable": "short", "Optionable and shortable": "optionshort"}}, "Earnings Date": {"prefix": "earningsdate", "option": {"Any": "", "Today": "today", "Today Before Market Open": "todaybefore", "Today After Market Close": "todayafter", "Tomorrow": "tomorrow", "Tomorrow Before Market Open": "tomorrowbefore", "Tomorrow After Market Close": "tomorrowafter", "Yesterday": "yesterday", "Yesterday Before Market Open": "yesterdaybefore", "Yesterday After Market Close": "yesterdayafter", "Next 5 Days": "nextdays5", "Previous 5 Days": "prevdays5", "This Week": "thisweek", "Next Week": "nextweek", "Previous Week": "prevweek", "This Month": "thismonth"}}, "Performance": {"prefix": "ta_perf", "option": {"Any": "", "Today Up": "dup", "Today Down": "ddown", "Today -15%": "d15u", "Today -10%": "d10u", "Today -5%": "d5u", "Today +5%": "d5o", "Today +10%": "d10o", "Today +15%": "d15o", "Week -30%": "1w30u", "Week -20%": "1w20u", "Week -10%": "1w10u", "Week Down": "1wdown", "Week Up": "1wup", "Week +10%": "1w10o", "Week +20%": "1w20o", "Week +30%": "1w30o", "Month -50%": "4w50u", "Month -30%": "4w30u", "Month -20%": "4w20u", "Month -10%": "4w10u", "Month Down": "4wdown", "Month Up": "4wup", "Month +10%": "4w10o", "Month +20%": "4w20o", "Month +30%": "4w30o", "Month +50%": "4w50o", "Quarter -50%": "13w50u", "Quarter -30%": "13w30u", "Quarter -20%": "13w20u", "Quarter -10%": "13w10u", "Quarter Down": "13wdown", "Quarter Up": "13wup", "Quarter +10%": "13w10o", "Quarter +20%": "13w20o", "Quarter +30%": "13w30o", "Quarter +50%": "13w50o", "Half -75%": "26w75u", "Half -50%": "26w50u", "Half -30%": "26w30u", "Half -20%": "26w20u", "Half -10%": "26w10u", "Half Down": "26wdown", "Half Up": "26wup", "Half +10%": "26w10o", "Half +20%": "26w20o", "Half +30%": "26w30o", "Half +50%": "26w50o", "Half +100%": "26w100o", "Year -75%": "52w75u", "Year -50%": "52w50u", "Year -30%": "52w30u", "Year -20%": "52w20u", "Year -10%": "52w10u", "Year Down": "52wdown", "Year Up": "52wup", "Year +10%": "52w10o", "Year +20%": "52w20o", "Year +30%": "52w30o", "Year +50%": "52w50o", "Year +100%": "52w100o", "Year +200%": "52w200o", "Year +300%": "52w300o", "Year +500%": "52w500o", "YTD -75%": "ytd75u", "YTD -50%": "ytd50u", "YTD -30%": "ytd30u", "YTD -20%": "ytd20u", "YTD -10%": "ytd10u", "YTD -5%": "ytd5u", "YTD Down": "ytddown", "YTD Up": "ytdup", "YTD +5%": "ytd5o", "YTD +10%": "ytd10o", "YTD +20%": "ytd20o", "YTD +30%": "ytd30o", "YTD +50%": "ytd50o", "YTD +100%": "ytd100o"}}, "Performance 2": {"prefix": "ta_perf2", "option": {"Any": "", "Today Up": "dup", "Today Down": "ddown", "Today -15%": "d15u", "Today -10%": "d10u", "Today -5%": "d5u", "Today +5%": "d5o", "Today +10%": "d10o", "Today +15%": "d15o", "Week -30%": "1w30u", "Week -20%": "1w20u", "Week -10%": "1w10u", "Week Down": "1wdown", "Week Up": "1wup", "Week +10%": "1w10o", "Week +20%": "1w20o", "Week +30%": "1w30o", "Month -50%": "4w50u", "Month -30%": "4w30u", "Month -20%": "4w20u", "Month -10%": "4w10u", "Month Down": "4wdown", "Month Up": "4wup", "Month +10%": "4w10o", "Month +20%": "4w20o", "Month +30%": "4w30o", "Month +50%": "4w50o", "Quarter -50%": "13w50u", "Quarter -30%": "13w30u", "Quarter -20%": "13w20u", "Quarter -10%": "13w10u", "Quarter Down": "13wdown", "Quarter Up": "13wup", "Quarter +10%": "13w10o", "Quarter +20%": "13w20o", "Quarter +30%": "13w30o", "Quarter +50%": "13w50o", "Half -75%": "26w75u", "Half -50%": "26w50u", "Half -30%": "26w30u", "Half -20%": "26w20u", "Half -10%": "26w10u", "Half Down": "26wdown", "Half Up": "26wup", "Half +10%": "26w10o", "Half +20%": "26w20o", "Half +30%": "26w30o", "Half +50%": "26w50o", "Half +100%": "26w100o", "Year -75%": "52w75u", "Year -50%": "52w50u", "Year -30%": "52w30u", "Year -20%": "52w20u", "Year -10%": "52w10u", "Year Down": "52wdown", "Year Up": "52wup", "Year +10%": "52w10o", "Year +20%": "52w20o", "Year +30%": "52w30o", "Year +50%": "52w50o", "Year +100%": "52w100o", "Year +200%": "52w200o", "Year +300%": "52w300o", "Year +500%": "52w500o", "YTD -75%": "ytd75u", "YTD -50%": "ytd50u", "YTD -30%": "ytd30u", "YTD -20%": "ytd20u", "YTD -10%": "ytd10u", "YTD -5%": "ytd5u", "YTD Down": "ytddown", "YTD Up": "ytdup", "YTD +5%": "ytd5o", "YTD +10%": "ytd10o", "YTD +20%": "ytd20o", "YTD +30%": "ytd30o", "YTD +50%": "ytd50o", "YTD +100%": "ytd100o"}}, "Volatility": {"prefix": "ta_volatility", "option": {"Any": "", "Week - Over 3%": "wo3", "Week - Over 4%": "wo4", "Week - Over 5%": "wo5", "Week - Over 6%": "wo6", "Week - Over 7%": "wo7", "Week - Over 8%": "wo8", "Week - Over 9%": "wo9", "Week - Over 10%": "wo10", "Week - Over 12%": "wo12", "Week - Over 15%": "wo15", "Month - Over 2%": "mo2", "Month - Over 3%": "mo3", "Month - Over 4%": "mo4", "Month - Over 5%": "mo5", "Month - Over 6%": "mo6", "Month - Over 7%": "mo7", "Month - Over 8%": "mo8", "Month - Over 9%": "mo9", "Month - Over 10%": "mo10", "Month - Over 12%": "mo12", "Month - Over 15%": "mo15"}}, "RSI (14)": {"prefix": "ta_rsi", "option": {"Any": "", "Overbought (90)": "ob90", "Overbought (80)": "ob80", "Overbought (70)": "ob70", "Overbought (60)": "ob60", "Oversold (40)": "os40", "Oversold (30)": "os30", "Oversold (20)": "os20", "Oversold (10)": "os10", "Not Overbought (<60)": "nob60", "Not Overbought (<50)": "nob50", "Not Oversold (>50)": "nos50", "Not Oversold (>40)": "nos40"}}, "Gap": {"prefix": "ta_gap", "option": {"Any": "", "Up": "u", "Up 0%": "u0", "Up 1%": "u1", "Up 2%": "u2", "Up 3%": "u3", "Up 4%": "u4", "Up 5%": "u5", "Up 6%": "u6", "Up 7%": "u7", "Up 8%": "u8", "Up 9%": "u9", "Up 10%": "u10", "Up 15%": "u15", "Up 20%": "u20", "Down": "d", "Down 0%": "d0", "Down 1%": "d1", "Down 2%": "d2", "Down 3%": "d3", "Down 4%": "d4", "Down 5%": "d5", "Down 6%": "d6", "Down 7%": "d7", "Down 8%": "d8", "Down 9%": "d9", "Down 10%": "d10", "Down 15%": "d15", "Down 20%": "d20"}}, "20-Day Simple Moving Average": {"prefix": "ta_sma20", "option": {"Any": "", "Price below SMA20": "pb", "Price 10% below SMA20": "pb10", "Price 20% below SMA20": "pb20", "Price 30% below SMA20": "pb30", "Price 40% below SMA20": "pb40", "Price 50% below SMA20": "pb50", "Price above SMA20": "pa", "Price 10% above SMA20": "pa10", "Price 20% above SMA20": "pa20", "Price 30% above SMA20": "pa30", "Price 40% above SMA20": "pa40", "Price 50% above SMA20": "pa50", "Price crossed SMA20": "pc", "Price crossed SMA20 above": "pca", "Price crossed SMA20 below": "pcb", "SMA20 crossed SMA50": "cross50", "SMA20 crossed SMA50 above": "cross50a", "SMA20 crossed SMA50 below": "cross50b", "SMA20 crossed SMA200": "cross200", "SMA20 crossed SMA200 above": "cross200a", "SMA20 crossed SMA200 below": "cross200b", "SMA20 above SMA50": "sa50", "SMA20 below SMA50": "sb50", "SMA20 above SMA200": "sa200", "SMA20 below SMA200": "sb200"}}, "50-Day Simple Moving Average": {"prefix": "ta_sma50", "option": {"Any": "", "Price below SMA50": "pb", "Price 10% below SMA50": "pb10", "Price 20% below SMA50": "pb20", "Price 30% below SMA50": "pb30", "Price 40% below SMA50": "pb40", "Price 50% below SMA50": "pb50", "Price above SMA50": "pa", "Price 10% above SMA50": "pa10", "Price 20% above SMA50": "pa20", "Price 30% above SMA50": "pa30", "Price 40% above SMA50": "pa40", "Price 50% above SMA50": "pa50", "Price crossed SMA50": "pc", "Price crossed SMA50 above": "pca", "Price crossed SMA50 below": "pcb", "SMA50 crossed SMA20": "cross20", "SMA50 crossed SMA20 above": "cross20a", "SMA50 crossed SMA20 below": "cross20b", "SMA50 crossed SMA200": "cross200", "SMA50 crossed SMA200 above": "cross200a", "SMA50 crossed SMA200 below": "cross200b", "SMA50 above SMA20": "sa20", "SMA50 below SMA20": "sb20", "SMA50 above SMA200": "sa200", "SMA50 below SMA200": "sb200"}}, "200-Day Simple Moving Average": {"prefix": "ta_sma200", "option": {"Any": "", "Price below SMA200": "pb", "Price 10% below SMA200": "pb10", "Price 20% below SMA200": "pb20", "Price 30% below SMA200": "pb30", "Price 40% below SMA200": "pb40", "Price 50% below SMA200": "pb50", "Price 60% below SMA200": "pb60", "Price 70% below SMA200": "pb70", "Price 80% below SMA200": "pb80", "Price 90% below SMA200": "pb90", "Price above SMA200": "pa", "Price 10% above SMA200": "pa10", "Price 20% above SMA200": "pa20", "Price 30% above SMA200": "pa30", "Price 40% above SMA200": "pa40", "Price 50% above SMA200": "pa50", "Price 60% above SMA200": "pa60", "Price 70% above SMA200": "pa70", "Price 80% above SMA200": "pa80", "Price 90% above SMA200": "pa90", "Price 100% above SMA200": "pa100", "Price crossed SMA200": "pc", "Price crossed SMA200 above": "pca", "Price crossed SMA200 below": "pcb", "SMA200 crossed SMA20": "cross20", "SMA200 crossed SMA20 above": "cross20a", "SMA200 crossed SMA20 below": "cross20b", "SMA200 crossed SMA50": "cross50", "SMA200 crossed SMA50 above": "cross50a", "SMA200 crossed SMA50 below": "cross50b", "SMA200 above SMA20": "sa20", "SMA200 below SMA20": "sb20", "SMA200 above SMA50": "sa50", "SMA200 below SMA50": "sb50"}}, "Change": {"prefix": "ta_change", "option": {"Any": "", "Up": "u", "Up 1%": "u1", "Up 2%": "u2", "Up 3%": "u3", "Up 4%": "u4", "Up 5%": "u5", "Up 6%": "u6", "Up 7%": "u7", "Up 8%": "u8", "Up 9%": "u9", "Up 10%": "u10", "Up 15%": "u15", "Up 20%": "u20", "Down": "d", "Down 1%": "d1", "Down 2%": "d2", "Down 3%": "d3", "Down 4%": "d4", "Down 5%": "d5", "Down 6%": "d6", "Down 7%": "d7", "Down 8%": "d8", "Down 9%": "d9", "Down 10%": "d10", "Down 15%": "d15", "Down 20%": "d20"}}, "Change from Open": {"prefix": "ta_changeopen", "option": {"Any": "", "Up": "u", "Up 1%": "u1", "Up 2%": "u2", "Up 3%": "u3", "Up 4%": "u4", "Up 5%": "u5", "Up 6%": "u6", "Up 7%": "u7", "Up 8%": "u8", "Up 9%": "u9", "Up 10%": "u10", "Up 15%": "u15", "Up 20%": "u20", "Down": "d", "Down 1%": "d1", "Down 2%": "d2", "Down 3%": "d3", "Down 4%": "d4", "Down 5%": "d5", "Down 6%": "d6", "Down 7%": "d7", "Down 8%": "d8", "Down 9%": "d9", "Down 10%": "d10", "Down 15%": "d15", "Down 20%": "d20"}}, "20-Day High/Low": {"prefix": "ta_highlow20d", "option": {"Any": "", "New High": "nh", "New Low": "nl", "5% or more below High": "b5h", "10% or more below High": "b10h", "15% or more below High": "b15h", "20% or more below High": "b20h", "30% or more below High": "b30h", "40% or more below High": "b40h", "50% or more below High": "b50h", "0-3% below High": "b0to3h", "0-5% below High": "b0to5h", "0-10% below High": "b0to10h", "5% or more above Low": "a5h", "10% or more above Low": "a10h", "15% or more above Low": "a15h", "20% or more above Low": "a20h", "30% or more above Low": "a30h", "40% or more above Low": "a40h", "50% or more above Low": "a50h", "0-3% above Low": "a0to3h", "0-5% above Low": "a0to5h", "0-10% above Low": "a0to10h"}}, "50-Day High/Low": {"prefix": "ta_highlow50d", "option": {"Any": "", "New High": "nh", "New Low": "nl", "5% or more below High": "b5h", "10% or more below High": "b10h", "15% or more below High": "b15h", "20% or more below High": "b20h", "30% or more below High": "b30h", "40% or more below High": "b40h", "50% or more below High": "b50h", "0-3% below High": "b0to3h", "0-5% below High": "b0to5h", "0-10% below High": "b0to10h", "5% or more above Low": "a5h", "10% or more above Low": "a10h", "15% or more above Low": "a15h", "20% or more above Low": "a20h", "30% or more above Low": "a30h", "40% or more above Low": "a40h", "50% or more above Low": "a50h", "0-3% above Low": "a0to3h", "0-5% above Low": "a0to5h", "0-10% above Low": "a0to10h"}}, "52-Week High/Low": {"prefix": "ta_highlow52w", "option": {"Any": "", "New High": "nh", "New Low": "nl", "5% or more below High": "b5h", "10% or more below High": "b10h", "15% or more below High": "b15h", "20% or more below High": "b20h", "30% or more below High": "b30h", "40% or more below High": "b40h", "50% or more below High": "b50h", "60% or more below High": "b60h", "70% or more below High": "b70h", "80% or more below High": "b80h", "90% or more below High": "b90h", "0-3% below High": "b0to3h", "0-5% below High": "b0to5h", "0-10% below High": "b0to10h", "5% or more above Low": "a5h", "10% or more above Low": "a10h", "15% or more above Low": "a15h", "20% or more above Low": "a20h", "30% or more above Low": "a30h", "40% or more above Low": "a40h", "50% or more above Low": "a50h", "60% or more above Low": "a60h", "70% or more above Low": "a70h", "80% or more above Low": "a80h", "90% or more above Low": "a90h", "100% or more above Low": "a100h", "120% or more above Low": "a120h", "150% or more above Low": "a150h", "200% or more above Low": "a200h", "300% or more above Low": "a300h", "500% or more above Low": "a500h", "0-3% above Low": "a0to3h", "0-5% above Low": "a0to5h", "0-10% above Low": "a0to10h"}}, "Pattern": {"prefix": "ta_pattern", "option": {"Any": "", "Horizontal S/R": "horizontal", "Horizontal S/R (Strong)": "horizontal2", "TL Resistance": "tlresistance", "TL Resistance (Strong)": "tlresistance2", "TL Support": "tlsupport", "TL Support (Strong)": "tlsupport2", "Wedge Up": "wedgeup", "Wedge Up (Strong)": "wedgeup2", "Wedge Down": "wedgedown", "Wedge Down (Strong)": "wedgedown2", "Triangle Ascending": "wedgeresistance", "Triangle Ascending (Strong)": "wedgeresistance2", "Triangle Descending": "wedgesupport", "Triangle Descending (Strong)": "wedgesupport2", "Wedge": "wedge", "Wedge (Strong)": "wedge2", "Channel Up": "channelup", "Channel Up (Strong)": "channelup2", "Channel Down": "channeldown", "Channel Down (Strong)": "channeldown2", "Channel": "channel", "Channel (Strong)": "channel2", "Double Top": "doubletop", "Double Bottom": "doublebottom", "Multiple Top": "multipletop", "Multiple Bottom": "multiplebottom", "Head & Shoulders": "headandshoulders", "Head & Shoulders Inverse": "headandshouldersinv"}}, "Candlestick": {"prefix": "ta_candlestick", "option": {"Any": "", "Long Lower Shadow": "lls", "Long Upper Shadow": "lus", "Hammer": "h", "Inverted Hammer": "ih", "Spinning Top White": "stw", "Spinning Top Black": "stb", "Doji": "d", "Dragonfly Doji": "dd", "Gravestone Doji": "gd", "Marubozu White": "mw", "Marubozu Black": "mb"}}, "Beta": {"prefix": "ta_beta", "option": {"Any": "", "Under 0": "u0", "Under 0.5": "u0.5", "Under 1": "u1", "Under 1.5": "u1.5", "Under 2": "u2", "Over 0": "o0", "Over 0.5": "o0.5", "Over 1": "o1", "Over 1.5": "o1.5", "Over 2": "o2", "Over 2.5": "o2.5", "Over 3": "o3", "Over 4": "o4", "0 to 0.5": "0to0.5", "0 to 1": "0to1", "0.5 to 1": "0.5to1", "0.5 to 1.5": "0.5to1.5", "1 to 1.5": "1to1.5", "1 to 2": "1to2"}}, "Average True Range": {"prefix": "ta_averagetruerange", "option": {"Any": "", "Over 0.25": "o0.25", "Over 0.5": "o0.5", "Over 0.75": "o0.75", "Over 1": "o1", "Over 1.5": "o1.5", "Over 2": "o2", "Over 2.5": "o2.5", "Over 3": "o3", "Over 3.5": "o3.5", "Over 4": "o4", "Over 4.5": "o4.5", "Over 5": "o5", "Under 0.25": "u0.25", "Under 0.5": "u0.5", "Under 0.75": "u0.75", "Under 1": "u1", "Under 1.5": "u1.5", "Under 2": "u2", "Under 2.5": "u2.5", "Under 3": "u3", "Under 3.5": "u3.5", "Under 4": "u4", "Under 4.5": "u4.5", "Under 5": "u5"}}, "Average Volume": {"prefix": "sh_avgvol", "option": {"Any": "", "Under 50K": "u50", "Under 100K": "u100", "Under 500K": "u500", "Under 750K": "u750", "Under 1M": "u1000", "Over 50K": "o50", "Over 100K": "o100", "Over 200K": "o200", "Over 300K": "o300", "Over 400K": "o400", "Over 500K": "o500", "Over 750K": "o750", "Over 1M": "o1000", "Over 2M": "o2000", "100K to 500K": "100to500", "100K to 1M": "100to1000", "500K to 1M": "500to1000", "500K to 10M": "500to10000"}}, "Relative Volume": {"prefix": "sh_relvol", "option": {"Any": "", "Over 10": "o10", "Over 5": "o5", "Over 3": "o3", "Over 2": "o2", "Over 1.5": "o1.5", "Over 1": "o1", "Over 0.75": "o0.75", "Over 0.5": "o0.5", "Over 0.25": "o0.25", "Under 2": "u2", "Under 1.5": "u1.5", "Under 1": "u1", "Under 0.75": "u0.75", "Under 0.5": "u0.5", "Under 0.25": "u0.25", "Under 0.1": "u0.1"}}, "Current Volume": {"prefix": "sh_curvol", "option": {"Any": "", "Under 50K": "u50", "Under 100K": "u100", "Under 500K": "u500", "Under 750K": "u750", "Under 1M": "u1000", "Over 0": "o0", "Over 50K": "o50", "Over 100K": "o100", "Over 200K": "o200", "Over 300K": "o300", "Over 400K": "o400", "Over 500K": "o500", "Over 750K": "o750", "Over 1M": "o1000", "Over 2M": "o2000", "Over 5M": "o5000", "Over 10M": "o10000", "Over 20M": "o20000"}}, "Price": {"prefix": "sh_price", "option": {"Any": "", "Under $1": "u1", "Under $2": "u2", "Under $3": "u3", "Under $4": "u4", "Under $5": "u5", "Under $7": "u7", "Under $10": "u10", "Under $15": "u15", "Under $20": "u20", "Under $30": "u30", "Under $40": "u40", "Under $50": "u50", "Over $1": "o1", "Over $2": "o2", "Over $3": "o3", "Over $4": "o4", "Over $5": "o5", "Over $7": "o7", "Over $10": "o10", "Over $15": "o15", "Over $20": "o20", "Over $30": "o30", "Over $40": "o40", "Over $50": "o50", "Over $60": "o60", "Over $70": "o70", "Over $80": "o80", "Over $90": "o90", "Over $100": "o100", "$1 to $5": "1to5", "$1 to $10": "1to10", "$1 to $20": "1to20", "$5 to $10": "5to10", "$5 to $20": "5to20", "$5 to $50": "5to50", "$10 to $20": "10to20", "$10 to $50": "10to50", "$20 to $50": "20to50", "$50 to $100": "50to100"}}, "Target Price": {"prefix": "targetprice", "option": {"Any": "", "50% Above Price": "a50", "40% Above Price": "a40", "30% Above Price": "a30", "20% Above Price": "a20", "10% Above Price": "a10", "5% Above Price": "a5", "Above Price": "above", "Below Price": "below", "5% Below Price": "b5", "10% Below Price": "b10", "20% Below Price": "b20", "30% Below Price": "b30", "40% Below Price": "b40", "50% Below Price": "b50"}}, "IPO Date": {"prefix": "ipodate", "option": {"Any": "", "Today": "today", "Yesterday": "yesterday", "In the last week": "prevweek", "In the last month": "prevmonth", "In the last quarter": "prevquarter", "In the last year": "prevyear", "In the last 2 years": "prev2yrs", "In the last 3 years": "prev3yrs", "In the last 5 years": "prev5yrs", "More than a year ago": "more1", "More than 5 years ago": "more5", "More than 10 years ago": "more10", "More than 15 years ago": "more15", "More than 20 years ago": "more20", "More than 25 years ago": "more25"}}, "Shares Outstanding": {"prefix": "sh_outstanding", "option": {"Any": "", "Under 1M": "u1", "Under 5M": "u5", "Under 10M": "u10", "Under 20M": "u20", "Under 50M": "u50", "Under 100M": "u100", "Over 1M": "o1", "Over 2M": "o2", "Over 5M": "o5", "Over 10M": "o10", "Over 20M": "o20", "Over 50M": "o50", "Over 100M": "o100", "Over 200M": "o200", "Over 500M": "o500", "Over 1000M": "o1000"}}, "Float": {"prefix": "sh_float", "option": {"Any": "", "Under 1M": "u1", "Under 5M": "u5", "Under 10M": "u10", "Under 20M": "u20", "Under 50M": "u50", "Under 100M": "u100", "Over 1M": "o1", "Over 2M": "o2", "Over 5M": "o5", "Over 10M": "o10", "Over 20M": "o20", "Over 50M": "o50", "Over 100M": "o100", "Over 200M": "o200", "Over 500M": "o500", "Over 1000M": "o1000"}}}, "order": {"Ticker": "o=ticker", "Company": "o=company", "Sector": "o=sector", "Industry": "o=industry", "Country": "o=country", "Market Cap.": "o=marketcap", "Price/Earnings": "o=pe", "Forward Price/Earnings": "o=forwardpe", "PEG (Price/Earnings/Growth)": "o=peg", "Price/Sales": "o=ps", "Price/Book": "o=pb", "Price/Cash": "o=pc", "Price/Free Cash Flow": "o=pfcf", "Dividend Yield": "o=dividendyield", "Payout Ratio": "o=payoutratio", "EPS (ttm)": "o=eps", "EPS growth this year": "o=epsyoy", "EPS growth next year": "o=epsyoy1", "EPS growth past 5 years": "o=eps5years", "EPS growth next 5 years": "o=estltgrowth", "Sales growth past 5 years": "o=sales5years", "EPS growth qtr over qtr": "o=epsqoq", "Sales growth qtr over qtr": "o=salesqoq", "Shares Outstanding": "o=sharesoutstanding2", "Shares Float": "o=sharesfloat", "Insider Ownership": "o=insiderown", "Insider Transactions": "o=insidertrans", "Institutional Ownership": "o=instown", "Institutional Transactions": "o=insttrans", "Short Interest Share": "o=shortinterestshare", "Short Interest Ratio": "o=shortinterestratio", "Earnings Date": "o=earningsdate", "Return on Assets": "o=roa", "Return on Equity": "o=roe", "Return on Investment": "o=roi", "Current Ratio": "o=curratio", "Quick Ratio": "o=quickratio", "LT Debt/Equity": "o=ltdebteq", "Total Debt/Equity": "o=debteq", "Gross Margin": "o=grossmargin", "Operating Margin": "o=opermargin", "Net Profit Margin": "o=netmargin", "Analyst Recommendation": "o=recom", "Performance (Week)": "o=perf1w", "Performance (Month)": "o=perf4w", "Performance (Quarter)": "o=perf13w", "Performance (Half Year)": "o=perf26w", "Performance (Year)": "o=perf52w", "Performance (Year To Date)": "o=perfytd", "Beta": "o=beta", "Average True Range": "o=averagetruerange", "Volatility (Week)": "o=volatility1w", "Volatility (Month)": "o=volatility4w", "20-Day SMA (Relative)": "o=sma20", "50-Day SMA (Relative)": "o=sma50", "200-Day SMA (Relative)": "o=sma200", "50-Day High (Relative)": "o=high50d", "50-Day Low (Relative)": "o=low50d", "52-Week High (Relative)": "o=high52w", "52-Week Low (Relative)": "o=low52w", "Relative Strength Index (14)": "o=rsi", "Average Volume (3 Month)": "o=averagevolume", "Relative Volume": "o=relativevolume", "Change": "o=change", "Change from Open": "o=changeopen", "Gap": "o=gap", "Volume": "o=volume", "Price": "o=price", "Target Price": "o=targetprice", "IPO Date": "o=ipodate"}}