"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schedule = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Schedule for scheduled scaling actions
 */
class Schedule {
    constructor() { }
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that Application AutoScaling will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        if (duration.isUnresolved()) {
            const validDurationUnit = ['minute', 'minutes', 'hour', 'hours', 'day', 'days'];
            if (!validDurationUnit.includes(duration.unitLabel())) {
                throw new Error("Allowed units for scheduling are: 'minute', 'minutes', 'hour', 'hours', 'day' or 'days'");
            }
            return new LiteralSchedule(`rate(${duration.formatTokenToNumber()})`);
        }
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Construct a Schedule from a moment in time
     */
    static at(moment) {
        return new LiteralSchedule(`at(${formatISO(moment)})`);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_applicationautoscaling_CronOptions(options);
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error('Cannot supply both \'day\' and \'weekDay\', use at most one');
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new class extends Schedule {
            constructor() {
                super(...arguments);
                this.expressionString = `cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`;
            }
            _bind(scope) {
                if (!options.minute) {
                    core_1.Annotations.of(scope).addWarning('cron: If you don\'t pass \'minute\', by default the event runs every minute. Pass \'minute: \'*\'\' if that\'s what you intend, or \'minute: 0\' to run once per hour instead.');
                }
                return new LiteralSchedule(this.expressionString);
            }
        };
    }
}
exports.Schedule = Schedule;
_a = JSII_RTTI_SYMBOL_1;
Schedule[_a] = { fqn: "@aws-cdk/aws-applicationautoscaling.Schedule", version: "1.150.0" };
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
    _bind() { }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
function formatISO(date) {
    if (!date) {
        return undefined;
    }
    return date.getUTCFullYear() +
        '-' + pad(date.getUTCMonth() + 1) +
        '-' + pad(date.getUTCDate()) +
        'T' + pad(date.getUTCHours()) +
        ':' + pad(date.getUTCMinutes()) +
        ':' + pad(date.getUTCSeconds());
    function pad(num) {
        if (num < 10) {
            return '0' + num;
        }
        return num;
    }
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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