"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
    /**
     * Newer type of long term storage with higher throughput tiers.
     * Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_2"] = "PERSISTENT_2";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_fsx_LustreFileSystemProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_b = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _b === void 0 ? void 0 : _b.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_fsx_FileSystemAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLustreFileSystemAttributes);
            }
            throw error;
        }
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1 && deploymentType !== LustreDeploymentType.PERSISTENT_2) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1/PERSISTENT_2 deployment types, received: ' + deploymentType);
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_1) {
            if (![50, 100, 200].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB for PERSISTENT_1 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
        if (deploymentType === LustreDeploymentType.PERSISTENT_2) {
            if (![125, 250, 500, 1000].includes(perUnitStorageThroughput)) {
                throw new Error('perUnitStorageThroughput must be 125, 250, 500 or 1000 MB/s/TiB for PERSISTENT_2 deployment type, got: ' + perUnitStorageThroughput);
            }
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
_a = JSII_RTTI_SYMBOL_1;
LustreFileSystem[_a] = { fqn: "@aws-cdk/aws-fsx.LustreFileSystem", version: "1.153.1" };
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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