from unittest import TestCase
import numpy as np

from orange_cb_recsys.content_analyzer.field_content_production_techniques.embedding_technique.embedding_source import \
    GensimDownloader, Wikipedia2VecDownloader, BinaryFile


class TestGensimDownloader(TestCase):
    def test_load(self):
        self.skipTest("SLOW")
        source = GensimDownloader('glove-twitter-25')
        result = source.load("title plot")

        expected = np.ndarray(shape=(2, 25))
        expected[0, :] = [8.50130022e-01, 4.52620000e-01, -7.05750007e-03,
                          -8.77380013e-01, 4.24479991e-01, -8.36589992e-01,
                          8.04159999e-01, 3.74080002e-01, 4.30849999e-01,
                          -6.39360011e-01, 1.19390003e-01, 1.13419998e+00,
                          -3.20650005e+00, 9.31460023e-01, 3.65420014e-01,
                          -3.19309998e-03, 1.97899997e-01, -3.29540014e-01,
                          2.96719998e-01, 4.88689989e-01, -1.37870002e+00,
                          7.52340019e-01, 2.03339994e-01, -6.79979980e-01,
                          -8.91939998e-01]
        expected[1, :] = [7.26029992e-01, 1.46909997e-01, 1.05829999e-01,
                          2.84680009e-01, 2.31950000e-01, -7.86419988e-01,
                          1.33580005e+00, -8.31910014e-01, 4.39669997e-01,
                          -3.01629990e-01, 2.93879986e-01, 4.53700006e-01,
                          -2.18440008e+00, 2.45710000e-01, 3.21599990e-01,
                          6.92149997e-01, 6.65279984e-01, 5.34259975e-01,
                          3.30240000e-03, -4.91389990e-01, -2.80680005e-02,
                          6.41950011e-01, -9.63369980e-02, -9.50479984e-01,
                          -3.88559997e-01]

        self.assertTrue(np.allclose(result, expected))


class TestWikipedia2VecDownloader(TestCase):
    def test_load(self):
        self.skipTest("SLOW")

        source = Wikipedia2VecDownloader('../../../../datasets/wikipedia2vec/MODEL_FILE')
        expected = np.ndarray(shape=(2, 100))
        expected[0, :] = [0.9113617, -0.16422762, 0.29195637, -0.5079845, -0.24498372,
                          -0.23359998, 0.11357953, -0.26665968, -0.42770126, 0.14040668,
                          -0.23625416, -0.3995287, 0.2802729, 1.4072003, -0.6991014,
                          0.35159022, 0.1359734, -0.20122118, -0.50963837, 0.36504865,
                          0.14292698, -0.25394648, 0.08690099, 0.02234293, -0.16732648,
                          -0.02937562, 0.04579485, -0.5521207, 0.35329795, -0.07194141,
                          0.3464605, -0.19305374, -0.3200527, 0.18569043, -0.13610989,
                          -0.2870403, 0.5056123, -0.4935016, 0.36145186, -0.328249,
                          -0.38885072, 0.36092448, -0.41003436, -0.49896812, -0.97277784,
                          0.06608371, 0.2831374, -0.02973103, -0.93057984, -0.6563575,
                          0.0017389, -0.13686186, 1.1041245, 0.8039884, 0.43903843,
                          -0.32882544, -0.6202658, 0.04793411, -0.08302873, 0.34030053,
                          -0.04389457, -0.22238144, -0.3966373, -0.19364174, 0.8547167,
                          -0.09434482, 0.01979835, -0.42662966, 0.90295327, -0.10957272,
                          -0.01022598, -0.7432095, 0.6144721, 0.5314728, 0.7246612,
                          -0.10316867, -0.3097405, 0.29169014, -0.16370276, -0.69296,
                          -0.19743642, -0.11537066, 0.30410182, 0.10588095, 0.3580293,
                          -0.31855756, -0.61124784, -0.13916153, -0.14975981, -0.14217573,
                          -0.11661967, -0.16441974, -0.24196252, 0.39884397, -0.5931838,
                          -0.04866777, 0.50146234, -0.21017258, -0.2786765, 0.36317846]

        expected[1, :] = [0.28312996, -0.1788087, 0.03203144, 0.11723588, -0.39848948,
                          0.31468883, -0.3638821, -0.36848024, -0.4647756, -0.27375367,
                          0.43275204, 0.24046534, 0.3181958, 0.91019815, -0.08030581,
                          -0.02831636, 0.3622169, -0.14887914, -0.2134587, 0.5437428,
                          0.41059586, -0.29065675, 0.08602623, 0.41740757, -0.06743675,
                          -0.452779, -0.15264812, -0.03262068, -0.31282657, 0.34982398,
                          0.06633418, 0.15853819, 0.2788552, 0.6395291, 0.48155418,
                          -0.47929102, 0.30027667, -0.1407314, -0.40413344, -0.55801684,
                          -0.37716717, -0.4143479, 0.10795735, 0.2482185, -0.41543058,
                          -0.33412656, -0.38177335, -0.37114695, 0.23572417, -0.02307804,
                          -0.36781183, -0.00912181, 0.57776487, 0.0338354, 0.18606435,
                          -0.7382905, -0.33887747, 0.13776354, 0.19354428, 0.1029704,
                          -0.642619, 0.13491046, 0.36207572, -0.26311886, 0.22916538,
                          -0.79787374, 0.56587845, -0.02978418, 1.0875645, 0.0167608,
                          -0.07216629, 0.80384785, 0.336056, 0.36411795, 0.09852715,
                          0.17258792, 0.72595966, -0.09887329, -0.15544432, -0.42295107,
                          -0.03302035, 0.16508514, 0.30657756, 0.07338545, -0.23908757,
                          -0.27414083, -0.19644827, 0.2025203, -0.0500539, 0.13785328,
                          0.18675512, 0.10200696, 0.09887865, 0.34445098, -0.45849535,
                          0.11825627, 0.00971196, -0.4790937, 0.05219276, 0.04766607]

        result = source.load(["text", "news"])
        self.assertTrue(np.allclose(result, expected))


class TestBinaryFile(TestCase):
    def test_load(self):
        self.skipTest("SLOW")
        source = BinaryFile('../../../datasets/word2vec/google_news.bin', "Word2Vec")
        result = source.load("text news")
        expected = np.ndarray(shape=(2, 300))
        expected[0, :] = [0.27929688, -0.08447266, -0.04150391, 0.12011719, -0.01794434,
                          0.12353516, 0.1328125, -0.08251953, 0.11572266, 0.10693359,
                          -0.08056641, -0.01397705, -0.13671875, 0.11230469, -0.0546875,
                          0.109375, 0.12060547, 0.06396484, -0.02001953, -0.41992188,
                          0.18652344, 0.14550781, -0.08837891, -0.23828125, -0.21386719,
                          -0.24902344, -0.06884766, 0.10693359, 0.20996094, 0.04589844,
                          -0.15527344, -0.29882812, -0.11376953, -0.02416992, 0.01477051,
                          0.01489258, -0.06835938, 0.33203125, -0.10888672, -0.14160156,
                          0.05639648, 0.05078125, 0.19921875, 0.40820312, 0.04833984,
                          0.07714844, 0.06689453, -0.171875, -0.28515625, -0.203125,
                          -0.2109375, -0.18164062, 0.03369141, -0.21777344, 0.25976562,
                          0.15234375, 0.13964844, -0.22167969, -0.1484375, 0.03808594,
                          0.02990723, -0.10644531, -0.10595703, 0.11181641, -0.02770996,
                          0.19726562, -0.28320312, 0.08886719, 0.06347656, 0.21386719,
                          0.16308594, 0.08300781, -0.03088379, 0.09716797, 0.10253906,
                          -0.17089844, 0.22265625, 0.38085938, 0.23535156, -0.10839844,
                          0.08642578, 0.06494141, 0.21972656, 0.02978516, 0.359375,
                          0.00282288, 0.03564453, -0.15332031, 0.07666016, 0.13378906,
                          -0.03369141, -0.10839844, -0.01031494, -0.22070312, -0.05810547,
                          -0.078125, -0.09570312, -0.03613281, 0.14160156, 0.17578125,
                          -0.19433594, 0.06152344, -0.08300781, 0.05273438, 0.06982422,
                          0.32226562, 0.02722168, -0.01245117, -0.08300781, 0.19824219,
                          -0.08544922, 0.14550781, 0.04541016, -0.10400391, 0.34570312,
                          0.02416992, 0.12158203, -0.25585938, 0.28710938, 0.17675781,
                          0.10791016, -0.23144531, -0.34179688, 0.20605469, 0.09814453,
                          -0.17089844, -0.31640625, -0.10253906, -0.14648438, 0.10839844,
                          -0.296875, -0.11474609, -0.11767578, 0.00897217, 0.05151367,
                          0.05200195, -0.05322266, 0.10449219, 0.16015625, -0.2578125,
                          0.15527344, -0.33789062, -0.02099609, -0.18066406, 0.06982422,
                          -0.08447266, 0.02819824, 0.12695312, -0.08642578, -0.06396484,
                          0.23339844, 0.1875, 0.05175781, -0.03564453, 0.27734375,
                          -0.48046875, -0.22363281, 0.125, -0.23535156, 0.0625,
                          -0.11132812, 0.41796875, 0.17285156, -0.01159668, 0.125,
                          -0.06176758, 0.03637695, -0.1875, -0.30078125, -0.14746094,
                          -0.03088379, 0.11035156, -0.22558594, -0.16601562, 0.11425781,
                          -0.23632812, 0.02148438, -0.12988281, -0.23242188, -0.16992188,
                          -0.22558594, -0.2734375, 0.00686646, 0.04370117, 0.203125,
                          0.06640625, -0.20605469, 0.25585938, 0.01251221, 0.11279297,
                          0.06738281, 0.02929688, -0.19140625, -0.17480469, -0.00854492,
                          0.28710938, 0.11621094, -0.4921875, 0.08398438, 0.03491211,
                          -0.06835938, -0.12695312, -0.2109375, -0.03637695, -0.02185059,
                          0.06933594, 0.0859375, -0.06542969, 0.02160645, -0.0378418,
                          0.17773438, 0.00854492, 0.08007812, -0.12597656, -0.12011719,
                          0.04492188, 0.03320312, -0.05249023, -0.11523438, 0.0189209,
                          -0.21777344, 0.15234375, 0.05981445, -0.28515625, 0.06347656,
                          0.08154297, -0.10449219, 0.12060547, 0.0390625, 0.17285156,
                          0.11132812, -0.17285156, -0.04272461, -0.02478027, 0.04614258,
                          -0.16992188, 0.03320312, 0.08886719, 0.24707031, -0.00491333,
                          0.02819824, 0.05859375, 0.00193787, -0.05126953, 0.328125,
                          0.13671875, 0.10644531, 0.39648438, -0.22851562, 0.234375,
                          0.08642578, 0.16699219, -0.24121094, 0.06933594, -0.03295898,
                          -0.44335938, 0.06884766, -0.34375, -0.03491211, -0.02880859,
                          0.17578125, 0.06591797, -0.03015137, -0.171875, -0.04858398,
                          0.08837891, 0.296875, -0.30859375, -0.13574219, -0.44921875,
                          0.0177002, 0.234375, -0.03320312, 0.09228516, 0.05712891,
                          -0.04760742, 0.06445312, -0.03320312, -0.09228516, 0.18554688,
                          -0.0703125, 0.23632812, -0.18164062, 0.21386719, 0.28515625,
                          0.02648926, 0.09814453, 0.11767578, 0.04663086, 0.2734375,
                          -0.04394531, -0.07666016, 0.01269531, -0.23046875, 0.04077148,
                          -0.06835938, 0.11767578, -0.22070312, 0.00878906, -0.00051498]

        expected[1, :] = [-0.13867188, 0.04370117, -0.13085938, -0.16796875, -0.06054688,
                          -0.07080078, 0.00854492, -0.09960938, 0.19628906, 0.14648438,
                          -0.23046875, -0.09619141, -0.140625, 0.25585938, -0.16699219,
                          0.14257812, -0.16992188, 0.06884766, 0.23925781, -0.28515625,
                          0.17089844, 0.17089844, 0.05615234, -0.06445312, 0.13867188,
                          0.32226562, 0.05078125, 0.27148438, -0.00564575, 0.02587891,
                          -0.05639648, 0.03125, 0.171875, -0.18261719, 0.05029297,
                          -0.01062012, -0.02600098, 0.20019531, -0.07568359, 0.125,
                          0.01300049, -0.17089844, -0.09667969, 0.21972656, 0.06494141,
                          0.24804688, 0.11083984, 0.0267334, -0.10986328, 0.17578125,
                          0.328125, -0.11865234, 0.03222656, 0.30273438, 0.09179688,
                          -0.05224609, -0.27539062, -0.046875, 0.20996094, -0.20410156,
                          0.10595703, -0.08398438, -0.09765625, -0.16015625, -0.0045166,
                          0.09228516, 0.05737305, -0.04418945, 0.06445312, 0.16992188,
                          -0.06396484, 0.02160645, 0.01672363, 0.01916504, -0.14453125,
                          0.22949219, -0.14746094, 0.05859375, 0.00610352, -0.08544922,
                          0.34960938, 0.14355469, 0.07128906, -0.0534668, 0.01928711,
                          -0.25, -0.21972656, -0.2421875, 0.01611328, -0.01818848,
                          0.31054688, 0.08154297, 0.171875, 0.05664062, -0.04736328,
                          -0.07421875, 0.00244141, 0.11035156, 0.31445312, -0.05175781,
                          -0.03710938, 0.00430298, 0.08496094, -0.14453125, -0.06347656,
                          0.06298828, -0.07861328, -0.12695312, 0.15039062, -0.20214844,
                          0.05639648, 0.13769531, 0.16992188, 0.01782227, -0.07617188,
                          0.04760742, 0.16015625, 0.20214844, 0.33007812, 0.01123047,
                          -0.06787109, -0.14453125, -0.234375, 0.06542969, 0.06933594,
                          0.00680542, 0.04785156, 0.00430298, 0.19433594, -0.15625,
                          -0.34375, -0.21777344, 0.09082031, 0.05957031, 0.07666016,
                          -0.01806641, -0.00915527, 0.03295898, -0.03027344, -0.04833984,
                          0.36132812, -0.08544922, 0.00897217, -0.15917969, 0.03344727,
                          -0.31445312, 0.13378906, -0.10839844, -0.10693359, -0.16503906,
                          0.13378906, 0.12597656, -0.33398438, 0.01257324, -0.08105469,
                          -0.05151367, -0.0071106, -0.1015625, -0.18652344, -0.07373047,
                          -0.10498047, 0.203125, -0.17578125, 0.26953125, 0.00927734,
                          -0.05126953, -0.04150391, 0.01013184, 0.31640625, -0.14550781,
                          0.25, -0.13476562, 0.10693359, -0.2578125, -0.01000977,
                          -0.10351562, 0.27539062, -0.15722656, 0.06298828, -0.23828125,
                          0.09179688, -0.27148438, 0.18457031, -0.24023438, 0.25585938,
                          0.08300781, -0.18359375, 0.06542969, 0.1640625, -0.00408936,
                          0.09033203, 0.00247192, -0.14355469, -0.20507812, -0.10253906,
                          0.23046875, 0.02746582, 0.19726562, 0.06884766, 0.078125,
                          0.05444336, 0.11865234, 0.03735352, -0.16992188, 0.00372314,
                          -0.08691406, 0.15136719, 0.24023438, -0.1015625, 0.02685547,
                          0.00479126, 0.10205078, 0.078125, 0.1640625, -0.03979492,
                          0.27539062, -0.03540039, 0.00088882, -0.1953125, 0.11621094,
                          0.18652344, -0.1796875, -0.25390625, -0.12890625, 0.00653076,
                          -0.14550781, 0.06396484, 0.11132812, -0.00793457, -0.0859375,
                          0.19238281, 0.18457031, -0.06445312, -0.05395508, 0.0246582,
                          -0.13964844, 0.10791016, -0.01855469, 0.39453125, 0.33203125,
                          0.07470703, 0.00787354, -0.12988281, -0.09814453, 0.00271606,
                          0.203125, 0.16894531, 0.109375, 0.10693359, -0.0123291,
                          -0.05615234, 0.04516602, -0.16992188, -0.06591797, 0.08203125,
                          -0.02038574, -0.06884766, 0.04101562, 0.0625, -0.01586914,
                          0.12109375, 0.15917969, -0.0291748, 0.20019531, -0.03491211,
                          0.51171875, 0.19921875, -0.01275635, 0.00793457, 0.00714111,
                          0.140625, 0.15917969, 0.24511719, -0.45898438, 0.25976562,
                          0.05078125, 0.25976562, -0.08740234, 0.26171875, -0.04833984,
                          -0.25976562, -0.05541992, -0.08740234, -0.05126953, -0.10498047,
                          0.01446533, -0.08496094, 0.21386719, -0.00939941, 0.16113281,
                          0.14453125, -0.06884766, 0.00074005, -0.07226562, 0.19824219,
                          -0.04614258, -0.08740234, 0.08251953, 0.22949219, 0.05932617]

        self.assertTrue(np.allclose(result, expected))

        source = BinaryFile('../../../datasets/doc2vec/doc2vec.bin', "Doc2Vec")
        result = source.load("text news")
        expected = np.ndarray(shape=(2, 300))
        expected[0, :] = [3.32591474e-01, -3.09491992e-01, -1.16927773e-01, -1.89882010e-01,
                          -2.66758334e-02, -1.25457957e-01, 1.11133024e-01, -8.65858048e-03,
                          -2.10775975e-02, -1.42615661e-01, -1.70673415e-01, -1.75790548e-01,
                          -3.33384126e-02, 9.07543227e-02, -1.05401312e-04, -1.04231127e-01,
                          -4.81868796e-02, -8.52165148e-02, 5.88280661e-03, -1.41212806e-01,
                          -9.76510998e-03, 4.25822847e-02, 9.25908685e-02, -7.90970176e-02,
                          -6.11959957e-02, -6.23688139e-02, -1.03086740e-01, 1.09199122e-01,
                          3.34305204e-02, -1.09094396e-01, 1.07806116e-01, -3.81813273e-02,
                          -1.18087851e-01, -3.38167787e-01, -2.00259864e-01, 8.63896906e-02,
                          1.54839680e-02, 4.63569649e-02, -2.14320451e-01, 1.28902167e-01,
                          -6.03596754e-02, 2.40994036e-01, 4.07791808e-02, -6.95588887e-02,
                          -7.87252039e-02, -8.49624723e-02, 1.24098711e-01, 1.38703920e-02,
                          1.09158374e-01, 3.35489726e-03, 1.43248484e-01, 3.16842161e-02,
                          1.01523712e-01, -1.73473701e-01, 1.05244122e-01, -4.37277481e-02,
                          8.35154280e-02, 9.01857167e-02, -9.40518752e-02, -4.33304720e-02,
                          8.02260917e-03, 7.79413357e-02, 1.65174112e-01, -2.10044030e-02,
                          1.23664446e-01, 1.49019584e-01, -1.83845088e-01, -1.89474210e-01,
                          -7.80021921e-02, -3.47397029e-02, -1.37102261e-01, -1.13735735e-01,
                          5.46273775e-02, 2.67339557e-01, -8.34997371e-02, -1.34896889e-01,
                          1.27642736e-01, 6.98359907e-02, 2.52409190e-01, -1.19120620e-01,
                          -1.91941202e-01, -9.47562382e-02, 9.51269269e-02, 1.10305570e-01,
                          -1.00850299e-01, -1.62187025e-01, -9.78800654e-02, -3.43192637e-01,
                          1.24059647e-01, 3.08723878e-02, -5.55606671e-02, -5.29498123e-02,
                          9.92605835e-02, -2.15869740e-01, -1.03096729e-02, 2.07323842e-02,
                          -1.38094753e-01, -1.12982899e-01, -6.20741136e-02, 2.29527447e-02,
                          6.84464499e-02, -9.36816707e-02, 6.73113838e-02, -1.19771622e-01,
                          1.82321370e-01, -1.25769958e-01, -1.56315446e-01, 3.57681885e-03,
                          -1.22599274e-01, -8.45444128e-02, -2.04405952e-02, -4.03077006e-02,
                          1.16646484e-01, 1.34257033e-01, -1.56645730e-01, 1.47093963e-02,
                          1.05933353e-01, 7.76969790e-02, -1.00241594e-01, -1.33975595e-01,
                          1.79218538e-02, -3.42416674e-01, -2.20950618e-02, 1.13678128e-02,
                          -6.95204437e-02, -2.73086429e-02, 6.55554831e-02, -6.07350655e-02,
                          -3.82313170e-02, 9.07379314e-02, -2.24110544e-01, -1.80637628e-01,
                          -5.91759682e-02, -4.48031425e-02, -7.62447044e-02, -1.23184755e-01,
                          7.68664330e-02, -1.52610198e-01, 2.49477983e-01, -1.05579071e-01,
                          -1.17066419e-02, 7.70153850e-02, 1.10536352e-01, -1.68868750e-01,
                          -1.25838658e-02, 5.44804446e-02, 1.68925777e-01, -2.85610318e-01,
                          7.80191422e-02, 2.53162324e-01, -2.37224028e-02, -2.75686588e-02,
                          3.97270769e-02, -1.16238100e-02, -8.70959312e-02, -1.11751027e-01,
                          -1.34716526e-01, -1.53817505e-01, -8.82015377e-02, 2.11588696e-01,
                          -9.14662704e-02, -4.82495353e-02, 4.86858822e-02, 2.31947243e-01,
                          -1.73707470e-01, -1.35220647e-01, 1.93649605e-01, 2.05569029e-01,
                          1.02160104e-01, 1.00329429e-01, 7.04126572e-03, -3.04290466e-02,
                          -1.31014630e-01, 7.97956064e-02, 1.04789637e-01, -7.98387453e-02,
                          -3.86632495e-02, -1.06474854e-01, -1.86798975e-01, -1.47570208e-01,
                          -3.39680873e-02, 1.26506567e-01, -3.66205759e-02, -1.07454918e-01,
                          1.93108797e-01, 8.89965594e-02, 2.81857908e-01, -8.98421183e-02,
                          1.51605070e-01, 5.32575399e-02, 1.28689423e-01, -1.61661625e-01,
                          -5.88420499e-03, 2.04903170e-01, 1.42696097e-01, -1.02039538e-01,
                          1.49165601e-01, -5.14920428e-02, 6.56654015e-02, -2.23770455e-01,
                          1.98588997e-01, -1.76789872e-02, -1.05234176e-01, 1.23672701e-01,
                          -9.62969288e-02, 1.10550702e-01, -1.01940274e-01, -1.12086497e-01,
                          -1.33916279e-02, 2.32869890e-02, 2.43199259e-01, -2.71918215e-02,
                          2.07770430e-02, -5.45424297e-02, -2.95656957e-02, 9.75178629e-02,
                          -4.19010930e-02, -9.81168970e-02, -3.58092822e-02, -3.26528937e-01,
                          2.92078465e-01, 1.75689355e-01, 7.95550458e-03, -5.35775349e-02,
                          -1.64883137e-01, 9.53269228e-02, -1.57614470e-01, 1.72146171e-01,
                          -4.23732847e-02, -4.64421660e-02, 2.30887234e-02, -1.38779119e-01,
                          4.70183715e-02, 8.91419128e-02, 2.55701184e-01, -1.81416452e-01,
                          -6.63795099e-02, -3.24904293e-01, -5.99929020e-02, 1.79415941e-01,
                          -2.24056974e-01, -2.23247811e-01, -1.14762716e-01, 2.02016905e-01,
                          -8.93826634e-02, -4.04954180e-02, -6.40705675e-02, -5.15503399e-02,
                          -1.75742693e-02, -5.62064871e-02, -2.38100007e-01, 1.63455978e-01,
                          2.54988998e-01, -1.79487318e-01, -8.99407342e-02, -7.97683597e-02,
                          1.41119763e-01, -1.04419306e-01, 8.54527578e-02, 1.33719463e-02,
                          5.18752933e-02, -4.19723131e-02, -3.76287363e-02, -4.32124957e-02,
                          -2.01199651e-01, 2.66648054e-01, -1.08547188e-01, -2.95744866e-01,
                          1.59967780e-01, 1.58440575e-01, 1.14920646e-01, 6.13304079e-02,
                          2.37842724e-02, 1.27396420e-01, -5.32822013e-02, -1.73073396e-01,
                          -2.95929927e-02, -6.24100156e-02, -2.12781560e-02, -1.37174070e-01,
                          -1.84251294e-01, 3.01615484e-02, 2.12930683e-02, 2.11409226e-01,
                          5.56483082e-02, 5.42792715e-02, -4.39312831e-02, -1.79031834e-01,
                          -1.84587780e-02, -1.03078477e-01, -9.33797359e-02, 1.26520559e-01,
                          -1.74526244e-01, 8.32095072e-02, -9.77820680e-02, -3.24912578e-01,
                          -1.44377351e-01, -1.75939187e-01, -3.34159657e-02, -1.45787140e-02]

        expected[1, :] = [0.15289317, -0.0782943, 0.04543685, 0.03167117, 0.04125804,
                          0.04799604, -0.06481372, -0.04505093, -0.14690772, -0.06540094,
                          0.04550788, -0.04023168, -0.09635589, 0.12668592, 0.18476722,
                          0.10981479, -0.22389457, -0.05319795, -0.12355348, 0.01905239,
                          -0.18043397, 0.03257597, 0.0464316, -0.10982691, 0.0656085,
                          -0.05259619, -0.2389114, -0.02606498, -0.00364286, 0.05953985,
                          0.06976728, 0.07399364, -0.2370548, -0.03587143, 0.0220487,
                          0.08242403, -0.16394056, 0.01150678, -0.14957795, 0.06170287,
                          0.02099631, 0.08778779, -0.00438846, -0.1897867, 0.14198437,
                          -0.08642647, 0.03237326, -0.0141041, 0.14832641, 0.02490721,
                          0.20422931, 0.04441843, 0.1467739, -0.04676935, 0.17325892,
                          0.21757147, 0.1730551, -0.02312613, -0.0534586, 0.02154517,
                          0.02861254, -0.08458275, 0.09775919, 0.1031286, 0.00370332,
                          -0.11418209, 0.01107013, -0.18499497, 0.10542057, -0.14540465,
                          0.00857523, -0.08540139, -0.04837888, 0.2551243, -0.20386524,
                          -0.03769662, -0.02683698, 0.04772722, 0.25871986, -0.01971802,
                          -0.22276707, 0.08751022, 0.08742964, -0.07358093, -0.02776588,
                          -0.09943029, -0.1007816, -0.1133064, 0.027592, 0.00055542,
                          -0.03526301, -0.00560357, 0.04161033, -0.13717675, 0.01541165,
                          -0.15786435, 0.0396997, 0.04754698, -0.066662, -0.17449231,
                          0.08619366, 0.02444649, 0.16845043, 0.03962767, 0.12506047,
                          -0.15268034, -0.02262232, 0.06172806, -0.02297839, 0.20350246,
                          0.04325144, 0.04592946, -0.12978323, 0.01214922, -0.04184311,
                          -0.17732696, -0.01723116, -0.09144585, 0.04984059, 0.04579171,
                          0.09399638, 0.03074398, 0.13281024, 0.03526729, -0.04178892,
                          0.13509043, -0.01846117, -0.1347117, -0.1906798, -0.0255361,
                          -0.0177574, 0.06398717, -0.0075289, 0.11851032, 0.02088471,
                          -0.04442274, -0.07336057, 0.10466082, 0.08244446, -0.00391588,
                          0.03282203, 0.14932613, -0.07355812, 0.03801617, -0.15333943,
                          0.11779818, 0.18683971, -0.11930901, 0.12186129, 0.04718727,
                          -0.14809564, -0.14859916, -0.02766251, -0.1662515, 0.1869445,
                          -0.13204327, 0.12978531, -0.06250325, -0.12274851, 0.0467319,
                          0.07979856, -0.12195876, -0.10565057, 0.03270352, -0.01808363,
                          0.16895309, 0.01900798, 0.03984949, 0.0784145, 0.07256132,
                          0.13885066, -0.07310494, -0.11788159, -0.12994939, 0.06722286,
                          -0.11798514, -0.04847602, 0.15857509, -0.21807814, -0.09064635,
                          0.06358983, 0.0276533, -0.11735569, -0.19164741, 0.29601428,
                          0.02827495, -0.01064462, 0.24005859, -0.05335181, 0.14396288,
                          -0.07047689, -0.35292953, -0.06133325, -0.00101552, 0.0296257,
                          -0.08730184, -0.02528892, 0.10505053, 0.12444009, -0.10683569,
                          -0.00660268, 0.01637497, -0.02553134, -0.09758145, 0.04783034,
                          -0.10707644, 0.03513033, -0.09367235, -0.2000741, -0.07538611,
                          0.21823801, 0.27942234, 0.16475765, 0.11521529, -0.19632049,
                          0.09783484, -0.01418906, -0.10218392, 0.01460731, -0.14681594,
                          0.19380412, 0.13030158, 0.10973753, 0.03104388, -0.09121986,
                          -0.05474691, -0.10294704, -0.10514182, 0.04485561, 0.135345,
                          -0.14136976, -0.20574634, -0.0080816, 0.15020408, 0.01585441,
                          -0.07305589, 0.06006964, -0.15695834, 0.0304837, 0.20200647,
                          0.03556017, 0.16939993, 0.06204869, 0.01436886, -0.19452588,
                          0.06239187, 0.05015276, -0.18474348, 0.04404762, -0.05656374,
                          -0.23439904, -0.04669996, 0.20781869, -0.09336343, -0.16964744,
                          0.20422806, -0.02667397, 0.05270841, -0.2243127, -0.12729517,
                          0.16797791, -0.09198651, 0.12622732, -0.08810753, -0.00571408,
                          -0.10548767, 0.11299454, -0.01513917, 0.05739972, 0.07167779,
                          -0.06481692, 0.08272585, -0.02899203, 0.23187093, 0.20122191,
                          -0.02317913, 0.2076713, -0.14810611, -0.02602801, -0.14180899,
                          -0.13128494, -0.00752035, -0.05175586, 0.13625576, 0.09037174,
                          0.01002607, 0.01641791, -0.13961627, 0.06713493, 0.04727478,
                          -0.11128273, -0.01255747, -0.20458147, 0.05414136, -0.03743472,
                          -0.3279437, -0.04739661, 0.11645208, -0.09729925, -0.09865585]

        self.assertTrue(np.allclose(result, expected))

        source = BinaryFile('../../../datasets/fasttext/wiki.simple.bin', "fasttext")
        result = source.load("text news")
        expected = np.ndarray(shape=(2, 300))
        expected[0, :] = [0.20265126, -0.21283358, 0.14194603, 0.33037463, 0.41914618,
                          0.15711078, -0.53413284, 0.02762997, -0.4063238, -0.44236043,
                          0.19665046, -0.4055665, -0.44119176, 0.38790253, -0.26242533,
                          -0.36879623, -0.20069647, -0.10250822, -0.50207686, -0.35730335,
                          0.06335476, 0.73933786, 0.11101818, -0.2784164, -0.01197096,
                          -0.12922496, 0.3368253, -0.23762392, -0.20161277, 0.3494078,
                          0.23495756, 0.09377261, -0.721257, 0.18058123, -0.20233215,
                          -0.42834234, 0.2598636, 0.30087867, 0.13686465, -0.13219346,
                          0.6159414, -0.7395258, -0.33667007, 0.05805199, 0.3754556,
                          -0.2020773, 0.0754751, -0.22536536, -0.33241814, -0.0665015,
                          -0.11007189, -0.2694806, 0.13119058, -0.37797686, -0.03514016,
                          -0.23820072, 0.3968444, 0.39613697, 0.44374695, -0.07472508,
                          0.07599565, 0.24302596, -0.340522, 0.0066885, 0.1533564,
                          0.15721421, -0.4150926, -0.01028172, -0.10909708, 0.05736974,
                          0.14382888, 0.17113256, -0.19844438, 0.61571974, -0.09871307,
                          -0.34877625, 0.02524109, -0.22571068, 0.29495266, -0.49806812,
                          -0.32246125, 0.05215746, 0.04231, -0.03803429, -0.4342325,
                          0.5000546, 0.11892998, 0.06892694, 0.50459373, 0.11589471,
                          -0.49837324, 0.1118788, 0.06382235, -0.14535412, -0.11556458,
                          0.3889322, -0.49182093, 0.4937278, -0.12737173, 0.09726905,
                          0.2096973, -0.22088236, 0.05826837, 0.20914072, -0.13349253,
                          -0.02749143, -0.08141553, -0.4040055, 0.19592877, 0.34658158,
                          0.2125258, 0.15288296, 0.61846995, -0.23142041, -0.02473347,
                          -0.00421861, 0.25411284, 0.29874337, -0.01599878, -0.13417433,
                          -0.00180563, 0.29612297, 0.07234211, -0.18437372, 0.0867601,
                          -0.14390057, -0.5114709, 0.18734442, -0.2201622, 0.06342915,
                          -0.1720867, -0.35517645, -0.05359796, -0.09220639, -0.23433195,
                          0.54796106, -0.1095366, 0.2681576, -0.39470208, 0.07993273,
                          0.43899587, 0.27142036, -0.36522874, 0.15208222, 0.06611345,
                          0.13593692, 0.20696463, -0.55657077, -0.07059377, -0.2977042,
                          -0.08788259, -0.15642352, 0.1089663, 0.185574, -0.10140529,
                          0.37777558, -0.36947367, -0.1551182, -0.12709399, 0.67676693,
                          0.29411024, -0.30430093, 0.26259544, -0.01770081, -0.86365396,
                          0.01749651, -0.21317881, 0.15759598, 0.5772509, 0.74017143,
                          0.27313715, 0.3090748, -0.14768393, 0.00949798, -0.6146934,
                          -0.14036936, -0.43399084, -0.3475925, -0.24635337, 0.04097411,
                          -0.9016421, -0.09448994, 0.13511698, 0.45544633, -0.31879598,
                          -0.6458499, -0.1308179, -0.12942445, 0.13893668, -0.17408696,
                          -0.2624275, 0.35858792, -0.25121766, 0.11553709, -0.06607655,
                          -0.12223328, -0.65684617, -0.35322914, -0.07341404, -0.13507004,
                          -0.1474892, 0.14239295, 0.27803457, 0.28336865, -0.11589239,
                          0.08695987, -0.02754431, -0.22740752, 0.12800531, -0.67438674,
                          0.25728008, -0.28257787, 0.32899034, 0.05612519, -0.14490944,
                          -0.44030094, -0.138331, 0.29780743, 0.14685835, 0.09039471,
                          0.03897339, -0.14761756, -0.12335532, -0.5062072, -0.2637029,
                          0.06947806, -0.42540285, 0.16323572, -0.15194912, 0.22176957,
                          0.2552345, 0.7569656, 0.14681932, -0.2934617, -0.02774804,
                          -0.0584788, 0.19847272, 0.15071422, 0.10361392, -0.00257493,
                          0.2228439, -0.4432684, 0.21382017, -0.06001007, -0.6113228,
                          -0.14729208, 0.4382558, -0.09843273, 0.33768573, 0.35191357,
                          -0.01992144, 0.37499142, 0.06319309, 0.07888996, -0.03344069,
                          0.01924008, 0.03195959, 0.2716108, 0.09556211, 0.41367817,
                          -0.2710366, -0.48075238, -0.21826798, 0.35785788, 0.04058509,
                          0.15952648, 0.4304269, -0.31193888, -0.05303663, -0.5039003,
                          -0.2496129, 0.43995544, 0.24552917, 0.31998643, 0.25880936,
                          -0.02902658, 0.23226047, -0.12927634, -0.07708461, -0.09671998,
                          0.38768873, -0.35123688, 0.00947487, -0.5105597, -0.08035715,
                          0.03961244, -0.11575527, -0.05171135, 0.23336245, 0.02142764,
                          -0.25515625, -0.46669328, -0.06520634, -0.31691206, 0.05380315,
                          0.6493854, -0.22080694, 0.05451676, -0.09713368, 0.0028772]
        expected[1, :] = [0.44140396, -0.16512305, -0.05835297, 0.3722541, 0.06371133,
                          0.06065029, 0.06663077, -0.18531364, -0.1730505, 0.06456384,
                          0.15869819, -0.41249874, -0.02186436, -0.2988041, -0.14314261,
                          -0.47785437, -0.21317543, 0.13526537, -0.2196289, -0.16715379,
                          0.04820282, 0.5822747, -0.03824522, 0.19387215, 0.07264534,
                          -0.7886305, -0.10755262, -0.2287159, 0.2711898, -0.28963944,
                          -0.5079648, 0.10594227, -0.5821161, -0.2112332, -0.07257365,
                          -0.23768449, -0.23774132, 0.15633559, -0.19565214, -0.17835818,
                          -0.15586743, 0.01204043, -0.27066886, 0.13278507, 0.03657596,
                          -0.1220506, 0.0933435, 0.03486151, 0.3004428, -0.1934173,
                          -0.01768212, -0.31368077, -0.18841547, 0.4988608, -0.10564907,
                          0.3439958, -0.01935345, 0.13479942, 0.28455764, 0.1832007,
                          -0.23508763, 0.4911381, -0.04732992, -0.04582207, 0.0488975,
                          -0.19011584, -0.2823922, 0.40297124, 0.1787498, -0.03268332,
                          0.27818993, 0.19737643, -0.33094156, -0.12263158, -0.22905694,
                          0.1106456, 0.11630575, -0.06752507, 0.1914318, -0.22503883,
                          0.06131548, -0.01526864, 0.7918036, -0.11958119, -0.14359862,
                          -0.06936139, -0.00962653, 0.01325183, 0.0886803, -0.22291175,
                          -0.28760627, -0.20491956, 0.41083053, 0.21269096, -0.07994829,
                          0.49757817, -0.46096689, -0.28990343, -0.04413332, 0.14557861,
                          -0.06882112, -0.160904, 0.26561776, 0.15733427, -0.6524361,
                          0.13915342, 0.13773847, 0.47502616, -0.18162337, -0.09757914,
                          -0.07843821, 0.03559673, 0.12673298, -0.3936927, 0.19280481,
                          -0.02963025, 0.18091993, 0.5243473, -0.03895733, 0.20965905,
                          0.21822323, -0.2422387, 0.07124823, -0.32052293, -0.07033145,
                          -0.43469316, 0.30554292, -0.44378805, 0.2311067, 0.3351523,
                          0.2993047, 0.09660671, -0.38278762, -0.3550027, -0.41756475,
                          0.37902156, -0.35120466, -0.14956477, 0.21724972, -0.2474722,
                          0.22780372, 0.11818378, 0.15721498, -0.03136751, 0.08163261,
                          -0.54194534, -0.14791222, -0.02690194, 0.15339397, 0.33709326,
                          -0.08696684, 0.15374035, -0.03982278, 0.1616175, 0.4353388,
                          0.5435433, -0.37198183, -0.09595806, 0.07710841, -0.03629696,
                          -0.00884286, -0.23087126, -0.129574, -0.24357997, -0.10973019,
                          -0.04560639, -0.30194598, -0.30643693, 0.11073697, 0.06591746,
                          0.0389521, 0.12174832, 0.13400546, 0.13599335, 0.3315635,
                          0.629914, 0.132276, -0.05388134, 0.04836098, -0.10355444,
                          -0.42241433, 0.14227682, 0.06354158, 0.26077715, 0.25609484,
                          0.04091172, -0.10878282, 0.31179592, -0.30064255, 0.08403483,
                          0.2248777, -0.06197835, 0.33993068, 0.22315222, -0.32167986,
                          -0.17155413, -0.49069425, 0.16134955, -0.02955464, -0.6230852,
                          0.20967445, 0.6108316, 0.06048961, -0.4814418, 0.26789495,
                          0.14616613, 0.18880369, -0.16119532, -0.29113933, 0.11602445,
                          0.07735816, -0.04898551, 0.14211503, 0.17361684, -0.05891947,
                          0.44504893, 0.11537931, -0.28760794, -0.32343984, -0.04157984,
                          0.23971106, 0.18956056, -0.26029688, 0.2852674, -0.2845836,
                          -0.26341477, 0.00289561, 0.3694028, -0.19868606, 0.38676587,
                          0.13096467, 0.3062931, -0.7531132, 0.22585148, -0.05396655,
                          -0.01131617, 0.4333124, -0.04644657, -0.32544544, 0.33940423,
                          0.68359274, 0.1888048, -0.03318648, 0.12484468, -0.11435971,
                          -0.6927454, -0.17025834, -0.40711528, -0.10458315, -0.07473296,
                          -0.27696902, 0.12343498, -0.17807195, 0.07573699, -0.44746,
                          -0.09051498, -0.03607579, 0.01627962, 0.09814311, -0.09919167,
                          -0.28723803, 0.04392961, 0.4575911, 0.25710514, -0.07525393,
                          -0.24280912, -0.15236172, -0.3156707, -0.07079593, -0.47093952,
                          -0.01041144, 0.34181687, 0.5398934, 0.32434848, 0.38001508,
                          0.2971378, 0.41582862, -0.2589041, 0.27061078, -0.13501002,
                          -0.22250865, 0.2502009, 0.17115396, 0.4303569, 0.07039726,
                          0.24438983, -0.14475271, 0.4289765, -0.15569666, 0.29032007,
                          0.14424388, 0.15134367, -0.0173972, -0.09971657, -0.38534534,
                          -0.48812523, -0.24264745, -0.20514202, -0.05461162, 0.04838076]

        self.assertTrue(np.allclose(result, expected))
