# Copyright 2020, The TensorFlow Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Lint as: python3
"""An example for using tf_estimator_evaluation."""

from absl import app
from absl import flags
from absl import logging

import numpy as np
import tensorflow.compat.v1 as tf

from tensorflow_privacy.privacy.membership_inference_attack.tf_estimator_evaluation import MembershipInferenceTrainingHook
from tensorflow_privacy.privacy.membership_inference_attack.tf_estimator_evaluation import run_attack_on_tf_estimator_model

GradientDescentOptimizer = tf.train.GradientDescentOptimizer

FLAGS = flags.FLAGS

flags.DEFINE_float('learning_rate', .15, 'Learning rate for training')
flags.DEFINE_integer('batch_size', 256, 'Batch size')
flags.DEFINE_integer('epochs', 10, 'Number of epochs')
flags.DEFINE_string('model_dir', None, 'Model directory.')


def cnn_model_fn(features, labels, mode):
  """Model function for a CNN."""

  # Define CNN architecture using tf.keras.layers.
  input_layer = tf.reshape(features['x'], [-1, 28, 28, 1])
  y = tf.keras.layers.Conv2D(
      16, 8, strides=2, padding='same', activation='relu').apply(input_layer)
  y = tf.keras.layers.MaxPool2D(2, 1).apply(y)
  y = tf.keras.layers.Conv2D(
      32, 4, strides=2, padding='valid', activation='relu').apply(y)
  y = tf.keras.layers.MaxPool2D(2, 1).apply(y)
  y = tf.keras.layers.Flatten().apply(y)
  y = tf.keras.layers.Dense(32, activation='relu').apply(y)
  logits = tf.keras.layers.Dense(10).apply(y)

  if mode != tf.estimator.ModeKeys.PREDICT:
    vector_loss = tf.nn.sparse_softmax_cross_entropy_with_logits(
        labels=labels, logits=logits)
    scalar_loss = tf.reduce_mean(input_tensor=vector_loss)

  # Configure the training op (for TRAIN mode).
  if mode == tf.estimator.ModeKeys.TRAIN:
    optimizer = GradientDescentOptimizer(learning_rate=FLAGS.learning_rate)
    global_step = tf.train.get_global_step()
    train_op = optimizer.minimize(loss=scalar_loss, global_step=global_step)
    return tf.estimator.EstimatorSpec(
        mode=mode,
        loss=scalar_loss,
        train_op=train_op)

  # Add evaluation metrics (for EVAL mode).
  elif mode == tf.estimator.ModeKeys.EVAL:
    eval_metric_ops = {
        'accuracy':
            tf.metrics.accuracy(
                labels=labels, predictions=tf.argmax(input=logits, axis=1))
    }
    return tf.estimator.EstimatorSpec(
        mode=mode, loss=scalar_loss, eval_metric_ops=eval_metric_ops)

  # Output the prediction probability (for PREDICT mode).
  elif mode == tf.estimator.ModeKeys.PREDICT:
    predictions = tf.nn.softmax(logits)
    return tf.estimator.EstimatorSpec(mode=mode, predictions=predictions)


def load_mnist():
  """Loads MNIST and preprocesses to combine training and validation data."""
  (train_data,
   train_labels), (test_data,
                   test_labels) = tf.keras.datasets.mnist.load_data()

  train_data = np.array(train_data, dtype=np.float32) / 255
  test_data = np.array(test_data, dtype=np.float32) / 255

  train_labels = np.array(train_labels, dtype=np.int32)
  test_labels = np.array(test_labels, dtype=np.int32)

  return train_data, train_labels, test_data, test_labels


def main(unused_argv):
  tf.logging.set_verbosity(tf.logging.INFO)
  logging.set_verbosity(logging.INFO)
  logging.set_stderrthreshold(logging.INFO)
  logging.get_absl_handler().use_absl_log_file()

  # Load training and test data.
  train_data, train_labels, test_data, test_labels = load_mnist()

  # Instantiate the tf.Estimator.
  mnist_classifier = tf.estimator.Estimator(model_fn=cnn_model_fn,
                                            model_dir=FLAGS.model_dir)

  # A function to construct input_fn given (data, label), to be used by the
  # membership inference training hook.
  def input_fn_constructor(x, y):
    return tf.estimator.inputs.numpy_input_fn(x={'x': x}, y=y, shuffle=False)

  with tf.Graph().as_default():
    # Get a summary writer for the hook to write to tensorboard.
    # Can set summary_writer to None if not needed.
    if FLAGS.model_dir:
      summary_writer = tf.summary.FileWriter(FLAGS.model_dir)
    else:
      summary_writer = None
    mia_hook = MembershipInferenceTrainingHook(mnist_classifier,
                                               (train_data, train_labels),
                                               (test_data, test_labels),
                                               input_fn_constructor,
                                               [],
                                               summary_writer)

  # Create tf.Estimator input functions for the training and test data.
  train_input_fn = tf.estimator.inputs.numpy_input_fn(
      x={'x': train_data},
      y=train_labels,
      batch_size=FLAGS.batch_size,
      num_epochs=FLAGS.epochs,
      shuffle=True)
  eval_input_fn = tf.estimator.inputs.numpy_input_fn(
      x={'x': test_data}, y=test_labels, num_epochs=1, shuffle=False)

  # Training loop.
  steps_per_epoch = 60000 // FLAGS.batch_size
  for epoch in range(1, FLAGS.epochs + 1):
    # Train the model, with the membership inference hook.
    mnist_classifier.train(input_fn=train_input_fn, steps=steps_per_epoch,
                           hooks=[mia_hook])

    # Evaluate the model and print results
    eval_results = mnist_classifier.evaluate(input_fn=eval_input_fn)
    test_accuracy = eval_results['accuracy']
    print('Test accuracy after %d epochs is: %.3f' % (epoch, test_accuracy))

  print('End of training attack')
  run_attack_on_tf_estimator_model(mnist_classifier,
                                   (train_data, train_labels),
                                   (test_data, test_labels),
                                   input_fn_constructor,
                                   ['lr'])


if __name__ == '__main__':
  app.run(main)
