"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 */
class LogGroupLogDestination {
    constructor(logGroup) {
        this.logGroup = logGroup;
    }
    /**
     * Binds this destination to the CloudWatch Logs.
     */
    bind(_stage) {
        return {
            destinationArn: this.logGroup.logGroupArn
        };
    }
}
exports.LogGroupLogDestination = LogGroupLogDestination;
/**
 * $context variables that can be used to customize access log pattern.
 */
class AccessLogField {
    /**
     * The API owner's AWS account ID.
     */
    static contextAccountId() {
        return '$context.requestId';
    }
    /**
     * The identifier API Gateway assigns to your API.
     */
    static contextApiId() {
        return '$context.apiId';
    }
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     *
     * @param property A property key of the claims.
     */
    static contextAuthorizerClaims(property) {
        return `$context.authorizer.claims.${property}`;
    }
    /**
     * The principal user identification associated with the token sent by the client and returned
     * from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     */
    static contextAuthorizerPrincipalId() {
        return '$context.authorizer.principalId';
    }
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @param property key of the context map.
     */
    static contextAuthorizer(property) {
        return `$context.authorizer.${property}`;
    }
    /**
     * The AWS endpoint's request ID.
     */
    static contextAwsEndpointRequestId() {
        return '$context.awsEndpointRequestId';
    }
    /**
     * The full domain name used to invoke the API. This should be the same as the incoming `Host` header.
     */
    static contextDomainName() {
        return '$context.domainName';
    }
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     */
    static contextDomainPrefix() {
        return '$context.domainPrefix';
    }
    /**
     * A string containing an API Gateway error message.
     */
    static contextErrorMessage() {
        return '$context.error.message';
    }
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     */
    static contextErrorMessageString() {
        return '$context.error.messageString';
    }
    /**
     * A type of GatewayResponse. This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-websocket-api-logging.html
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     */
    static contextErrorResponseType() {
        return '$context.error.responseType';
    }
    /**
     * A string containing a detailed validation error message.
     */
    static contextErrorValidationErrorString() {
        return '$context.error.validationErrorString';
    }
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     */
    static contextExtendedRequestId() {
        return '$context.extendedRequestId';
    }
    /**
     * The HTTP method used. Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     */
    static contextHttpMethod() {
        return '$context.httpMethod';
    }
    /**
     * The AWS account ID associated with the request.
     */
    static contextIdentityAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     * For methods that don't require an API key, this variable is
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     */
    static contextIdentityApiKey() {
        return '$context.identity.apiKey';
    }
    /**
     * The API key ID associated with an API request that requires an API key.
     */
    static contextIdentityApiKeyId() {
        return '$context.identity.apiKeyId';
    }
    /**
     * The principal identifier of the caller making the request.
     */
    static contextIdentityCaller() {
        return '$context.identity.caller';
    }
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     */
    static contextIdentityCognitoAuthenticationProvider() {
        return '$context.identity.cognitoAuthenticationProvider';
    }
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoAuthenticationType() {
        return '$context.identity.cognitoAuthenticationType';
    }
    /**
     * The Amazon Cognito identity ID of the caller making the request. Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityId() {
        return '$context.identity.cognitoIdentityId';
    }
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     * Available only if the request was signed with Amazon Cognito credentials.
     */
    static contextIdentityCognitoIdentityPoolId() {
        return '$context.identity.cognitoIdentityPoolId';
    }
    /**
     * The AWS organization ID.
     */
    static contextIdentityPrincipalOrgId() {
        return '$context.identity.principalOrgId';
    }
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     */
    static contextIdentitySourceIp() {
        return '$context.identity.sourceIp';
    }
    /**
     * The principal identifier of the user making the request. Used in Lambda authorizers.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     */
    static contextIdentityUser() {
        return '$context.identity.user';
    }
    /**
     * The User-Agent header of the API caller.
     */
    static contextIdentityUserAgent() {
        return '$context.identity.userAgent';
    }
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     */
    static contextIdentityUserArn() {
        return '$context.identity.userArn';
    }
    /**
     * The request path.
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     */
    static contextPath() {
        return '$context.path';
    }
    /**
     * The request protocol, for example, HTTP/1.1.
     */
    static contextProtocol() {
        return '$context.protocol';
    }
    /**
     * The ID that API Gateway assigns to the API request.
     */
    static contextRequestId() {
        return '$context.requestId';
    }
    /**
     * The request header override.
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextRequestOverrideHeader(headerName) {
        return `$context.requestOverride.header.${headerName}`;
    }
    /**
     * The request path override. If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param pathName
     */
    static contextRequestOverridePath(pathName) {
        return `$context.requestOverride.path.${pathName}`;
    }
    /**
     * The request query string override.
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     *
     * @param querystringName
     */
    static contextRequestOverrideQuerystring(querystringName) {
        return `$context.requestOverride.querystring.${querystringName}`;
    }
    /**
     * The response header override.
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     *
     * @param headerName
     */
    static contextResponseOverrideHeader(headerName) {
        return `$context.responseOverride.header.${headerName}`;
    }
    /**
     * The response status code override.
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     */
    static contextResponseOverrideStatus() {
        return '$context.responseOverride.status';
    }
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     */
    static contextRequestTime() {
        return '$context.requestTime';
    }
    /**
     * The Epoch-formatted request time.
     */
    static contextRequestTimeEpoch() {
        return '$context.requestTimeEpoch';
    }
    /**
     * The identifier that API Gateway assigns to your resource.
     */
    static contextResourceId() {
        return '$context.resourceId';
    }
    /**
     * The path to your resource.
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     */
    static contextResourcePath() {
        return '$context.resourcePath';
    }
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     */
    static contextStage() {
        return '$context.stage';
    }
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`. Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWafResponseCode() {
        return '$context.wafResponseCode';
    }
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     * Will not be set if the stage is not associated with a web ACL.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     */
    static contextWebaclArn() {
        return '$context.webaclArn';
    }
    /**
     * The trace ID for the X-Ray trace.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     */
    static contextXrayTraceId() {
        return '$context.xrayTraceId';
    }
    /**
     * The authorizer latency in ms.
     */
    static contextAuthorizerIntegrationLatency() {
        return '$context.authorizer.integrationLatency';
    }
    /**
     * The integration latency in ms.
     */
    static contextIntegrationLatency() {
        return '$context.integrationLatency';
    }
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda,
     * not from the backend Lambda function.
     */
    static contextIntegrationStatus() {
        return '$context.integrationStatus.';
    }
    /**
     * The response latency in ms.
     */
    static contextResponseLatency() {
        return '$context.responseLatency';
    }
    /**
     * The response payload length.
     */
    static contextResponseLength() {
        return '$context.responseLength';
    }
    /**
     * The method response status.
     */
    static contextStatus() {
        return '$context.status';
    }
}
exports.AccessLogField = AccessLogField;
/**
 * factory methods for access log format.
 */
class AccessLogFormat {
    constructor(format) {
        this.format = format;
    }
    /**
     * Custom log format.
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     * @param format
     * @example custom(JSON.stringify({
     *  requestId: AccessLogField.contextRequestId(),
     *  sourceIp: AccessLogField.contextIdentitySourceIp(),
     *  method: AccessLogFiled.contextHttpMethod(),
     *  userContext: {
     *    sub: AccessLogField.contextAuthorizerClaims('sub'),
     *    email: AccessLogField.contextAuthorizerClaims('email')
     *  }
     * }))
     */
    static custom(format) {
        return new AccessLogFormat(format);
    }
    /**
     * Generate Common Log Format.
     */
    static clf() {
        const requester = [AccessLogField.contextIdentitySourceIp(), AccessLogField.contextIdentityCaller(), AccessLogField.contextIdentityUser()].join(' ');
        const requestTime = AccessLogField.contextRequestTime();
        const request = [AccessLogField.contextHttpMethod(), AccessLogField.contextResourcePath(), AccessLogField.contextProtocol()].join(' ');
        const status = [AccessLogField.contextStatus(), AccessLogField.contextResponseLength(), AccessLogField.contextRequestId()].join(' ');
        return new AccessLogFormat(`${requester} [${requestTime}] "${request}" ${status}`);
    }
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log. All fields are turned on by default with the
     * option to turn off specific fields.
     */
    static jsonWithStandardFields(fields = {
        ip: true,
        user: true,
        caller: true,
        requestTime: true,
        httpMethod: true,
        resourcePath: true,
        status: true,
        protocol: true,
        responseLength: true
    }) {
        return this.custom(JSON.stringify({
            requestId: AccessLogField.contextRequestId(),
            ip: fields.ip ? AccessLogField.contextIdentitySourceIp() : undefined,
            user: fields.user ? AccessLogField.contextIdentityUser() : undefined,
            caller: fields.caller ? AccessLogField.contextIdentityCaller() : undefined,
            requestTime: fields.requestTime ? AccessLogField.contextRequestTime() : undefined,
            httpMethod: fields.httpMethod ? AccessLogField.contextHttpMethod() : undefined,
            resourcePath: fields.resourcePath ? AccessLogField.contextResourcePath() : undefined,
            status: fields.status ? AccessLogField.contextStatus() : undefined,
            protocol: fields.protocol ? AccessLogField.contextProtocol() : undefined,
            responseLength: fields.responseLength ? AccessLogField.contextResponseLength() : undefined
        }));
    }
    /**
     * Output a format string to be used with CloudFormation.
     */
    toString() {
        return this.format;
    }
}
exports.AccessLogFormat = AccessLogFormat;
//# sourceMappingURL=data:application/json;base64,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