"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
class Stage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined && !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString()
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * Returns the invoke URL for a certain path.
     * @param path The resource path
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod, resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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