"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
module.exports = {
    'default token authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.Authorization'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'default request authorizer'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: [],
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'apigateway.amazonaws.com',
        }));
        test.ok(auth.authorizerArn.endsWith(`/authorizers/${auth.authorizerId}`), 'Malformed authorizer ARN');
        test.done();
    },
    'invalid request authorizer config'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        test.throws(() => new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            resultsCacheTtl: core_1.Duration.seconds(1),
            identitySources: [],
        }), Error, 'At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        test.done();
    },
    'token authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySource: 'method.request.header.whoami',
            validationRegex: 'a-hacker',
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            IdentityValidationExpression: 'a-hacker',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60
        }));
        test.done();
    },
    'request authorizer with all parameters specified'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('whoami')],
            authorizerName: 'myauthorizer',
            resultsCacheTtl: core_1.Duration.minutes(1),
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
            IdentitySource: 'method.request.header.whoami',
            Name: 'myauthorizer',
            AuthorizerResultTtlInSeconds: 60
        }));
        test.done();
    },
    'token authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole'
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'TOKEN',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName)
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    }
                ],
            }
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'request authorizer with assume role'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const role = new iam.Role(stack, 'authorizerassumerole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            roleName: 'authorizerassumerole'
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            assumeRole: role,
            resultsCacheTtl: core_1.Duration.seconds(0),
            identitySources: []
        });
        const restApi = new lib_1.RestApi(stack, 'myrestapi');
        restApi.root.addMethod('ANY', undefined, {
            authorizer: auth,
            authorizationType: lib_1.AuthorizationType.CUSTOM
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Authorizer', {
            Type: 'REQUEST',
            RestApiId: stack.resolve(restApi.restApiId),
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            Roles: [
                stack.resolve(role.roleName)
            ],
            PolicyDocument: {
                Statement: [
                    {
                        Resource: stack.resolve(func.functionArn),
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                    }
                ],
            }
        }, assert_1.ResourcePart.Properties, true));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Permission'));
        test.done();
    },
    'token authorizer throws when not attached to a rest api'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.TokenAuthorizer(stack, 'myauthorizer', {
            handler: func,
        });
        test.throws(() => stack.resolve(auth.authorizerArn), /must be attached to a RestApi/);
        test.done();
    },
    'request authorizer throws when not attached to a rest api'(test) {
        const stack = new core_1.Stack();
        const func = new lambda.Function(stack, 'myfunction', {
            handler: 'handler',
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
        });
        const auth = new lib_1.RequestAuthorizer(stack, 'myauthorizer', {
            handler: func,
            identitySources: [lib_1.IdentitySource.header('myheader')],
        });
        test.throws(() => stack.resolve(auth.authorizerArn), /must be attached to a RestApi/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5sYW1iZGEuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQXFFO0FBQ3JFLHdDQUF3QztBQUN4Qyw4Q0FBOEM7QUFDOUMsd0NBQWdEO0FBRWhELG1DQUEyRztBQUUzRyxpQkFBUztJQUNQLDBCQUEwQixDQUFDLElBQVU7UUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtZQUNwRCxPQUFPLEVBQUUsU0FBUztZQUNsQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ25DLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDdEQsT0FBTyxFQUFFLElBQUk7U0FDZCxDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLGFBQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEQsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUN2QyxVQUFVLEVBQUUsSUFBSTtZQUNoQixpQkFBaUIsRUFBRSx1QkFBaUIsQ0FBQyxNQUFNO1NBQzVDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxJQUFJLEVBQUUsT0FBTztZQUNiLFNBQVMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7WUFDM0MsY0FBYyxFQUFFLHFDQUFxQztTQUN0RCxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxNQUFNLEVBQUUsdUJBQXVCO1lBQy9CLFNBQVMsRUFBRSwwQkFBMEI7U0FDdEMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLGdCQUFnQixJQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO1FBRXRHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0QkFBNEIsQ0FBQyxJQUFVO1FBQ3JDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEQsT0FBTyxFQUFFLFNBQVM7WUFDbEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNuQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksdUJBQWlCLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUN4RCxPQUFPLEVBQUUsSUFBSTtZQUNiLGVBQWUsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNwQyxlQUFlLEVBQUUsRUFBRTtTQUNwQixDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLGFBQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEQsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUN2QyxVQUFVLEVBQUUsSUFBSTtZQUNoQixpQkFBaUIsRUFBRSx1QkFBaUIsQ0FBQyxNQUFNO1NBQzVDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxJQUFJLEVBQUUsU0FBUztZQUNmLFNBQVMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7U0FDNUMsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsTUFBTSxFQUFFLHVCQUF1QjtZQUMvQixTQUFTLEVBQUUsMEJBQTBCO1NBQ3RDLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLEVBQUUsMEJBQTBCLENBQUMsQ0FBQztRQUV0RyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUNBQW1DLENBQUMsSUFBVTtRQUM1QyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BELE9BQU8sRUFBRSxTQUFTO1lBQ2xCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbkMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztTQUNwQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksdUJBQWlCLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUM3RCxPQUFPLEVBQUUsSUFBSTtZQUNiLGVBQWUsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNwQyxlQUFlLEVBQUUsRUFBRTtTQUNwQixDQUFDLEVBQUUsS0FBSyxFQUFFLHVHQUF1RyxDQUFDLENBQUM7UUFFcEgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdEQUFnRCxDQUFDLElBQVU7UUFDekQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtZQUNwRCxPQUFPLEVBQUUsU0FBUztZQUNsQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ25DLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDdEQsT0FBTyxFQUFFLElBQUk7WUFDYixjQUFjLEVBQUUsOEJBQThCO1lBQzlDLGVBQWUsRUFBRSxVQUFVO1lBQzNCLGNBQWMsRUFBRSxjQUFjO1lBQzlCLGVBQWUsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUNyQyxDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLGFBQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEQsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUN2QyxVQUFVLEVBQUUsSUFBSTtZQUNoQixpQkFBaUIsRUFBRSx1QkFBaUIsQ0FBQyxNQUFNO1NBQzVDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxJQUFJLEVBQUUsT0FBTztZQUNiLFNBQVMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7WUFDM0MsY0FBYyxFQUFFLDhCQUE4QjtZQUM5Qyw0QkFBNEIsRUFBRSxVQUFVO1lBQ3hDLElBQUksRUFBRSxjQUFjO1lBQ3BCLDRCQUE0QixFQUFFLEVBQUU7U0FDakMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsa0RBQWtELENBQUMsSUFBVTtRQUMzRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BELE9BQU8sRUFBRSxTQUFTO1lBQ2xCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbkMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztTQUNwQyxDQUFDLENBQUM7UUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLHVCQUFpQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDeEQsT0FBTyxFQUFFLElBQUk7WUFDYixlQUFlLEVBQUUsQ0FBQyxvQkFBYyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUNsRCxjQUFjLEVBQUUsY0FBYztZQUM5QixlQUFlLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDckMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLEdBQUcsSUFBSSxhQUFPLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7WUFDdkMsVUFBVSxFQUFFLElBQUk7WUFDaEIsaUJBQWlCLEVBQUUsdUJBQWlCLENBQUMsTUFBTTtTQUM1QyxDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkJBQTZCLEVBQUU7WUFDM0QsSUFBSSxFQUFFLFNBQVM7WUFDZixTQUFTLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDO1lBQzNDLGNBQWMsRUFBRSw4QkFBOEI7WUFDOUMsSUFBSSxFQUFFLGNBQWM7WUFDcEIsNEJBQTRCLEVBQUUsRUFBRTtTQUNqQyxDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEQsT0FBTyxFQUFFLFNBQVM7WUFDbEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNuQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUU7WUFDdkQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDO1lBQy9ELFFBQVEsRUFBRSxzQkFBc0I7U0FDakMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDdEQsT0FBTyxFQUFFLElBQUk7WUFDYixVQUFVLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLGFBQU8sQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEQsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUN2QyxVQUFVLEVBQUUsSUFBSTtZQUNoQixpQkFBaUIsRUFBRSx1QkFBaUIsQ0FBQyxNQUFNO1NBQzVDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxJQUFJLEVBQUUsT0FBTztZQUNiLFNBQVMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7U0FDNUMsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1FBRWpELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNoRCxLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO2FBQzdCO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxRQUFRLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO3dCQUN6QyxNQUFNLEVBQUUsdUJBQXVCO3dCQUMvQixNQUFNLEVBQUUsT0FBTztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGLEVBQUUscUJBQVksQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUVuQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLHFCQUFZLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDO1FBRTdELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxQ0FBcUMsQ0FBQyxJQUFVO1FBQzlDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEQsT0FBTyxFQUFFLFNBQVM7WUFDbEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNuQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUU7WUFDdkQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDO1lBQy9ELFFBQVEsRUFBRSxzQkFBc0I7U0FDakMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsSUFBSSx1QkFBaUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ3hELE9BQU8sRUFBRSxJQUFJO1lBQ2IsVUFBVSxFQUFFLElBQUk7WUFDaEIsZUFBZSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ3BDLGVBQWUsRUFBRSxFQUFFO1NBQ3BCLENBQUMsQ0FBQztRQUVILE1BQU0sT0FBTyxHQUFHLElBQUksYUFBTyxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNoRCxPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ3ZDLFVBQVUsRUFBRSxJQUFJO1lBQ2hCLGlCQUFpQixFQUFFLHVCQUFpQixDQUFDLE1BQU07U0FDNUMsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzNELElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztTQUM1QyxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7UUFFakQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ2hELEtBQUssRUFBRTtnQkFDTCxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7YUFDN0I7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLFFBQVEsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7d0JBQ3pDLE1BQU0sRUFBRSx1QkFBdUI7d0JBQy9CLE1BQU0sRUFBRSxPQUFPO3FCQUNoQjtpQkFDRjthQUNGO1NBQ0YsRUFBRSxxQkFBWSxDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRW5DLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDLENBQUM7UUFFN0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlEQUF5RCxDQUFDLElBQVU7UUFDbEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtZQUNwRCxPQUFPLEVBQUUsU0FBUztZQUNsQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ25DLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDdEQsT0FBTyxFQUFFLElBQUk7U0FDZCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLCtCQUErQixDQUFDLENBQUM7UUFFdEYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJEQUEyRCxDQUFDLElBQVU7UUFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtZQUNwRCxPQUFPLEVBQUUsU0FBUztZQUNsQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ25DLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSx1QkFBaUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ3hELE9BQU8sRUFBRSxJQUFJO1lBQ2IsZUFBZSxFQUFFLENBQUUsb0JBQWMsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUU7U0FDdkQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRSwrQkFBK0IsQ0FBQyxDQUFDO1FBRXRGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UsIFJlc291cmNlUGFydCB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgeyBBdXRob3JpemF0aW9uVHlwZSwgSWRlbnRpdHlTb3VyY2UsIFJlcXVlc3RBdXRob3JpemVyLCBSZXN0QXBpLCBUb2tlbkF1dGhvcml6ZXIgfSBmcm9tICcuLi8uLi9saWInO1xuXG5leHBvcnQgPSB7XG4gICdkZWZhdWx0IHRva2VuIGF1dGhvcml6ZXInKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgZnVuYyA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdteWZ1bmN0aW9uJywge1xuICAgICAgaGFuZGxlcjogJ2hhbmRsZXInLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnZm9vJyksXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICB9KTtcblxuICAgIGNvbnN0IGF1dGggPSBuZXcgVG9rZW5BdXRob3JpemVyKHN0YWNrLCAnbXlhdXRob3JpemVyJywge1xuICAgICAgaGFuZGxlcjogZnVuY1xuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzdEFwaSA9IG5ldyBSZXN0QXBpKHN0YWNrLCAnbXlyZXN0YXBpJyk7XG4gICAgcmVzdEFwaS5yb290LmFkZE1ldGhvZCgnQU5ZJywgdW5kZWZpbmVkLCB7XG4gICAgICBhdXRob3JpemVyOiBhdXRoLFxuICAgICAgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkNVU1RPTVxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6QXV0aG9yaXplcicsIHtcbiAgICAgIFR5cGU6ICdUT0tFTicsXG4gICAgICBSZXN0QXBpSWQ6IHN0YWNrLnJlc29sdmUocmVzdEFwaS5yZXN0QXBpSWQpLFxuICAgICAgSWRlbnRpdHlTb3VyY2U6ICdtZXRob2QucmVxdWVzdC5oZWFkZXIuQXV0aG9yaXphdGlvbidcbiAgICB9KSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OlBlcm1pc3Npb24nLCB7XG4gICAgICBBY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgUHJpbmNpcGFsOiAnYXBpZ2F0ZXdheS5hbWF6b25hd3MuY29tJyxcbiAgICB9KSk7XG5cbiAgICB0ZXN0Lm9rKGF1dGguYXV0aG9yaXplckFybi5lbmRzV2l0aChgL2F1dGhvcml6ZXJzLyR7YXV0aC5hdXRob3JpemVySWR9YCksICdNYWxmb3JtZWQgYXV0aG9yaXplciBBUk4nKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWZhdWx0IHJlcXVlc3QgYXV0aG9yaXplcicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBjb25zdCBmdW5jID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ215ZnVuY3Rpb24nLCB7XG4gICAgICBoYW5kbGVyOiAnaGFuZGxlcicsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdmb28nKSxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMl9YLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYXV0aCA9IG5ldyBSZXF1ZXN0QXV0aG9yaXplcihzdGFjaywgJ215YXV0aG9yaXplcicsIHtcbiAgICAgIGhhbmRsZXI6IGZ1bmMsXG4gICAgICByZXN1bHRzQ2FjaGVUdGw6IER1cmF0aW9uLnNlY29uZHMoMCksXG4gICAgICBpZGVudGl0eVNvdXJjZXM6IFtdLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzdEFwaSA9IG5ldyBSZXN0QXBpKHN0YWNrLCAnbXlyZXN0YXBpJyk7XG4gICAgcmVzdEFwaS5yb290LmFkZE1ldGhvZCgnQU5ZJywgdW5kZWZpbmVkLCB7XG4gICAgICBhdXRob3JpemVyOiBhdXRoLFxuICAgICAgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkNVU1RPTVxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6QXV0aG9yaXplcicsIHtcbiAgICAgIFR5cGU6ICdSRVFVRVNUJyxcbiAgICAgIFJlc3RBcGlJZDogc3RhY2sucmVzb2x2ZShyZXN0QXBpLnJlc3RBcGlJZCksXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpQZXJtaXNzaW9uJywge1xuICAgICAgQWN0aW9uOiAnbGFtYmRhOkludm9rZUZ1bmN0aW9uJyxcbiAgICAgIFByaW5jaXBhbDogJ2FwaWdhdGV3YXkuYW1hem9uYXdzLmNvbScsXG4gICAgfSkpO1xuXG4gICAgdGVzdC5vayhhdXRoLmF1dGhvcml6ZXJBcm4uZW5kc1dpdGgoYC9hdXRob3JpemVycy8ke2F1dGguYXV0aG9yaXplcklkfWApLCAnTWFsZm9ybWVkIGF1dGhvcml6ZXIgQVJOJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW52YWxpZCByZXF1ZXN0IGF1dGhvcml6ZXIgY29uZmlnJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnbXlmdW5jdGlvbicsIHtcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgfSk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgUmVxdWVzdEF1dGhvcml6ZXIoc3RhY2ssICdteWF1dGhvcml6ZXInLCB7XG4gICAgICBoYW5kbGVyOiBmdW5jLFxuICAgICAgcmVzdWx0c0NhY2hlVHRsOiBEdXJhdGlvbi5zZWNvbmRzKDEpLFxuICAgICAgaWRlbnRpdHlTb3VyY2VzOiBbXSxcbiAgICB9KSwgRXJyb3IsICdBdCBsZWFzdCBvbmUgSWRlbnRpdHkgU291cmNlIGlzIHJlcXVpcmVkIGZvciBhIFJFUVVFU1QtYmFzZWQgTGFtYmRhIGF1dGhvcml6ZXIgaWYgY2FjaGluZyBpcyBlbmFibGVkLicpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rva2VuIGF1dGhvcml6ZXIgd2l0aCBhbGwgcGFyYW1ldGVycyBzcGVjaWZpZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgZnVuYyA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdteWZ1bmN0aW9uJywge1xuICAgICAgaGFuZGxlcjogJ2hhbmRsZXInLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnZm9vJyksXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICB9KTtcblxuICAgIGNvbnN0IGF1dGggPSBuZXcgVG9rZW5BdXRob3JpemVyKHN0YWNrLCAnbXlhdXRob3JpemVyJywge1xuICAgICAgaGFuZGxlcjogZnVuYyxcbiAgICAgIGlkZW50aXR5U291cmNlOiAnbWV0aG9kLnJlcXVlc3QuaGVhZGVyLndob2FtaScsXG4gICAgICB2YWxpZGF0aW9uUmVnZXg6ICdhLWhhY2tlcicsXG4gICAgICBhdXRob3JpemVyTmFtZTogJ215YXV0aG9yaXplcicsXG4gICAgICByZXN1bHRzQ2FjaGVUdGw6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgfSk7XG5cbiAgICBjb25zdCByZXN0QXBpID0gbmV3IFJlc3RBcGkoc3RhY2ssICdteXJlc3RhcGknKTtcbiAgICByZXN0QXBpLnJvb3QuYWRkTWV0aG9kKCdBTlknLCB1bmRlZmluZWQsIHtcbiAgICAgIGF1dGhvcml6ZXI6IGF1dGgsXG4gICAgICBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGUuQ1VTVE9NXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpBdXRob3JpemVyJywge1xuICAgICAgVHlwZTogJ1RPS0VOJyxcbiAgICAgIFJlc3RBcGlJZDogc3RhY2sucmVzb2x2ZShyZXN0QXBpLnJlc3RBcGlJZCksXG4gICAgICBJZGVudGl0eVNvdXJjZTogJ21ldGhvZC5yZXF1ZXN0LmhlYWRlci53aG9hbWknLFxuICAgICAgSWRlbnRpdHlWYWxpZGF0aW9uRXhwcmVzc2lvbjogJ2EtaGFja2VyJyxcbiAgICAgIE5hbWU6ICdteWF1dGhvcml6ZXInLFxuICAgICAgQXV0aG9yaXplclJlc3VsdFR0bEluU2Vjb25kczogNjBcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAncmVxdWVzdCBhdXRob3JpemVyIHdpdGggYWxsIHBhcmFtZXRlcnMgc3BlY2lmaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnbXlmdW5jdGlvbicsIHtcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgfSk7XG5cbiAgICBjb25zdCBhdXRoID0gbmV3IFJlcXVlc3RBdXRob3JpemVyKHN0YWNrLCAnbXlhdXRob3JpemVyJywge1xuICAgICAgaGFuZGxlcjogZnVuYyxcbiAgICAgIGlkZW50aXR5U291cmNlczogW0lkZW50aXR5U291cmNlLmhlYWRlcignd2hvYW1pJyldLFxuICAgICAgYXV0aG9yaXplck5hbWU6ICdteWF1dGhvcml6ZXInLFxuICAgICAgcmVzdWx0c0NhY2hlVHRsOiBEdXJhdGlvbi5taW51dGVzKDEpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzdEFwaSA9IG5ldyBSZXN0QXBpKHN0YWNrLCAnbXlyZXN0YXBpJyk7XG4gICAgcmVzdEFwaS5yb290LmFkZE1ldGhvZCgnQU5ZJywgdW5kZWZpbmVkLCB7XG4gICAgICBhdXRob3JpemVyOiBhdXRoLFxuICAgICAgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkNVU1RPTVxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6QXV0aG9yaXplcicsIHtcbiAgICAgIFR5cGU6ICdSRVFVRVNUJyxcbiAgICAgIFJlc3RBcGlJZDogc3RhY2sucmVzb2x2ZShyZXN0QXBpLnJlc3RBcGlJZCksXG4gICAgICBJZGVudGl0eVNvdXJjZTogJ21ldGhvZC5yZXF1ZXN0LmhlYWRlci53aG9hbWknLFxuICAgICAgTmFtZTogJ215YXV0aG9yaXplcicsXG4gICAgICBBdXRob3JpemVyUmVzdWx0VHRsSW5TZWNvbmRzOiA2MFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0b2tlbiBhdXRob3JpemVyIHdpdGggYXNzdW1lIHJvbGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgZnVuYyA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdteWZ1bmN0aW9uJywge1xuICAgICAgaGFuZGxlcjogJ2hhbmRsZXInLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnZm9vJyksXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdhdXRob3JpemVyYXNzdW1lcm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdhcGlnYXRld2F5LmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIHJvbGVOYW1lOiAnYXV0aG9yaXplcmFzc3VtZXJvbGUnXG4gICAgfSk7XG5cbiAgICBjb25zdCBhdXRoID0gbmV3IFRva2VuQXV0aG9yaXplcihzdGFjaywgJ215YXV0aG9yaXplcicsIHtcbiAgICAgIGhhbmRsZXI6IGZ1bmMsXG4gICAgICBhc3N1bWVSb2xlOiByb2xlXG4gICAgfSk7XG5cbiAgICBjb25zdCByZXN0QXBpID0gbmV3IFJlc3RBcGkoc3RhY2ssICdteXJlc3RhcGknKTtcbiAgICByZXN0QXBpLnJvb3QuYWRkTWV0aG9kKCdBTlknLCB1bmRlZmluZWQsIHtcbiAgICAgIGF1dGhvcml6ZXI6IGF1dGgsXG4gICAgICBhdXRob3JpemF0aW9uVHlwZTogQXV0aG9yaXphdGlvblR5cGUuQ1VTVE9NXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpBcGlHYXRld2F5OjpBdXRob3JpemVyJywge1xuICAgICAgVHlwZTogJ1RPS0VOJyxcbiAgICAgIFJlc3RBcGlJZDogc3RhY2sucmVzb2x2ZShyZXN0QXBpLnJlc3RBcGlJZCksXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpSb2xlJykpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBSb2xlczogW1xuICAgICAgICBzdGFjay5yZXNvbHZlKHJvbGUucm9sZU5hbWUpXG4gICAgICBdLFxuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVzb3VyY2U6IHN0YWNrLnJlc29sdmUoZnVuYy5mdW5jdGlvbkFybiksXG4gICAgICAgICAgICBBY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgIH1cbiAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcywgdHJ1ZSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpQZXJtaXNzaW9uJykpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3JlcXVlc3QgYXV0aG9yaXplciB3aXRoIGFzc3VtZSByb2xlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnbXlmdW5jdGlvbicsIHtcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgfSk7XG5cbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnYXV0aG9yaXplcmFzc3VtZXJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnYXBpZ2F0ZXdheS5hbWF6b25hd3MuY29tJyksXG4gICAgICByb2xlTmFtZTogJ2F1dGhvcml6ZXJhc3N1bWVyb2xlJ1xuICAgIH0pO1xuXG4gICAgY29uc3QgYXV0aCA9IG5ldyBSZXF1ZXN0QXV0aG9yaXplcihzdGFjaywgJ215YXV0aG9yaXplcicsIHtcbiAgICAgIGhhbmRsZXI6IGZ1bmMsXG4gICAgICBhc3N1bWVSb2xlOiByb2xlLFxuICAgICAgcmVzdWx0c0NhY2hlVHRsOiBEdXJhdGlvbi5zZWNvbmRzKDApLFxuICAgICAgaWRlbnRpdHlTb3VyY2VzOiBbXVxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzdEFwaSA9IG5ldyBSZXN0QXBpKHN0YWNrLCAnbXlyZXN0YXBpJyk7XG4gICAgcmVzdEFwaS5yb290LmFkZE1ldGhvZCgnQU5ZJywgdW5kZWZpbmVkLCB7XG4gICAgICBhdXRob3JpemVyOiBhdXRoLFxuICAgICAgYXV0aG9yaXphdGlvblR5cGU6IEF1dGhvcml6YXRpb25UeXBlLkNVU1RPTVxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QXBpR2F0ZXdheTo6QXV0aG9yaXplcicsIHtcbiAgICAgIFR5cGU6ICdSRVFVRVNUJyxcbiAgICAgIFJlc3RBcGlJZDogc3RhY2sucmVzb2x2ZShyZXN0QXBpLnJlc3RBcGlJZCksXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpSb2xlJykpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBSb2xlczogW1xuICAgICAgICBzdGFjay5yZXNvbHZlKHJvbGUucm9sZU5hbWUpXG4gICAgICBdLFxuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVzb3VyY2U6IHN0YWNrLnJlc29sdmUoZnVuYy5mdW5jdGlvbkFybiksXG4gICAgICAgICAgICBBY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgIH1cbiAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcywgdHJ1ZSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpQZXJtaXNzaW9uJykpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3Rva2VuIGF1dGhvcml6ZXIgdGhyb3dzIHdoZW4gbm90IGF0dGFjaGVkIHRvIGEgcmVzdCBhcGknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnbXlmdW5jdGlvbicsIHtcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgfSk7XG4gICAgY29uc3QgYXV0aCA9IG5ldyBUb2tlbkF1dGhvcml6ZXIoc3RhY2ssICdteWF1dGhvcml6ZXInLCB7XG4gICAgICBoYW5kbGVyOiBmdW5jLFxuICAgIH0pO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gc3RhY2sucmVzb2x2ZShhdXRoLmF1dGhvcml6ZXJBcm4pLCAvbXVzdCBiZSBhdHRhY2hlZCB0byBhIFJlc3RBcGkvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdyZXF1ZXN0IGF1dGhvcml6ZXIgdGhyb3dzIHdoZW4gbm90IGF0dGFjaGVkIHRvIGEgcmVzdCBhcGknKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnbXlmdW5jdGlvbicsIHtcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgfSk7XG4gICAgY29uc3QgYXV0aCA9IG5ldyBSZXF1ZXN0QXV0aG9yaXplcihzdGFjaywgJ215YXV0aG9yaXplcicsIHtcbiAgICAgIGhhbmRsZXI6IGZ1bmMsXG4gICAgICBpZGVudGl0eVNvdXJjZXM6IFsgSWRlbnRpdHlTb3VyY2UuaGVhZGVyKCdteWhlYWRlcicpIF0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBzdGFjay5yZXNvbHZlKGF1dGguYXV0aG9yaXplckFybiksIC9tdXN0IGJlIGF0dGFjaGVkIHRvIGEgUmVzdEFwaS8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuIl19