"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.withResolved = exports.isResolvableObject = exports.Tokenization = exports.Token = exports.TokenComparison = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * An enum-like class that represents the result of comparing two Tokens.
 *
 * The return type of {@link Token.compareStrings}.
 *
 * @stability stable
 */
class TokenComparison {
    constructor() {
    }
}
exports.TokenComparison = TokenComparison;
_a = JSII_RTTI_SYMBOL_1;
TokenComparison[_a] = { fqn: "@aws-cdk/core.TokenComparison", version: "1.107.0" };
/**
 * This means we're certain the two components are NOT Tokens, and identical.
 *
 * @stability stable
 */
TokenComparison.SAME = new TokenComparison();
/**
 * This means we're certain the two components are NOT Tokens, and different.
 *
 * @stability stable
 */
TokenComparison.DIFFERENT = new TokenComparison();
/**
 * This means exactly one of the components is a Token.
 *
 * @stability stable
 */
TokenComparison.ONE_UNRESOLVED = new TokenComparison();
/**
 * This means both components are Tokens.
 *
 * @stability stable
 */
TokenComparison.BOTH_UNRESOLVED = new TokenComparison();
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 *
 * @stability stable
 */
class Token {
    constructor() {
    }
    /**
     * Returns true if obj represents an unresolved value.
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     * @stability stable
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token.
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     *
     * @stability stable
     */
    static asString(value, options = {}) {
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token.
     *
     * @stability stable
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token.
     *
     * @stability stable
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value.
     *
     * @stability stable
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    /**
     * Compare two strings that might contain Tokens with each other.
     *
     * @stability stable
     */
    static compareStrings(possibleToken1, possibleToken2) {
        const firstIsUnresolved = Token.isUnresolved(possibleToken1);
        const secondIsUnresolved = Token.isUnresolved(possibleToken2);
        if (firstIsUnresolved && secondIsUnresolved) {
            return TokenComparison.BOTH_UNRESOLVED;
        }
        if (firstIsUnresolved || secondIsUnresolved) {
            return TokenComparison.ONE_UNRESOLVED;
        }
        return possibleToken1 === possibleToken2 ? TokenComparison.SAME : TokenComparison.DIFFERENT;
    }
}
exports.Token = Token;
_b = JSII_RTTI_SYMBOL_1;
Token[_b] = { fqn: "@aws-cdk/core.Token", version: "1.107.0" };
/**
 * Less oft-needed functions to manipulate Tokens.
 *
 * @stability stable
 */
class Tokenization {
    constructor() {
    }
    /**
     * Un-encode a string potentially containing encoded tokens.
     *
     * @stability stable
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a string which is either a complete encoded token, or doesn't contain tokens at all.
     *
     * It's illegal for the string to be a concatenation of an encoded token and something else.
     *
     * @stability stable
     */
    static reverseCompleteString(s) {
        const fragments = Tokenization.reverseString(s);
        if (fragments.length !== 1) {
            throw new Error(`Tokenzation.reverseCompleteString: argument must not be a concatentation, got '${s}'`);
        }
        return fragments.firstToken;
    }
    /**
     * Un-encode a Tokenized value from a number.
     *
     * @stability stable
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list.
     *
     * @stability stable
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Reverse any value into a Resolvable, if possible.
     *
     * In case of a string, the string must not be a concatenation.
     *
     * @stability stable
     */
    static reverse(x, options = {}) {
        if (Tokenization.isResolvable(x)) {
            return x;
        }
        if (typeof x === 'string') {
            if (options.failConcat === false) {
                // Handle this specially because reverseCompleteString might fail
                const fragments = Tokenization.reverseString(x);
                return fragments.length === 1 ? fragments.firstToken : undefined;
            }
            return Tokenization.reverseCompleteString(x);
        }
        if (Array.isArray(x)) {
            return Tokenization.reverseList(x);
        }
        if (typeof x === 'number') {
            return Tokenization.reverseNumber(x);
        }
        return undefined;
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     *
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     * @stability stable
     */
    static resolve(obj, options) {
        var _d;
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: ((_d = options.preparing) !== null && _d !== void 0 ? _d : false),
        });
    }
    /**
     * Return whether the given object is an IResolvable object.
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     *
     * @stability stable
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token.
     *
     * If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     *
     * @stability stable
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.uncachedString({
                produce: context => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== 'number' ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== 'number' ? x : `${x}`;
        }
    }
}
exports.Tokenization = Tokenization;
_c = JSII_RTTI_SYMBOL_1;
Tokenization[_c] = { fqn: "@aws-cdk/core.Tokenization", version: "1.107.0" };
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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