# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['numpydoc_decorator']

package_data = \
{'': ['*']}

install_requires = \
['typing_extensions']

setup_kwargs = {
    'name': 'numpydoc-decorator',
    'version': '1.5.1',
    'description': '',
    'long_description': '# numpydoc_decorator\n\nThis package allows you to build [numpy-style\ndocstrings](https://numpydoc.readthedocs.io/en/latest/format.html#sections)\nprogrammatically and apply them using a decorator. This can be useful\nbecause:\n\n-   Parts of your documentation, such as parameter descriptions, can be\n    shared between functions, avoiding the need to repeat yourself.\n\n-   Type information for parameters and return values is automatically\n    picked up from type annotations and added to the docstring, avoiding\n    the need to maintain type information in two places.\n\n## Installation\n\n`pip install numpydoc_decorator`\n\n## Usage\n\n### Documentation a function\n\nHere is an example of documenting a function:\n\n```python\nfrom numpydoc_decorator import doc\n\n\n@doc(\n    summary="Say hello to someone.",\n    extended_summary="""\n        Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do\n        eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad\n        minim veniam, quis nostrud exercitation ullamco laboris nisi ut\n        aliquip ex ea commodo consequat. Duis aute irure dolor in\n        reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla\n        pariatur. Excepteur sint occaecat cupidatat non proident, sunt in\n        culpa qui officia deserunt mollit anim id est laborum.\n    """,\n    parameters=dict(\n        name="The name of the person to greet.",\n        language="The language in which to greet as an ISO 639-1 code.",\n    ),\n    returns="A pleasant greeting.",\n    raises=dict(\n        NotImplementedError="If the requested language has not been implemented yet.",\n        ValueError="If the language is not a valid ISO 639-1 code."\n    ),\n    see_also=dict(\n        print="You could use this function to print your greeting.",\n    ),\n    notes="""\n        This function is useful when greeting someone else. If you would\n        like something to talk about next, you could try [1]_.\n    """,\n    references={\n        "1": """\n            O. McNoleg, "The integration of GIS, remote sensing, expert systems\n            and adaptive co-kriging for environmental habitat modelling of the\n            Highland Haggis using object-oriented, fuzzy-logic and neural-\n            network techniques," Computers & Geosciences, vol. 22, pp. 585-588,\n            1996.\n        """,\n    },\n    examples="""\n        Here is how to greet a friend in English:\n\n        >>> print(greet("Ford Prefect"))\n        Hello Ford Prefect!\n\n        Here is how to greet someone in another language:\n\n        >>> print(greet("Tricia MacMillan", language="fr"))\n        Salut Tricia MacMillan!\n\n    """,\n)\ndef greet(\n    name: str,\n    language: str = "en",\n) -> str:\n    if len(language) != 2:\n        raise ValueError("language must be an ISO 639-1 code")\n    if language == "en":\n        return f"Hello {name}!"\n    elif language == "fr":\n        return f"Salut {name}!"\n    else:\n        raise NotImplementedError(f"language {language} not implemented")\n```\n\nHere is the docstring that will be created and attached to the\ndecorated function:\n\n```\n>>> print(greet.__doc__)\n\nSay hello to someone.\n\nLorem ipsum dolor sit amet, consectetur adipiscing elit, sed do\neiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad\nminim veniam, quis nostrud exercitation ullamco laboris nisi ut\naliquip ex ea commodo consequat. Duis aute irure dolor in\nreprehenderit in voluptate velit esse cillum dolore eu fugiat nulla\npariatur. Excepteur sint occaecat cupidatat non proident, sunt in\nculpa qui officia deserunt mollit anim id est laborum.\n\nParameters\n----------\nname : str\n    The name of the person to greet.\nlanguage : str, optional, default: \'en\'\n    The language in which to greet as an ISO 639-1 code.\n\nReturns\n-------\nstr\n    A pleasant greeting.\n\nRaises\n------\nNotImplementedError\n    If the requested language has not been implemented yet.\nValueError\n    If the language is not a valid ISO 639-1 code.\n\nSee Also\n--------\nprint : You could use this function to print your greeting.\n\nNotes\n-----\nThis function is useful when greeting someone else. If you would like\nsomething to talk about next, you could try [1]_.\n\nReferences\n----------\n.. [1] O. McNoleg, "The integration of GIS, remote sensing, expert systems\n    and adaptive co-kriging for environmental habitat modelling of the\n    Highland Haggis using object-oriented, fuzzy-logic and neural- network\n    techniques," Computers & Geosciences, vol. 22, pp. 585-588, 1996.\n\nExamples\n--------\nHere is how to greet a friend in English:\n\n>>> print(greet("Ford Prefect"))\nHello Ford Prefect!\n\nHere is how to greet someone in another language:\n\n>>> print(greet("Tricia MacMillan", language="fr"))\nSalut Tricia MacMillan!\n\n```\n\n### Shared parameters\n\nIf you have parameters which are common to multiple functions, here\nis an approach you can take:\n\n```python\nfrom numpydoc_decorator import doc\nfrom typing_extensions import Annotated\n\n\nclass params:\n    name = Annotated[str, "The name of a person."]\n    language = Annotated[str, "An ISO 639-1 language code."]\n\n\n@doc(\n    summary="Say hello to someone you know.",\n    returns="A personal greeting.",\n)\ndef say_hello(\n    name: params.name,\n    language: params.language,\n) -> str:\n    pass\n\n\n@doc(\n    summary="Say goodbye to someone you know.",\n    returns="A personal parting.",\n)\ndef say_goodbye(\n    name: params.name,\n    language: params.language,\n) -> str:\n    pass\n```\n\nHere are the generated docstrings:\n\n```\n>>> print(say_hello.__doc__)\n\nSay hello to someone you know.\n\nParameters\n----------\nname : str\n    The name of a person.\nlanguage : str\n    An ISO 639-1 language code.\n\nReturns\n-------\nstr\n    A personal greeting.\n```\n\n```\n>>> print(say_goodbye.__doc__)\n\nSay goodbye to someone you know.\n\nParameters\n----------\nname : str\n    The name of a person.\nlanguage : str\n    An ISO 639-1 language code.\n\nReturns\n-------\nstr\n    A personal parting.\n```\n\n## Notes\n\nThere are probably lots of edge cases that this package has not\ncovered yet. If you find something doesn\'t work as expected, or\ndeviates from the [numpydoc style guide](https://numpydoc.readthedocs.io/en/latest/format.html)\nin an unreasonable way, please feel free to submit a pull request.\n\nNote that this package does deviate from the numpydoc style guide\nunder some circumstances. For example, if a function does not have\nany type annotations, then there will be no type information in the\ndocstring. The rationale for this is that all type information, if\nprovided, should be provided through type annotations. However, some\nfunctions may choose not to annotate types for some or all parameters,\nbut we still want to document them as best we can.\n',
    'author': 'Alistair Miles',
    'author_email': 'alimanfoo@googlemail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<3.11',
}


setup(**setup_kwargs)
