# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aiomisc',
 'aiomisc.service',
 'aiomisc_log',
 'aiomisc_log.formatter',
 'aiomisc_pytest',
 'aiomisc_worker']

package_data = \
{'': ['*']}

install_requires = \
['colorlog>=6.0,<7.0']

extras_require = \
{':sys_platform == "linux"': ['logging-journald'],
 'aiohttp': ['aiohttp'],
 'asgi': ['aiohttp-asgi>=0.4.3,<0.5.0'],
 'carbon': ['aiocarbon>=0.15,<0.16'],
 'cron': ['croniter>=1.3.8,<2.0.0'],
 'raven': ['aiohttp', 'raven'],
 'rich': ['rich'],
 'uvloop': ['uvloop>=0.14,<1']}

entry_points = \
{'pytest11': ['aiomisc = aiomisc_pytest.pytest_plugin']}

setup_kwargs = {
    'name': 'aiomisc',
    'version': '16.3.15',
    'description': 'aiomisc - miscellaneous utils for asyncio',
    'long_description': 'aiomisc - miscellaneous utils for asyncio\n=========================================\n\n.. image:: https://coveralls.io/repos/github/aiokitchen/aiomisc/badge.svg?branch=master\n   :target: https://coveralls.io/github/aiokitchen/aiomisc\n   :alt: Coveralls\n\n.. image:: https://github.com/aiokitchen/aiomisc/workflows/tox/badge.svg\n   :target: https://github.com/aiokitchen/aiomisc/actions?query=workflow%3Atox\n   :alt: Actions\n\n.. image:: https://img.shields.io/pypi/v/aiomisc.svg\n   :target: https://pypi.python.org/pypi/aiomisc/\n   :alt: Latest Version\n\n.. image:: https://img.shields.io/pypi/wheel/aiomisc.svg\n   :target: https://pypi.python.org/pypi/aiomisc/\n\n.. image:: https://img.shields.io/pypi/pyversions/aiomisc.svg\n   :target: https://pypi.python.org/pypi/aiomisc/\n\n.. image:: https://img.shields.io/pypi/l/aiomisc.svg\n   :target: https://pypi.python.org/pypi/aiomisc/\n\n\nMiscellaneous utils for asyncio.\n\nAs a programmer, you are no stranger to the challenges that come with building\nand maintaining software applications. One area that can be particularly\ndifficult is making architecture of the software that using asynchronous I/O.\n\nThis is where aiomisc comes in. aiomisc is a Python library that provides a\ncollection of utility functions and classes for working with asynchronous I/O\nin a more intuitive and efficient way. It is built on top of the ``asyncio``\nlibrary and is designed to make it easier for developers to write\nasynchronous code that is both reliable and scalable.\n\nWith aiomisc, you can take advantage of powerful features like\n``worker pools``, ``connection pools``, ``circuit breaker pattern``,\nand retry mechanisms such as ``asyncbackoff`` and ``asyncretry`` to make your\nasyncio code more robust and easier to maintain. In this documentation,\nwe\'ll take a closer look at what ``aiomisc`` has to offer and how it can\nhelp you streamline your asyncio service development.\n\nInstallation\n------------\n\nInstallation is possible in standard ways, such as PyPI or installation from\na git repository directly.\n\nInstalling from PyPI_:\n\n.. code-block:: bash\n\n    pip3 install aiomisc\n\nInstalling from github.com:\n\n.. code-block:: bash\n\n    # Using git tool\n    pip3 install git+https://github.com/aiokitchen/aiomisc.git\n\n    # Alternative way using http\n    pip3 install \\\n        https://github.com/aiokitchen/aiomisc/archive/refs/heads/master.zip\n\nThe package contains several extras and you can install additional dependencies\nif you specify them in this way.\n\nWith uvloop_:\n\n.. code-block:: bash\n\n    pip3 install "aiomisc[uvloop]"\n\n\nWith aiohttp_:\n\n.. code-block:: bash\n\n    pip3 install "aiomisc[aiohttp]"\n\nComplete table of extras bellow:\n\n+-----------------------------------+------------------------------------------------+\n| example                           |  description                                   |\n+===================================+================================================+\n| ``pip install aiomisc[aiohttp]``  | For running aiohttp_ applications.             |\n+-----------------------------------+------------------------------------------------+\n| ``pip install aiomisc[asgi]``     | For running ASGI_ applications                 |\n+-----------------------------------+------------------------------------------------+\n| ``pip install aiomisc[carbon]``   | Sending metrics to carbon_ (part of graphite_) |\n+-----------------------------------+------------------------------------------------+\n| ``pip install aiomisc[cron]``     | use croniter_ for scheduling tasks             |\n+-----------------------------------+------------------------------------------------+\n| ``pip install aiomisc[raven]``    | Sending exceptions to sentry_ using raven_     |\n+-----------------------------------+------------------------------------------------+\n| ``pip install aiomisc[rich]``     | You might using rich_ for logging              |\n+-----------------------------------+------------------------------------------------+\n| ``pip install aiomisc[uvloop]``   | use uvloop_ as a default event loop            |\n+-----------------------------------+------------------------------------------------+\n\n.. _ASGI: https://asgi.readthedocs.io/en/latest/\n.. _PyPI: https://pypi.org/\n.. _aiohttp: https://pypi.org/project/aiohttp\n.. _carbon: https://pypi.org/project/carbon\n.. _croniter: https://pypi.org/project/croniter\n.. _graphite: http://graphiteapp.org\n.. _raven: https://pypi.org/project/raven\n.. _rich: https://pypi.org/project/rich\n.. _sentry: https://sentry.io/\n.. _uvloop: https://pypi.org/project/uvloop\n\nYou can combine extras values by separating them with commas, for example:\n\n.. code-block:: bash\n\n    pip3 install "aiomisc[aiohttp,cron,rich,uvloop]"\n\n\nQuick Start\n-----------\n\nThis section will cover how this library creates and uses the event loop and\ncreates services. Of course, you can\'t write about everything here, but you\ncan read about a lot in the Tutorial_ section, and you can\nalways refer to the Modules_ and `API reference`_ sections for help.\n\nEvent-loop and entrypoint\n+++++++++++++++++++++++++\n\nLet\'s look at this simple example first:\n\n.. code-block:: python\n\n    import asyncio\n    import logging\n\n    import aiomisc\n\n    log = logging.getLogger(__name__)\n\n    async def main():\n        log.info(\'Starting\')\n        await asyncio.sleep(3)\n        log.info(\'Exiting\')\n\n\n    if __name__ == \'__main__\':\n        with entrypoint(log_level="info", log_format="color") as loop:\n            loop.run_until_complete(main())\n\n\nThis code declares an asynchronous ``main()`` function that exits after\n3 seconds. It would seem nothing interesting, but the whole point is in\nthe ``entrypoint``.\n\nWhat does the ``entrypoint`` do, it would seem not so much, it creates an\nevent-loop and transfers control to the user. However, under the hood, the\nlogger is configured in a separate thread, a pool of threads is created,\nservices are started, but more on that later and there are no services\nin this example.\n\nAlternatively, you can choose not to use an entrypoint, just create an\nevent-loop and set this as a default event loop for current thread:\n\n.. code-block:: python\n    :name: test_index_get_loop\n\n    import asyncio\n    import aiomisc\n\n    # * Installs uvloop event loop is it\'s has been installed.\n    # * Creates and set `aiomisc.thread_pool.ThreadPoolExecutor`\n    #   as a default executor\n    # * Sets just created event-loop as a current event-loop for this thread.\n    aiomisc.new_event_loop()\n\n    async def main():\n        await asyncio.sleep(1)\n\n    if __name__ == \'__main__\':\n        loop = asyncio.get_event_loop()\n        loop.run_until_complete(main())\n\nThe example above is useful if your code is already using an implicitly created\nevent loop, you will have to modify less code, just add\n``aiomisc.new_event_loop()`` and all calls to ``asyncio.get_event_loop()``\nwill return the created instance.\n\nHowever, you can do with one call. Following example closes implicitly created\nasyncio event loop and install a new one:\n\n.. code-block:: python\n    :name: test_index_new_loop\n\n    import asyncio\n    import aiomisc\n\n    async def main():\n        await asyncio.sleep(3)\n\n    if __name__ == \'__main__\':\n        loop = aiomisc.new_event_loop()\n        loop.run_until_complete(main())\n\nServices\n++++++++\n\nThe main thing that an ``entrypoint`` does is start and gracefully\nstop services.\n\nThe service concept within this library means a class derived from\nthe ``aiosmic.Service`` class and implementing the\n``async def start(self) -> None:`` method and optionally the\n``async def stop(self, exc: Optional[ Exception]) -> None`` method.\n\nThe concept of stopping a service is not necessarily is pressing ``Ctrl+C``\nkeys by user, it\'s actually just exiting the ``entrypoint`` context manager.\n\nThe example below shows what your service might look like:\n\n.. code-block:: python\n\n    from aiomisc import entrypoint, Service\n\n    class MyService(Service):\n        async def start(self):\n            do_something_when_start()\n\n        async def stop(self, exc):\n            do_graceful_shutdown()\n\n\n    with entrypoint(MyService()) as loop:\n        loop.run_forever()\n\nThe entry point can start as many instances of the service as it likes,\nand all of them will start concurrently.\n\nThere is also a way if the ``start`` method is a payload for a service,\nand then there is no need to implement the stop method, since the running\ntask with the ``start`` function will be canceled at the stop stage.\nBut in this case, you will have to notify the ``entrypoint`` that the\ninitialization of the service instance is complete and it can continue.\n\nLike this:\n\n.. code-block:: python\n\n    import asyncio\n    from threading import Event\n    from aiomisc import entrypoint, Service\n\n    event = Event()\n\n    class MyService(Service):\n        async def start(self):\n            # Send signal to entrypoint for continue running\n            self.start_event.set()\n            await asyncio.sleep(3600)\n\n\n    with entrypoint(MyService()) as loop:\n        assert event.is_set()\n\n.. note::\n\n    The ``entrypoint`` passes control to the body of the context manager only\n    after all service instances have started. As mentioned above, a start is\n    considered to be the completion of the ``start`` method or the setting of\n    an start event with ``self.start_event.set()``.\n\nThe whole power of this library is in the set of already implemented or\nabstract services.\nSuch as: ``AIOHTTPService``, ``ASGIService``, ``TCPServer``,\n``UDPServer``, ``TCPClient``, ``PeriodicService``, ``CronService`` and so on.\n\nUnfortunately in this section it is not possible to pay more attention to this,\nplease pay attention to the Tutorial_ section section, there are more\nexamples and explanations, and of cource you always can find out an answer on\nthe `/api/index` or in the source code. The authors have tried to make\nthe source code as clear and simple as possible, so feel free to explore it.\n\n\nVersioning\n----------\n\nThis software follows `Semantic Versioning`_\n\n\nSummary: it\'s given a version number MAJOR.MINOR.PATCH, increment the:\n\n* MAJOR version when you make incompatible API changes\n* MINOR version when you add functionality in a backwards compatible manner\n* PATCH version when you make backwards compatible bug fixes\n* Additional labels for pre-release and build metadata are available as\n  extensions to the MAJOR.MINOR.PATCH format.\n\nIn this case, the package version is assigned automatically with poem-plugins_,\nit using on the tag in the repository as a major and minor and the counter,\nwhich takes the number of commits between tag to the head of branch.\n\n.. _poem-plugins: https://pypi.org/project/poem-plugins\n\n\nHow to develop?\n---------------\n\nThis project, like most open source projects, is developed by enthusiasts,\nyou can join the development, submit issues, or send your merge requests.\n\nIn order to start developing in this repository, you need to do the\nfollowing things.\n\nShould be installed:\n\n* Python 3.7+ as ``python3``\n* Installed Poetry_ as ``poetry``\n\n.. _Poetry: https://python-poetry.org/docs/\n\nFor setting up developer environment just execute:\n\n    .. code-block::\n\n        # installing all dependencies\n        poetry install\n\n        # setting up pre-commit hooks\n        poetry run pre-commit install\n\n        # adding poem-plugins to the poetry\n        poetry self add poem-plugins\n\n\n.. _Semantic Versioning: http://semver.org/\n\n.. _API reference: https://aiomisc.readthedocs.io/en/latest/api/index.html\n.. _Modules: https://aiomisc.readthedocs.io/en/latest/modules.html\n.. _Tutorial: https://aiomisc.readthedocs.io/en/latest/tutorial.html\n',
    'author': 'Dmitry Orlov',
    'author_email': 'me@mosquito.su',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/aiokitchen/aiomisc',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
