import datetime
import io
from typing import Any, Optional, Dict, List
from typing_extensions import TypedDict

import pandas as pd
from gql import gql

from .client import execute
from .enums import ColumnType, DataSourceEventType, S3AuthenticationMode


class StorageInfoRaw(TypedDict):
    """Storage info for `fsspec` compatible input, e.g. public link.

    Attributes:
        connectionString: The connection string for the storage backend.
        options: Additional options for the storage backend.
    """
    connectionString: str
    options: Dict[str, Any]


class StorageInfoAzureBlob(TypedDict):
    """Storage info for Azure Blob Storage.

    Attributes:
        accountName: The name of the storage account.
        container: The name of the container.
        path: The path to the data within the container.
        isPublic: Whether the data is publicly accessible.
        accountKey: The account key for the storage account. Should only be provided when using account key to
            authenticate.
        sasToken: The SAS (Shared Access Signature) token for the storage account. Should only be provided when using a
            SAS token to authenticate.
    """
    accountName: str
    container: str
    path: str
    isPublic: bool
    accountKey: Optional[str]
    sasToken: Optional[str]


class StorageInfoS3(TypedDict):
    """Storage info for Amazon S3.

    Attributes:
        uri: The S3 URI for the data, e.g. `s3://bucket/data.csv`.
        authenticationMode: The authentication mode to use.
        awsAccessKeyId: ID for the access key. Should only be provided when using access key to authenticate.
        awsSecretAccessKey: Secret for the access key. Should only be provided when using access key to authenticate.
    """
    uri: str
    authenticationMode: S3AuthenticationMode
    awsAccessKeyId: Optional[str]
    awsSecretAccessKey: Optional[str]


class StorageInfoCache(TypedDict):
    """Storage info for cached data.

    Attributes:
        id: The ID of the cached dataset. This ID is generated by the NannyML Cloud server when uploading data using
            [Data.upload][nannyml_cloud_sdk.data.Data.upload].
    """
    id: str


class StorageInfo(TypedDict, total=False):
    """Storage info for data.

    NannyML Cloud accepts data from various sources. This type describes the different storage info types that can be
    used to specify the location of the data. Only one of the storage info types should be provided.

    Attributes:
        raw: Storage info for `fsspec` compatible input, e.g. public link.
        azure: Storage info for Azure Blob Storage.
        s3: Storage info for Amazon S3.
        cache: Storage info for cached data.
    """
    raw: Optional[StorageInfoRaw]
    azure: Optional[StorageInfoAzureBlob]
    s3: Optional[StorageInfoS3]
    cache: Optional[StorageInfoCache]


class ColumnDetails(TypedDict):
    """Details about a column in a model schema.

    Attributes:
        name: The name of the column.
        columnType: The type of the column.
        dataType: The data type of the column.
        className: Class name for prediction columns in a multiclass classification problem.
    """

    name: str
    columnType: ColumnType
    dataType: str
    className: Optional[str]
    columnFlags: List[str]


class DataSourceSummary(TypedDict):
    id: str
    name: str
    hasReferenceData: bool
    hasAnalysisData: bool
    nrRows: int


class DataSourceDetails(DataSourceSummary):
    columns: List[ColumnDetails]


class DataSourceFilter(TypedDict, total=False):
    name: str
    hasReferenceData: bool
    hasAnalysisData: bool


class DataSourceEvent(TypedDict):
    id: str
    eventType: DataSourceEventType
    timestamp: datetime.datetime
    nrRows: int


class DataSourceEventFilter(TypedDict, total=False):
    event_type: DataSourceEventType


COLUMN_DETAILS_FRAGMENT = f"""
    fragment ColumnDetails on Column {{
        {' '.join(ColumnDetails.__required_keys__)}
    }}
"""

DATA_SOURCE_SUMMARY_FRAGMENT = f"""
    fragment DataSourceSummary on DataSource {{
        {' '.join(DataSourceSummary.__required_keys__)}
    }}
"""

DATA_SOURCE_DETAILS_FRAGMENT = """
    fragment DataSourceDetails on DataSource {
        ...DataSourceSummary
        columns {
            ...ColumnDetails
        }
    }
""" + COLUMN_DETAILS_FRAGMENT + DATA_SOURCE_SUMMARY_FRAGMENT

DATA_SOURCE_EVENT_FRAGMENT = f"""
    fragment DataSourceEvent on DataSourceEvent {{
        {' '.join(DataSourceEvent.__required_keys__)}
    }}
"""

_UPLOAD_DATASET = gql("""
    mutation uploadDataset($file: Upload!) {
        upload_dataset(file: $file) {
            id
        }
    }
""")


# Some common snippets used across product modules

_ADD_DATA_TO_DATA_SOURCE = gql("""
    mutation addDataToDataSource($input: DataSourceDataInput!) {
        add_data_to_data_source(input: $input) {
            id
        }
    }
""")

_UPSERT_DATA_IN_DATA_SOURCE = gql("""
    mutation updateDataInDataSource($input: DataSourceDataInput!) {
        upsert_data_in_data_source(input: $input) {
            id
        }
    }
""")

_REMOVE_DATA_FROM_DATA_SOURCE = gql("""
    mutation removeDataFromDataSource($input: DataSourceDeleteInput!) {
        delete_data_from_data_source(input: $input) {
            id
        }
    }
""")


class Data:
    @classmethod
    def upload(cls, df: pd.DataFrame) -> StorageInfo:
        """Upload a pandas dataframe to NannyML Cloud

        Returns:
            str: The ID of the uploaded dataset
        """
        with io.BytesIO() as buffer:
            # Convert to parquet for transmission
            df.to_parquet(buffer)

            # Rewind buffer so we can transmit the file
            buffer.seek(0)

            # Set the content type so the server knows how to handle the file
            # This is the way gql expects to receive the `content-type` header, but it's not part of IOBase. Mypy
            # doesn't like the use of an unknown attribute, so we suppress mypy here.
            buffer.content_type = 'application/vnd.apache.parquet'  # type: ignore

            # Finally upload the data in parquet format
            upload = execute(_UPLOAD_DATASET, upload_files=True, variable_values={
                'file': buffer,
            })['upload_dataset']

            return {'cache': upload}
