import datetime
import functools
from typing import Optional, List, Dict

import pandas as pd
from frozendict import frozendict
from gql import gql

from nannyml_cloud_sdk._typing import TypedDict
from nannyml_cloud_sdk.client import execute
from nannyml_cloud_sdk.data import DATA_SOURCE_SUMMARY_FRAGMENT, DATA_SOURCE_EVENT_FRAGMENT, Data, DataSourceSummary, \
    DataSourceFilter, DataSourceEvent, _UPSERT_DATA_IN_DATA_SOURCE, _ADD_DATA_TO_DATA_SOURCE
from nannyml_cloud_sdk.enums import ProblemType, PerformanceMetric
from nannyml_cloud_sdk.errors import InvalidOperationError
from nannyml_cloud_sdk.model_evaluation.enums import HypothesisType
from nannyml_cloud_sdk.model_evaluation.run import RunSummary, RUN_SUMMARY_FRAGMENT
from nannyml_cloud_sdk.model_evaluation.schema import ModelSchema
from nannyml_cloud_sdk.schema import normalize


class ModelSummary(TypedDict):
    """Summary of a model.

    Attributes:
        id: Unique identifier of the model (generated by NannyML Cloud when a model is created).
        name: User-defined name of the model.
        problemType: Type of problem the model is trying to solve.
        createdAt: Timestamp when the model was created.
    """
    id: str
    name: str
    problemType: ProblemType
    createdAt: datetime.datetime


class ModelDetails(ModelSummary):
    """Detailed information about a model.

    Attributes:
        latestRun: The currently active run or latest run performed for the model. This is ``None`` if no runs have been
            performed yet.
    """

    latestRun: Optional[RunSummary]


_MODEL_SUMMARY_FRAGMENT = f"""
    fragment ModelSummary on EvaluationModel {{
        {' '.join(ModelSummary.__required_keys__)}
    }}
"""

_MODEL_DETAILS_FRAGMENT = """
    fragment ModelDetails on EvaluationModel {
        ...ModelSummary
        latestRun {
            ...RunSummary
        }
    }
""" + _MODEL_SUMMARY_FRAGMENT + RUN_SUMMARY_FRAGMENT

_LIST_MODELS = gql("""
    query listModels($filter: EvaluationModelsFilter) {
        evaluation_models(filter: $filter) {
            ...ModelSummary
        }
    }
""" + _MODEL_SUMMARY_FRAGMENT)

_READ_MODEL = gql("""
    query readModel($id: Int!) {
        evaluation_model(id: $id) {
            ...ModelDetails
        }
    }
""" + _MODEL_DETAILS_FRAGMENT)

_GET_MODEL_DATA_SOURCES = gql("""
    query getModelDataSources($modelId: Int!) {
        evaluation_model(id: $modelId) {
            referenceDataSource{
              ...DataSourceSummary
            }
            evaluationDataSource{
              ...DataSourceSummary
            }
        }
    }
""" + DATA_SOURCE_SUMMARY_FRAGMENT)

_GET_MODEL_REFERENCE_DATA_HISTORY = gql("""
    query getModelDataHistory($modelId: Int!) {
        evaluation_model(id: $modelId) {
            referenceDataSource{
                events {
                    ...DataSourceEvent
                }
                ...DataSourceSummary
            }
        }
    }
""" + DATA_SOURCE_SUMMARY_FRAGMENT + DATA_SOURCE_EVENT_FRAGMENT)

_GET_MODEL_EVALUATION_DATA_HISTORY = gql("""
    query getModelDataHistory($modelId: Int!) {
        evaluation_model(id: $modelId) {
            evaluationDataSource{
                events {
                    ...DataSourceEvent
                }
                ...DataSourceSummary
            }
        }
    }
""" + DATA_SOURCE_SUMMARY_FRAGMENT + DATA_SOURCE_EVENT_FRAGMENT)

_CREATE_MODEL = gql("""
    mutation createModel($input: CreateEvaluationModelInput!) {
        create_evaluation_model(input: $input) {
            ...ModelDetails
        }
    }
""" + _MODEL_DETAILS_FRAGMENT)

_DELETE_MODEL = gql("""
    mutation deleteModel($id: Int!) {
        delete_evaluation_model(evaluationModelId: $id) {
            id
        }
    }
""")


class MetricConfiguration(TypedDict):
    """Configuration for a model evaluation metric.

    Attributes:
        enabled: Whether the metric is enabled or disabled.
        rope_lower_bound: Lower bound of the region of practical equivalence (ROPE) for the metric. This is required
            when the hypothesis is `MODEL_PERFORMANCE_WITHIN_RANGE`.
        rope_upper_bound: Upper bound of the region of practical equivalence (ROPE) for the metric. This is required
            when the hypothesis is `MODEL_PERFORMANCE_WITHIN_RANGE`.
        hdi_width: Required width of the highest density interval (HDI) for the metric before evaluating the hypothesis.
    """
    enabled: bool
    rope_lower_bound: Optional[float]
    rope_upper_bound: Optional[float]
    hdi_width: Optional[float]


class Model:

    @classmethod
    def list(
            cls, name: Optional[str] = None, problem_type: Optional[ProblemType] = None
    ) -> List[ModelSummary]:
        """List defined models.

        Args:
            name: Optional name filter.
            problem_type: Optional problem type filter.

        Returns:
            List of models that match the provided filter criteria.
        """
        return execute(_LIST_MODELS, {
            'filter': {
                'name': name,
                'problemType': problem_type,
            }
        })['evaluation_models']

    @classmethod
    def get(cls, model_id: str) -> ModelDetails:
        """Get details for a model.

        Args:
            model_id: ID of the model to get details for.

        Returns:
            Detailed information about the model.
        """
        return execute(_READ_MODEL, {'id': int(model_id)})['evaluation_model']

    @classmethod
    def create(
            cls,
            name: str,
            schema: ModelSchema,
            reference_data: pd.DataFrame,
            hypothesis: HypothesisType,
            classification_threshold: float,
            metrics_configuration: Dict[PerformanceMetric, MetricConfiguration],
            key_performance_metric: PerformanceMetric,
            evaluation_data: Optional[pd.DataFrame] = None,
    ) -> ModelDetails:
        """Create a new model.

        Args:
            name: Name for the model.
            schema: Schema of the model. Typically, created using
                [Schema.from_df][nannyml_cloud_sdk.model_evaluation.Schema.from_df].
            hypothesis: The type of hypothesis the model is trying to validate. This can be one of the following:
                - `MODEL_PERFORMANCE_NO_WORSE_THAN_REFERENCE`: The model's performance is not worse than the reference.
                - `MODEL_PERFORMANCE_WITHIN_RANGE`: The model's performance is within a specified range.
            classification_threshold: The threshold used to turn predicted probabilities into binary predictions.
            reference_data: Reference data to use for the model.
            evaluation_data: Analysis data to use for the model. If the data contains targets, targets must always be
                provided together with analysis data.
            metrics_configuration: Configuration for each metric to be used in the model.
            key_performance_metric:  Key performance metric for the model.

        Returns:
            Detailed about the model once it has been created.
        """

        reference_data_source = {
            'name': 'reference',
            'hasReferenceData': True,
            'hasAnalysisData': False,
            'columns': schema['columns'],
            'storageInfo': Data.upload(reference_data),
        }

        evaluation_data_source = {
            'name': 'evaluation',
            'hasReferenceData': False,
            'hasAnalysisData': True,
            'columns': [
                column for column in schema['columns']
                if column['name'] in map(normalize, evaluation_data.columns)
            ],
            'storageInfo': Data.upload(evaluation_data),
        } if evaluation_data is not None else None

        return execute(_CREATE_MODEL, {
            'input': {
                'name': name,
                'problemType': schema['problemType'],
                'referenceDataSource': reference_data_source,
                'evaluationDataSource': evaluation_data_source,
                'kpm': key_performance_metric,
                'hypothesis': hypothesis,
                'classificationThreshold': classification_threshold,
                'metrics': [{
                    'metric': metric,
                    'enabled': config['enabled'],
                    'ropeLowerBound': config['rope_lower_bound'],
                    'ropeUpperBound': config['rope_upper_bound'],
                    'hdiWidth': config['hdi_width'],
                } for metric, config in metrics_configuration.items()]
            },
        })['create_evaluation_model']

    @classmethod
    def delete(cls, model_id: str) -> None:
        """Delete a model.

        Args:
            model_id: ID of the model to delete.
        """
        execute(_DELETE_MODEL, {'id': int(model_id)})

    @classmethod
    def add_evaluation_data(cls, model_id: str, data: pd.DataFrame) -> None:
        """Add evaluation data to a model.

        Args:
            model_id: ID of the model.
            data: Data to be added.

        Note:
            This method does not update existing data. It only adds new data. If you want to update existing data,
            use [upsert_evaluation_data][nannyml_cloud_sdk.model_evaluation.Model.upsert_evaluation_data] instead.
        """
        evaluation_data_source = cls._get_evaluation_data_source(model_id)
        execute(_ADD_DATA_TO_DATA_SOURCE, {
            'input': {
                'id': int(evaluation_data_source['id']),
                'storageInfo': Data.upload(data),
            },
        })

    @classmethod
    def upsert_evaluation_data(cls, model_id: str, data: pd.DataFrame) -> None:
        """Add or update analysis data for a model.

        Args:
            model_id: ID of the model.
            data: Data to be added/updated.

        Note:
            This method compares existing data with the new data to determine which rows to update and which to add.
            If you are certain you are only adding new data, it is recommended to use
            [add_evaluation_data][nannyml_cloud_sdk.model_evaluation.Model.add_evaluation_data]
            instead for better performance.
        """
        evaluation_data_source = cls._get_evaluation_data_source(model_id)
        execute(_UPSERT_DATA_IN_DATA_SOURCE, {
            'input': {
                'id': int(evaluation_data_source['id']),
                'storageInfo': Data.upload(data),
            },
        })

    @classmethod
    def get_reference_data_history(cls, model_id: str) -> List[DataSourceEvent]:
        """Get reference data history for a model.

        Args:
            model_id: ID of the model.

        Returns:
            List of events related to reference data for the model.
        """
        return execute(_GET_MODEL_REFERENCE_DATA_HISTORY, {
            'modelId': int(model_id),
        })['evaluation_model']['referenceDataSource']['events']

    @classmethod
    def get_evaluation_data_history(cls, model_id: str) -> List[DataSourceEvent]:
        """Get evaluation data history for a model.

        Args:
            model_id: ID of the model.

        Returns:
            List of events related to analysis data for the model.
        """
        return execute(_GET_MODEL_EVALUATION_DATA_HISTORY, {
            'modelId': int(model_id),
        })['evaluation_model']['evaluationDataSource']['events']

    @staticmethod
    @functools.lru_cache(maxsize=128)
    def _get_model_data_sources(
            model_id: str, filter: Optional[DataSourceFilter] = None
    ) -> Dict[str, DataSourceSummary]:
        """Get data sources for a model"""
        return execute(_GET_MODEL_DATA_SOURCES, {
            'modelId': int(model_id),
            'filter': filter,
        })['evaluation_model']

    @classmethod
    def _get_evaluation_data_source(cls, model_id: str) -> DataSourceSummary:
        """Helper method to get target data source for a model"""
        data_sources = cls._get_model_data_sources(model_id, frozendict({'name': 'evaluation'}))
        try:
            return data_sources['evaluationDataSource']
        except IndexError:
            raise InvalidOperationError(
                f"Model '{model_id}' has no evaluation data source."
            )
