import datetime
import inspect
from typing import TypedDict, Optional, List, Literal, overload, Union, Callable

from gql import gql

from nannyml_cloud_sdk.client import execute
from nannyml_cloud_sdk.enums import ProblemType


class CustomMetricSummary(TypedDict):
    """Summary of a custom metric.

    Attributes:
        id: Unique identifier of the custom metric (generated by NannyML Cloud when a custom metric is created).
        name: User-defined name of the custom metric.
        problemType: Type of problem the custom metric can be applied to.
        createdAt: Timestamp when the model was created.
    """
    id: int
    name: str
    problemType: ProblemType
    createdAt: datetime.datetime


class CustomMetricDetails(TypedDict):
    """Details of a custom metric.

    Attributes:
        description: User-defined description of the custom metric

    """

    description: str


class CustomClassificationMetricDetails(CustomMetricDetails):
    calculateFn: str
    estimateFn: str


class CustomRegressionMetricDetails(CustomMetricDetails):
    lossFn: str
    aggregateFn: str


TCustomMetricDetails = Union[CustomClassificationMetricDetails, CustomRegressionMetricDetails]
TCustomMetricSource = Union[str, Callable]


_CUSTOM_METRIC_SUMMARY_FRAGMENT = f"""
    fragment MetricSummary on Metric{{
        {' '.join(CustomMetricSummary.__required_keys__)}
    }}
"""


_CUSTOM_METRIC_DETAILS_FRAGMENT = """
    fragment MetricDetails on Metric {
        ...MetricSummary
        description
        ... on ClassificationMetric {
            calculateFn
            estimateFn
        }
        ... on RegressionMetric {
            lossFn
            aggregateFn
        }
    }
""" + _CUSTOM_METRIC_SUMMARY_FRAGMENT


_LIST_CUSTOM_METRICS = gql("""
    query listMetrics($filter: MetricsFilter) {
        monitoring_metrics(filter: $filter) {
            ...MetricSummary
        }
    }
""" + _CUSTOM_METRIC_SUMMARY_FRAGMENT)


_READ_CUSTOM_METRIC = gql("""
    query readMetric($id: Int!) {
        monitoring_metric(metricId: $id) {
            ...MetricDetails
        }
    }
""" + _CUSTOM_METRIC_DETAILS_FRAGMENT)


_CREATE_CUSTOM_METRIC = gql("""
    mutation createMetric($input: CreateMetricInput!) {
        create_monitoring_metric(metric: $input) {
            ...MetricDetails
        }
    }
""" + _CUSTOM_METRIC_DETAILS_FRAGMENT)


_DELETE_MONITORING_METRIC = gql("""
    mutation deleteMetric($id: Int!) {
        delete_monitoring_metric(metricId: $id) {
            ...MetricDetails
        }
    }
""" + _CUSTOM_METRIC_DETAILS_FRAGMENT)


class CustomMetric:
    """Class for interacting with custom metrics in NannyML Cloud."""

    @classmethod
    def list(cls, name: Optional[str] = None, problem_type: Optional[ProblemType] = None) -> List[CustomMetricSummary]:
        """List defined custom metrics.

        Args:
            name: Optional name filter.
            problem_type: Optional problem type filter.

        Returns:
            List of custom metrics that match the provided filter criteria.
        """
        res = execute(_LIST_CUSTOM_METRICS, {
            'filter': {
                'names': None if name is None else [name],
                'problemTypes': None if problem_type is None else [problem_type],
            }
        })['monitoring_metrics']

        return res

    @classmethod
    def get(cls, metric_id: int) -> TCustomMetricDetails:
        """Get details of a custom metric.

        Args:
            metric_id: Unique identifier of the custom metric.

        Returns:
            Details of the custom metric.
        """
        return execute(_READ_CUSTOM_METRIC, {'id': metric_id})['monitoring_metric']

    @overload
    @classmethod
    def create(
        cls,
        name: str,
        description: str,
        problem_type: Literal["REGRESSION"],
        *,
        loss_function: TCustomMetricSource,
        aggregation_function: TCustomMetricSource,
        lower_value_limit: Optional[float] = None,
        upper_value_limit: Optional[float] = None,
    ) -> CustomRegressionMetricDetails:
        ...

    @overload
    @classmethod
    def create(
        cls,
        name: str,
        description: str,
        problem_type: Literal["BINARY_CLASSIFICATION", "MULTICLASS_CLASSIFICATION"],
        *,
        calculation_function: TCustomMetricSource,
        estimation_function: Optional[TCustomMetricSource] = None,
        lower_value_limit: Optional[float] = None,
        upper_value_limit: Optional[float] = None,
    ) -> CustomClassificationMetricDetails:
        ...

    @classmethod
    def create(
        cls,
        name: str,
        description: str,
        problem_type: ProblemType,
        calculation_function: Optional[TCustomMetricSource] = None,
        estimation_function: Optional[TCustomMetricSource] = None,
        loss_function: Optional[TCustomMetricSource] = None,
        aggregation_function: Optional[TCustomMetricSource] = None,
        lower_value_limit: Optional[float] = None,
        upper_value_limit: Optional[float] = None,
    ) -> TCustomMetricDetails:
        """Create a new custom metric."""

        classification_params, regression_params = None, None

        if problem_type == 'REGRESSION':
            if loss_function is None or aggregation_function is None:
                raise ValueError('`loss_function` and `aggregation_function` must be provided '
                                 'for custom regression metrics')

            regression_params = {
                'lossFn': _get_source_str(loss_function),
                'aggregateFn': _get_source_str(aggregation_function),
            }
        else:
            if calculation_function is None:
                raise ValueError('`calculate_function` must be provided for custom classification metrics')

            classification_params = {
                'calculateFn': _get_source_str(calculation_function),
                'estimateFn': _get_source_str(estimation_function) if estimation_function is not None else None,
            }

        return execute(_CREATE_CUSTOM_METRIC, {
            'input': {
                'name': name,
                'problemType': problem_type,
                'description': description,
                'lowerValueLimit': lower_value_limit,
                'upperValueLimit': upper_value_limit,
                'classification': classification_params,
                'regression': regression_params,
            },
        })['create_monitoring_metric']

    @classmethod
    def delete(cls, metric_id: str) -> TCustomMetricDetails:
        """Delete a custom metric."""
        return execute(_DELETE_MONITORING_METRIC, {'id': metric_id})['delete_monitoring_metric']


def _get_source_str(source: TCustomMetricSource) -> str:
    if callable(source):
        return f'import pandas as pd\nimport numpy as np\n\n{inspect.getsource(source)}'
    return source
