import datetime
import functools
from typing import List, Optional

import pandas as pd
from frozendict import frozendict
from gql import gql

from ..client import execute
from ..data import (
    DATA_SOURCE_EVENT_FRAGMENT, DATA_SOURCE_SUMMARY_FRAGMENT, Data, DataSourceEvent, DataSourceFilter,
    DataSourceSummary, _ADD_DATA_TO_DATA_SOURCE, _UPSERT_DATA_IN_DATA_SOURCE, _REMOVE_DATA_FROM_DATA_SOURCE
)
from ..enums import ChunkPeriod, PerformanceMetric, ProblemType
from ..errors import InvalidOperationError
from .run import RUN_SUMMARY_FRAGMENT, RunSummary
from .schema import ModelSchema, normalize
from .._typing import TypedDict
from .configuration import RuntimeConfiguration


class ModelSummary(TypedDict):
    """Summary of a model.

    Attributes:
        id: Unique identifier of the model (generated by NannyML Cloud when a model is created).
        name: User-defined name of the model.
        problemType: Type of problem the model is trying to solve.
        createdAt: Timestamp when the model was created.
    """
    id: str
    name: str
    problemType: ProblemType
    createdAt: datetime.datetime


class ModelDetails(ModelSummary):
    """Detailed information about a model.

    Attributes:
        latestRun: The currently active run or latest run performed for the model. This is ``None`` if no runs have been
            performed yet.
        nextRun: The next run scheduled for the model. This is ``None`` if there is a run currently active.
    """

    latestRun: Optional[RunSummary]
    nextRun: Optional[RunSummary]


_MODEL_SUMMARY_FRAGMENT = f"""
    fragment ModelSummary on Model {{
        {' '.join(ModelSummary.__required_keys__)}
    }}
"""

_MODEL_DETAILS_FRAGMENT = """
    fragment ModelDetails on Model {
        ...ModelSummary
        latestRun {
            ...RunSummary
        }
        nextRun {
            ...RunSummary
        }
    }
""" + _MODEL_SUMMARY_FRAGMENT + RUN_SUMMARY_FRAGMENT

_LIST_MODELS = gql("""
    query listModels($filter: ModelsFilter) {
        monitoring_models(filter: $filter) {
            ...ModelSummary
        }
    }
""" + _MODEL_SUMMARY_FRAGMENT)

_READ_MODEL = gql("""
    query readModel($id: Int!) {
        monitoring_model(id: $id) {
            ...ModelDetails
        }
    }
""" + _MODEL_DETAILS_FRAGMENT)

_GET_MODEL_DATA_SOURCES = gql("""
    query getModelDataSources($modelId: Int!, $filter: DataSourcesFilter) {
        monitoring_model(id: $modelId) {
            dataSources(filter: $filter) {
                ...DataSourceSummary
            }
        }
    }
""" + DATA_SOURCE_SUMMARY_FRAGMENT)

_GET_MODEL_DATA_HISTORY = gql("""
    query getModelDataHistory($modelId: Int!, $dataSourceFilter: DataSourcesFilter) {
        monitoring_model(id: $modelId) {
            dataSources(filter: $dataSourceFilter) {
                events {
                    ...DataSourceEvent
                }
                ...DataSourceSummary
            }
        }
    }
""" + DATA_SOURCE_SUMMARY_FRAGMENT + DATA_SOURCE_EVENT_FRAGMENT)

_CREATE_MODEL = gql("""
    mutation createModel($input: CreateModelInput!) {
        create_monitoring_model(model: $input) {
            ...ModelDetails
        }
    }
""" + _MODEL_DETAILS_FRAGMENT)

_DELETE_MODEL = gql("""
    mutation deleteModel($id: Int!) {
        delete_monitoring_model(modelId: $id) {
            id
        }
    }
""")

_ADD_CUSTOM_METRIC_TO_MODEL = gql("""
    mutation addCustomMetricToModel($modelId: Int!, $metricId: Int!) {
        add_custom_metric_to_monitoring_model(input: {
            modelId: $modelId
            metricId: $metricId
        }) {
            id
        }
    }
""")

_REMOVE_CUSTOM_METRIC_FROM_MODEL = gql("""
    mutation removeCustomMetricFromModel($modelId: Int!, $metricId: Int!) {
        remove_custom_metric_from_monitoring_model(input: {
            modelId: $modelId
            metricId: $metricId
        }) {
            id
        }
    }
""")


class Model:
    """Operations for working with machine learning models."""

    @classmethod
    def list(cls, name: Optional[str] = None, problem_type: Optional[ProblemType] = None) -> List[ModelSummary]:
        """List defined models.

        Args:
            name: Optional name filter.
            problem_type: Optional problem type filter.

        Returns:
            List of models that match the provided filter criteria.
        """
        return execute(_LIST_MODELS, {
            'filter': {
                'name': name,
                'problemType': problem_type,
            }
        })['monitoring_models']

    @classmethod
    def get(cls, model_id: str) -> ModelDetails:
        """Get details for a model.

        Args:
            model_id: ID of the model to get details for.

        Returns:
            Detailed information about the model.
        """
        return execute(_READ_MODEL, {'id': int(model_id)})['monitoring_model']

    @classmethod
    def create(
        cls,
        name: str,
        schema: ModelSchema,
        reference_data: pd.DataFrame,
        analysis_data: pd.DataFrame,
        key_performance_metric: PerformanceMetric,
        key_performance_metric_component: Optional[str] = None,
        target_data: Optional[pd.DataFrame] = None,
        chunk_period: Optional[ChunkPeriod] = None,
        chunk_size: Optional[int] = None,
    ) -> ModelDetails:
        """Create a new model.

        Args:
            name: Name for the model.
            schema: Schema of the model. Typically, created using
                [Schema.from_df][nannyml_cloud_sdk.monitoring.Schema.from_df].
            reference_data: Reference data to use for the model.
            analysis_data: Analysis data to use for the model. If the data contains targets, targets must always be
                provided together with analysis data.
            key_performance_metric: Key performance metric for the model.
            key_performance_metric_component: Optional key performance metric component for the model.
                This is only required if the key performance metric is a composite metric, for example
                the `CONFUSION_MATRIX` metric that has components like `true_positives`, `false_positives`, etc.
            target_data: Optional target data to use for the model.
            chunk_period: Time period per chunk. Should only be set for time-based chunking.
            chunk_size: Number of rows per chunk. Should only be set for size-based chunking.

        Returns:
            Detailed about the model once it has been created.
        """
        if chunk_period is None and chunk_size is None:
            raise ValueError("`chunk_size` or `chunk_period` must be provided when creating a model")

        data_sources = [
            {
                'name': 'reference',
                'hasReferenceData': True,
                'hasAnalysisData': False,
                'columns': schema['columns'],
                'storageInfo': Data.upload(reference_data),
            },
            {
                'name': 'analysis',
                'hasReferenceData': False,
                'hasAnalysisData': True,
                'columns': [
                    column for column in schema['columns']
                    if column['name'] in map(normalize, analysis_data.columns)
                ],
                'storageInfo': Data.upload(analysis_data),
            },
        ]

        target_column = next((col['name'] for col in schema['columns'] if col['columnType'] == 'TARGET'), None)
        if target_column is None:
            raise ValueError("Schema must contain a target column")

        # Add target data source if target data is provided
        has_targets = True
        if target_data is not None:
            data_sources.append({
                'name': 'target',
                'hasReferenceData': False,
                'hasAnalysisData': True,
                'columns': [
                    column for column in schema['columns']
                    if column['name'] in map(normalize, target_data.columns)
                ],
                'storageInfo': Data.upload(target_data),
            })
        # Add empty target data source if target data is not provided in analysis
        elif target_column not in map(normalize, analysis_data.columns):
            has_targets = False
            data_sources.append({
                'name': 'target',
                'hasReferenceData': False,
                'hasAnalysisData': True,
                'columns': [
                    column for column in schema['columns']
                    if column['columnType'] in ('TARGET', 'IDENTIFIER')
                ],
                'storageInfo': None,
            })

        runtime_config = RuntimeConfiguration.default(
            chunking=chunk_period if chunk_period is not None else 'NUMBER_OF_ROWS',
            schema=schema,
            has_analysis_targets=has_targets,
            nr_of_rows=chunk_size
        )

        return execute(_CREATE_MODEL, {
            'input': {
                'name': name,
                'problemType': schema['problemType'],
                'dataSources': data_sources,
                'kpm': {
                    'metric': key_performance_metric,
                    'component': key_performance_metric_component,
                },
                'runtimeConfig': runtime_config,
                'runOnCreate': False,
            },
        })['create_monitoring_model']

    @classmethod
    def delete(cls, model_id: str) -> None:
        """Delete a model.

        Args:
            model_id: ID of the model to delete.
        """
        execute(_DELETE_MODEL, {'id': int(model_id)})

    @classmethod
    def add_analysis_data(cls, model_id: str, data: pd.DataFrame) -> None:
        """Add analysis data to a model.

        Args:
            model_id: ID of the model.
            data: Data to be added.

        Note:
            This method does not update existing data. It only adds new data. If you want to update existing data,
            use [upsert_analysis_data][nannyml_cloud_sdk.monitoring.Model.upsert_analysis_data] instead.
        """
        analysis_data_source, = cls._get_model_data_sources(model_id, frozendict({'name': 'analysis'}))
        execute(_ADD_DATA_TO_DATA_SOURCE, {
            'input': {
                'id': int(analysis_data_source['id']),
                'storageInfo': Data.upload(data),
            },
        })

    @classmethod
    def add_analysis_target_data(cls, model_id: str, data: pd.DataFrame) -> None:
        """Add (delayed) target data to a model.

        Args:
            model_id: ID of the model.
            data: Data to be added.

        Note:
            This method can only be used if the model has a target data source.
            If you want to add analysis data to a model without a target data source, use
            [add_analysis_data][nannyml_cloud_sdk.monitoring.Model.add_analysis_data] instead.

        Note:
            This method does not update existing data. It only adds new data. If you want to update existing data,
            use [upsert_analysis_target_data][nannyml_cloud_sdk.monitoring.Model.upsert_analysis_target_data] instead.
        """
        target_data_source = cls._get_target_data_source(model_id)
        execute(_ADD_DATA_TO_DATA_SOURCE, {
            'input': {
                'id': int(target_data_source['id']),
                'storageInfo': Data.upload(data),
            },
        })

    @classmethod
    def upsert_analysis_data(cls, model_id: str, data: pd.DataFrame) -> None:
        """Add or update analysis data for a model.

        Args:
            model_id: ID of the model.
            data: Data to be added/updated.

        Note:
            This method compares existing data with the new data to determine which rows to update and which to add.
            If you are certain you are only adding new data, it is recommended to use
            [add_analysis_data][nannyml_cloud_sdk.monitoring.Model.add_analysis_data] instead for better performance.
        """
        analysis_data_source, = cls._get_model_data_sources(model_id, frozendict({'name': 'analysis'}))
        execute(_UPSERT_DATA_IN_DATA_SOURCE, {
            'input': {
                'id': int(analysis_data_source['id']),
                'storageInfo': Data.upload(data),
            },
        })

    @classmethod
    def upsert_analysis_target_data(cls, model_id: str, data: pd.DataFrame) -> None:
        """Add or update (delayed) target data for a model.

        Args:
            model_id: ID of the model.
            data: Data to be added/updated.

        Note:
            This method can only be used if the model has a target data source. If you want to update analysis data in a
            model without a target data source, use
            [upsert_analysis_data][nannyml_cloud_sdk.monitoring.Model.upsert_analysis_data] instead.

        Note:
            This method compares existing data with the new data to determine which rows to update and which to add.
            If you are certain you are only adding new data, it is recommended to use
            [add_analysis_target_data][nannyml_cloud_sdk.monitoring.Model.add_analysis_target_data]
            instead for better performance.
        """
        target_data_source = cls._get_target_data_source(model_id)
        execute(_UPSERT_DATA_IN_DATA_SOURCE, {
            'input': {
                'id': int(target_data_source['id']),
                'storageInfo': Data.upload(data),
            },
        })

    @classmethod
    def delete_analysis_data(cls, model_id: str, data_ids: pd.DataFrame) -> None:
        """Delete analysis data from a model.

        Args:
            model_id: ID of the model.
            data_ids: ID's for the data to be deleted.
        """
        analysis_data_source, = cls._get_model_data_sources(model_id, frozendict({'name': 'analysis'}))
        execute(_REMOVE_DATA_FROM_DATA_SOURCE, {
            'input': {
                'id': int(analysis_data_source['id']),
                'dataIds': Data.upload(data_ids),
            },
        })

    @classmethod
    def delete_analysis_target_data(cls, model_id: str, data_ids: pd.DataFrame) -> None:
        """Delete target data from a model.

        Args:
            model_id: ID of the model.
            data_ids: ID's for the data to be deleted.
        """
        target_data_source = cls._get_target_data_source(model_id)
        execute(_REMOVE_DATA_FROM_DATA_SOURCE, {
            'input': {
                'id': int(target_data_source['id']),
                'dataIds': Data.upload(data_ids),
            },
        })

    @classmethod
    def get_reference_data_history(cls, model_id: str) -> List[DataSourceEvent]:
        """Get reference data history for a model.

        Args:
            model_id: ID of the model.

        Returns:
            List of events related to reference data for the model.
        """
        return execute(_GET_MODEL_DATA_HISTORY, {
            'modelId': int(model_id),
            'dataSourceFilter': {'name': 'reference'},
        })['monitoring_model']['dataSources'][0]['events']

    @classmethod
    def get_analysis_data_history(cls, model_id: str) -> List[DataSourceEvent]:
        """Get analysis data history for a model.

        Args:
            model_id: ID of the model.

        Returns:
            List of events related to analysis data for the model.
        """
        return execute(_GET_MODEL_DATA_HISTORY, {
            'modelId': int(model_id),
            'dataSourceFilter': {'name': 'analysis'},
        })['monitoring_model']['dataSources'][0]['events']

    @classmethod
    def get_analysis_target_data_history(cls, model_id: str) -> List[DataSourceEvent]:
        """Get target data history for a model.

        Args:
            model_id: ID of the model.

        Returns:
            List of events related to target data for the model.
        """
        return execute(_GET_MODEL_DATA_HISTORY, {
            'modelId': int(model_id),
            'dataSourceFilter': {'name': 'target'},
        })['monitoring_model']['dataSources'][0]['events']

    @staticmethod
    @functools.lru_cache(maxsize=128)
    def _get_model_data_sources(model_id: str, filter: Optional[DataSourceFilter] = None) -> List[DataSourceSummary]:
        """Get data sources for a model"""
        return execute(_GET_MODEL_DATA_SOURCES, {
            'modelId': int(model_id),
            'filter': filter,
        })['monitoring_model']['dataSources']

    @classmethod
    def _get_target_data_source(cls, model_id: str) -> DataSourceSummary:
        """Helper method to get target data source for a model"""
        data_sources = cls._get_model_data_sources(model_id, frozendict({'name': 'target'}))
        try:
            return data_sources[0]
        except IndexError:
            raise InvalidOperationError(
                f"Model '{model_id}' has no target data source. If targets are present, they are stored in the "
                "analysis data source. Use `delete_analysis_data` instead."
            )

    @classmethod
    def add_custom_metric(cls, model_id: str, metric_id: str) -> None:
        """Add a custom metric to a monitoring model."""
        execute(_ADD_CUSTOM_METRIC_TO_MODEL, {
            'modelId': int(model_id),
            'metricId': int(metric_id),
        })

    @classmethod
    def remove_custom_metric(cls, model_id: str, metric_id: str) -> None:
        """Remove a custom metric from a monitoring model."""
        execute(_REMOVE_CUSTOM_METRIC_FROM_MODEL, {
            'modelId': int(model_id),
            'metricId': int(metric_id),
        })
