"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const token_1 = require("./token");
const CFN_ELEMENT_SYMBOL = Symbol.for('@aws-cdk/core.CfnElement');
/**
 * An element of a CloudFormation stack.
 */
class CfnElement extends construct_compat_1.Construct {
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, CFN_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(this.node.path)}.LogicalID`,
        });
        this.node.addMetadata(cxschema.ArtifactMetadataEntryType.LOGICAL_ID, this.logicalId, this.constructor);
    }
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(x) {
        return CFN_ELEMENT_SYMBOL in x;
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStack() {
        const trace = this.node.metadata.find(md => md.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class CfnRefElement extends CfnElement {
    /**
     * Return a string that will be resolved to a CloudFormation `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     */
    get ref() {
        return token_1.Token.asString(cfn_reference_1.CfnReference.for(this, 'Ref'));
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./private/cfn-reference");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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