"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const construct_compat_1 = require("./construct-compat");
const custom_resource_1 = require("./custom-resource");
const cfn_utils_provider_1 = require("./private/cfn-utils-provider");
const stack_1 = require("./stack");
const stack_trace_1 = require("./stack-trace");
/**
 * Captures a synthesis-time JSON object a CloudFormation reference which
 * resolves during deployment to the resolved values of the JSON object.
 *
 * The main use case for this is to overcome a limitation in CloudFormation that
 * does not allow using intrinsic functions as dictionary keys (because
 * dictionary keys in JSON must be strings). Specifically this is common in IAM
 * conditions such as `StringEquals: { lhs: "rhs" }` where you want "lhs" to be
 * a reference.
 *
 * This object is resolvable, so it can be used as a value.
 *
 * This construct is backed by a custom resource.
 */
class CfnJson extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.creationStack = [];
        this.creationStack = stack_trace_1.captureStackTrace();
        // stringify the JSON object in a token-aware way.
        this.jsonString = stack_1.Stack.of(this).toJsonString(props.value);
        const resource = new custom_resource_1.CustomResource(this, 'Resource', {
            serviceToken: cfn_utils_provider_1.CfnUtilsProvider.getOrCreate(this),
            resourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            properties: {
                Value: this.jsonString,
            },
        });
        this.value = resource.getAtt('Value');
    }
    /**
     * This is required in case someone JSON.stringifys an object which refrences
     * this object. Otherwise, we'll get a cyclic JSON reference.
     */
    toJSON() {
        return this.jsonString;
    }
    resolve(_) {
        return this.value;
    }
}
exports.CfnJson = CfnJson;
//# sourceMappingURL=data:application/json;base64,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