"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
// import required to be here, otherwise causes a cycle when running the generated JavaScript
// tslint:disable-next-line:ordered-imports
const cfn_element_1 = require("./cfn-element");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const deps_1 = require("./deps");
const cfn_reference_1 = require("./private/cfn-reference");
const removal_policy_1 = require("./removal-policy");
const tag_manager_1 = require("./tag-manager");
const util_1 = require("./util");
/**
 * Represents a CloudFormation resource.
 */
class CfnResource extends cfn_element_1.CfnRefElement {
    /**
     * Creates a resource construct.
     * @param cfnResourceType The CloudFormation type of this resource (e.g. AWS::DynamoDB::Table)
     */
    constructor(scope, id, props) {
        super(scope, id);
        // MAINTAINERS NOTE: this class serves as the base class for the generated L1
        // ("CFN") resources (such as `s3.CfnBucket`). These resources will have a
        // property for each CloudFormation property of the resource. This means that
        // if at some point in the future a property is introduced with a name similar
        // to one of the properties here, it will be "masked" by the derived class. To
        // that end, we prefix all properties in this class with `cfnXxx` with the
        // hope to avoid those conflicts in the future.
        /**
         * Options for this resource, such as condition, update policy etc.
         */
        this.cfnOptions = {};
        /**
         * An object to be merged on top of the entire resource definition.
         */
        this.rawOverrides = {};
        /**
         * Logical IDs of dependencies.
         *
         * Is filled during prepare().
         */
        this.dependsOn = new Set();
        if (!props.type) {
            throw new Error('The `type` property is required');
        }
        this.cfnResourceType = props.type;
        this._cfnProperties = props.properties || {};
        // if aws:cdk:enable-path-metadata is set, embed the current construct's
        // path in the CloudFormation template, so it will be possible to trace
        // back to the actual construct path.
        if (this.node.tryGetContext(cxapi.PATH_METADATA_ENABLE_CONTEXT)) {
            this.cfnOptions.metadata = {
                [cxapi.PATH_METADATA_KEY]: this.node.path,
            };
        }
    }
    /**
     * Check whether the given construct is a CfnResource
     */
    static isCfnResource(construct) {
        return construct.cfnResourceType !== undefined;
    }
    /**
     * Sets the deletion policy of the resource based on the removal policy specified.
     */
    applyRemovalPolicy(policy, options = {}) {
        policy = policy || options.default || removal_policy_1.RemovalPolicy.RETAIN;
        let deletionPolicy;
        switch (policy) {
            case removal_policy_1.RemovalPolicy.DESTROY:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
                break;
            case removal_policy_1.RemovalPolicy.RETAIN:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
                break;
            case removal_policy_1.RemovalPolicy.SNAPSHOT:
                deletionPolicy = cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
                break;
            default:
                throw new Error(`Invalid removal policy: ${policy}`);
        }
        this.cfnOptions.deletionPolicy = deletionPolicy;
        if (options.applyToUpdateReplacePolicy !== false) {
            this.cfnOptions.updateReplacePolicy = deletionPolicy;
        }
    }
    /**
     * Returns a token for an runtime attribute of this resource.
     * Ideally, use generated attribute accessors (e.g. `resource.arn`), but this can be used for future compatibility
     * in case there is no generated attribute.
     * @param attributeName The name of the attribute.
     */
    getAtt(attributeName) {
        return cfn_reference_1.CfnReference.for(this, attributeName);
    }
    /**
     * Adds an override to the synthesized CloudFormation resource. To add a
     * property override, either use `addPropertyOverride` or prefix `path` with
     * "Properties." (i.e. `Properties.TopicName`).
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * For example,
     * ```typescript
     * addOverride('Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes', ['myattribute'])
     * addOverride('Properties.GlobalSecondaryIndexes.1.ProjectionType', 'INCLUDE')
     * ```
     * would add the overrides
     * ```json
     * "Properties": {
     *   "GlobalSecondaryIndexes": [
     *     {
     *       "Projection": {
     *         "NonKeyAttributes": [ "myattribute" ]
     *         ...
     *       }
     *       ...
     *     },
     *     {
     *       "ProjectionType": "INCLUDE"
     *       ...
     *     },
     *   ]
     *   ...
     * }
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermdediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = path.split('.');
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null && typeof (curr[key]) === 'object' && !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    /**
     * Adds an override to a resource property.
     *
     * Syntactic sugar for `addOverride("Properties.<...>", value)`.
     *
     * @param propertyPath The path of the property
     * @param value The value
     */
    addPropertyOverride(propertyPath, value) {
        this.addOverride(`Properties.${propertyPath}`, value);
    }
    /**
     * Adds an override that deletes the value of a property from the resource definition.
     * @param propertyPath The path to the property.
     */
    addPropertyDeletionOverride(propertyPath) {
        this.addPropertyOverride(propertyPath, undefined);
    }
    /**
     * Indicates that this resource depends on another resource and cannot be
     * provisioned unless the other resource has been successfully provisioned.
     *
     * This can be used for resources across stacks (or nested stack) boundaries
     * and the dependency will automatically be transferred to the relevant scope.
     */
    addDependsOn(target) {
        deps_1.addDependency(this, target, `"${this.node.path}" depends on "${target.node.path}"`);
    }
    /**
     * @returns a string representation of this resource
     */
    toString() {
        return `${super.toString()} [${this.cfnResourceType}]`;
    }
    /**
     * Called by the `addDependency` helper function in order to realize a direct
     * dependency between two resources that are directly defined in the same
     * stacks.
     *
     * Use `resource.addDependsOn` to define the dependency between two resources,
     * which also takes stack boundaries into account.
     *
     * @internal
     */
    _addResourceDependency(target) {
        this.dependsOn.add(target);
    }
    /**
     * Emits CloudFormation for this resource.
     * @internal
     */
    _toCloudFormation() {
        try {
            const ret = {
                Resources: {
                    // Post-Resolve operation since otherwise deepMerge is going to mix values into
                    // the Token objects returned by ignoreEmpty.
                    [this.logicalId]: new util_1.PostResolveToken({
                        Type: this.cfnResourceType,
                        Properties: util_1.ignoreEmpty(this.cfnProperties),
                        DependsOn: util_1.ignoreEmpty(renderDependsOn(this.dependsOn)),
                        CreationPolicy: util_1.capitalizePropertyNames(this, renderCreationPolicy(this.cfnOptions.creationPolicy)),
                        UpdatePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updatePolicy),
                        UpdateReplacePolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.updateReplacePolicy),
                        DeletionPolicy: util_1.capitalizePropertyNames(this, this.cfnOptions.deletionPolicy),
                        Metadata: util_1.ignoreEmpty(this.cfnOptions.metadata),
                        Condition: this.cfnOptions.condition && this.cfnOptions.condition.logicalId,
                    }, props => {
                        const renderedProps = this.renderProperties(props.Properties || {});
                        props.Properties = renderedProps && (Object.values(renderedProps).find(v => !!v) ? renderedProps : undefined);
                        return deepMerge(props, this.rawOverrides);
                    }),
                },
            };
            return ret;
        }
        catch (e) {
            // Change message
            e.message = `While synthesizing ${this.node.path}: ${e.message}`;
            // Adjust stack trace (make it look like node built it, too...)
            const trace = this.creationStack;
            if (trace) {
                const creationStack = ['--- resource created at ---', ...trace].join('\n  at ');
                const problemTrace = e.stack.substr(e.stack.indexOf(e.message) + e.message.length);
                e.stack = `${e.message}\n  ${creationStack}\n  --- problem discovered at ---${problemTrace}`;
            }
            // Re-throw
            throw e;
        }
        // returns the set of logical ID (tokens) this resource depends on
        // sorted by construct paths to ensure test determinism
        function renderDependsOn(dependsOn) {
            return Array
                .from(dependsOn)
                .sort((x, y) => x.node.path.localeCompare(y.node.path))
                .map(r => r.logicalId);
        }
        function renderCreationPolicy(policy) {
            if (!policy) {
                return undefined;
            }
            const result = { ...policy };
            if (policy.resourceSignal && policy.resourceSignal.timeout) {
                result.resourceSignal = policy.resourceSignal;
            }
            return result;
        }
    }
    get cfnProperties() {
        const props = this._cfnProperties || {};
        if (tag_manager_1.TagManager.isTaggable(this)) {
            const tagsProp = {};
            tagsProp[this.tags.tagPropertyName] = this.tags.renderTags();
            return deepMerge(props, tagsProp);
        }
        return props;
    }
    renderProperties(props) {
        return props;
    }
    /**
     * Return properties modified after initiation
     *
     * Resources that expose mutable properties should override this function to
     * collect and return the properties object for this resource.
     */
    get updatedProperites() {
        return this._cfnProperties;
    }
    validateProperties(_properties) {
        // Nothing
    }
}
exports.CfnResource = CfnResource;
var TagType;
(function (TagType) {
    TagType["STANDARD"] = "StandardTag";
    TagType["AUTOSCALING_GROUP"] = "AutoScalingGroupTag";
    TagType["MAP"] = "StringToStringMap";
    TagType["KEY_VALUE"] = "KeyValue";
    TagType["NOT_TAGGABLE"] = "NotTaggable";
})(TagType = exports.TagType || (exports.TagType = {}));
/**
 * Merges `source` into `target`, overriding any existing values.
 * `null`s will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    for (const source of sources) {
        if (typeof (source) !== 'object' || typeof (target) !== 'object') {
            throw new Error(`Invalid usage. Both source (${JSON.stringify(source)}) and target (${JSON.stringify(target)}) must be objects`);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof (value) === 'object' && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = {};
                }
                deepMerge(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0) {
                    delete target[key];
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
//# sourceMappingURL=data:application/json;base64,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