"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 * @experimental
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider.
 *
 * @experimental
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const asset = stack.addFileAsset({
            fileName: staging.stagedPath,
            sourceHash: staging.sourceHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        const timeout = (_a = props.timeout) !== null && _a !== void 0 ? _a : duration_1.Duration.minutes(15);
        const memory = (_b = props.memorySize) !== null && _b !== void 0 ? _b : size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: 'nodejs12.x',
            },
        });
        handler.addDependsOn(role);
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope, uniqueid, props) {
        var _a;
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = (_a = stack.node.tryFindChild(id)) !== null && _a !== void 0 ? _a : new CustomResourceProvider(stack, id, props);
        return provider.serviceToken;
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,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