"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
const options_1 = require("./options");
/**
 * Determines whether a given file should be excluded or not based on given
 * exclusion glob patterns.
 *
 * @param exclude  exclusion patterns
 * @param filePath file path to be assessed against the pattern
 *
 * @returns `true` if the file should be excluded
 */
function shouldExclude(exclude, filePath) {
    let excludeOutput = false;
    for (const pattern of exclude) {
        const negate = pattern.startsWith('!');
        const match = minimatch(filePath, pattern, { matchBase: true, flipNegate: true });
        if (!negate && match) {
            excludeOutput = true;
        }
        if (negate && match) {
            excludeOutput = false;
        }
    }
    return excludeOutput;
}
exports.shouldExclude = shouldExclude;
/**
 * Determines whether a symlink should be followed or not, based on a FollowMode.
 *
 * @param mode       the FollowMode.
 * @param sourceRoot the root of the source tree.
 * @param realPath   the real path of the target of the symlink.
 *
 * @returns true if the link should be followed.
 */
function shouldFollow(mode, sourceRoot, realPath) {
    switch (mode) {
        case options_1.SymlinkFollowMode.ALWAYS:
            return fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.EXTERNAL:
            return !_isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.BLOCK_EXTERNAL:
            return _isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.NEVER:
            return false;
        default:
            throw new Error(`Unsupported FollowMode: ${mode}`);
    }
    function _isInternal() {
        return path.resolve(realPath).startsWith(path.resolve(sourceRoot));
    }
}
exports.shouldFollow = shouldFollow;
//# sourceMappingURL=data:application/json;base64,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