"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Class that keeps track of the logical IDs that are assigned to resources
 *
 * Supports renaming the generated IDs.
 */
class LogicalIDs {
    constructor() {
        /**
         * The rename table (old to new)
         */
        this.renames = {};
        /**
         * All assigned names (new to old, may be identical)
         *
         * This is used to ensure that:
         *
         * - No 2 resources end up with the same final logical ID, unless they were the same to begin with.
         * - All renames have been used at the end of renaming.
         */
        this.reverse = {};
    }
    /**
     * Rename a logical ID from an old ID to a new ID
     */
    addRename(oldId, newId) {
        if (oldId in this.renames) {
            throw new Error(`A rename has already been registered for '${oldId}'`);
        }
        this.renames[oldId] = newId;
    }
    /**
     * Return the renamed version of an ID or the original ID.
     */
    applyRename(oldId) {
        let newId = oldId;
        if (oldId in this.renames) {
            newId = this.renames[oldId];
        }
        // If this newId has already been used, it must have been with the same oldId
        if (newId in this.reverse && this.reverse[newId] !== oldId) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Two objects have been assigned the same Logical ID: '${this.reverse[newId]}' and '${oldId}' are now both named '${newId}'.`);
        }
        this.reverse[newId] = oldId;
        validateLogicalId(newId);
        return newId;
    }
    /**
     * Throw an error if not all renames have been used
     *
     * This is to assure that users didn't make typoes when registering renames.
     */
    assertAllRenamesApplied() {
        const keys = new Set();
        Object.keys(this.renames).forEach(keys.add.bind(keys));
        Object.keys(this.reverse).map(newId => {
            keys.delete(this.reverse[newId]);
        });
        if (keys.size !== 0) {
            const unusedRenames = Array.from(keys.values());
            throw new Error(`The following Logical IDs were attempted to be renamed, but not found: ${unusedRenames.join(', ')}`);
        }
    }
}
exports.LogicalIDs = LogicalIDs;
const VALID_LOGICALID_REGEX = /^[A-Za-z][A-Za-z0-9]{1,254}$/;
/**
 * Validate logical ID is valid for CloudFormation
 */
function validateLogicalId(logicalId) {
    if (!VALID_LOGICALID_REGEX.test(logicalId)) {
        throw new Error(`Logical ID must adhere to the regular expression: ${VALID_LOGICALID_REGEX.toString()}, got '${logicalId}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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