"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cfn_fn_1 = require("../cfn-fn");
const construct_compat_1 = require("../construct-compat");
const asset_parameters_1 = require("../private/asset-parameters");
const _shared_1 = require("./_shared");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the original deployment environment
 *
 * This deployment environment is restricted in cross-environment deployments,
 * CI/CD deployments, and will use up CloudFormation parameters in your template.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 */
class LegacyStackSynthesizer {
    constructor() {
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    bind(stack) {
        this.stack = stack;
    }
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addFileAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        // See `addFileAsset` for explanation.
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            return this.stack.addDockerImageAsset(asset);
        }
        finally {
            this.cycle = false;
        }
    }
    synthesizeStackArtifacts(session) {
        _shared_1.assertBound(this.stack);
        // Just do the default stuff, nothing special
        _shared_1.addStackArtifactToAssembly(session, this.stack, {}, []);
    }
    doAddDockerImageAsset(asset) {
        var _a, _b;
        _shared_1.assertBound(this.stack);
        // check if we have an override from context
        const repositoryNameOverride = this.stack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = (_b = (_a = asset.repositoryName) !== null && _a !== void 0 ? _a : repositoryNameOverride) !== null && _b !== void 0 ? _b : ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.stack.region}.${this.stack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        _shared_1.assertBound(this.stack);
        if (!this._assetParameters) {
            this._assetParameters = new construct_compat_1.Construct(this.stack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
//# sourceMappingURL=data:application/json;base64,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