"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const lib_1 = require("../../lib");
const evaluate_cfn_1 = require("../evaluate-cfn");
const CFN_CONTEXT = {
    'AWS::Region': 'the_region',
    'AWS::AccountId': 'the_account',
    'AWS::URLSuffix': 'domain.aws',
};
let app;
let stack;
/**
 * Evaluate a possibly string-containing value the same way CFN would do
 *
 * (Be invariant to the specific Fn::Sub or Fn::Join we would output)
 */
function evalCFN(value) {
    return evaluate_cfn_1.evaluateCFN(stack.resolve(value), CFN_CONTEXT);
}
function isAssetManifest(x) {
    return x instanceof cxapi.AssetManifestArtifact;
}
module.exports = {
    'setUp'(cb) {
        app = new lib_1.App({
            context: {
                [cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: 'true',
            },
        });
        stack = new lib_1.Stack(app, 'Stack');
        cb();
    },
    'stack template is in asset manifest'(test) {
        var _a;
        // GIVEN
        new lib_1.CfnResource(stack, 'Resource', {
            type: 'Some::Resource',
        });
        // WHEN
        const asm = app.synth();
        // THEN -- the S3 url is advertised on the stack artifact
        const stackArtifact = asm.getStackArtifact('Stack');
        test.equals(stackArtifact.stackTemplateAssetObjectUrl, 's3://cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}/4bdae6e3b1b15f08c889d6c9133f24731ee14827a9a9ab9b6b6a9b42b6d34910');
        // THEN - the template is in the asset manifest
        const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
        test.ok(manifestArtifact);
        const manifest = JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
        const firstFile = (_a = (manifest.files ? manifest.files[Object.keys(manifest.files)[0]] : undefined)) !== null && _a !== void 0 ? _a : {};
        test.deepEqual(firstFile, {
            source: { path: 'Stack.template.json', packaging: 'file' },
            destinations: {
                'current_account-current_region': {
                    bucketName: 'cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}',
                    objectKey: '4bdae6e3b1b15f08c889d6c9133f24731ee14827a9a9ab9b6b6a9b42b6d34910',
                    assumeRoleArn: 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-publishing-role-${AWS::AccountId}-${AWS::Region}',
                },
            },
        });
        test.done();
    },
    'add file asset'(test) {
        // WHEN
        const location = stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.bucketName), 'cdk-hnb659fds-assets-the_account-the_region');
        test.equals(evalCFN(location.s3Url), 'https://s3.the_region.domain.aws/cdk-hnb659fds-assets-the_account-the_region/abcdef');
        // THEN - object key contains source hash somewhere
        test.ok(location.objectKey.indexOf('abcdef') > -1);
        test.done();
    },
    'add docker image asset'(test) {
        // WHEN
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.repositoryName), 'cdk-hnb659fds-container-assets-the_account-the_region');
        test.equals(evalCFN(location.imageUri), 'the_account.dkr.ecr.the_region.domain.aws/cdk-hnb659fds-container-assets-the_account-the_region:abcdef');
        test.done();
    },
    'synthesis'(test) {
        // GIVEN
        stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // WHEN
        const asm = app.synth();
        // THEN - we have an asset manifest with both assets and the stack template in there
        const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
        test.ok(manifestArtifact);
        const manifest = JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
        test.equals(Object.keys(manifest.files || {}).length, 2);
        test.equals(Object.keys(manifest.dockerImages || {}).length, 1);
        // THEN - every artifact has an assumeRoleArn
        for (const file of Object.values({ ...manifest.files, ...manifest.dockerImages })) {
            for (const destination of Object.values(file.destinations)) {
                test.ok(destination.assumeRoleArn);
            }
        }
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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